/****************************************************************************
*****************************************************************************

Parallel Architecture Simulator
Eric A. Brewer  and  Chris N. Dellarocas
Laboratory for Computer Science
Massachusetts Institute of Technology

Module:  Augment -- Main

Description:
    Interprets command line arguments, initializes symbol table, opens
    input file, calls 'BuildGraph' to build basic block graph, and
    'AugmentSource' to create output file.

Last Modified:  4-26-90  (eab)

Global Functions:
    main(int argc, char *argv[])

Global Variables:
    Symbols for specific assembler directives:
        char *end_dir;   '.end'    (end subroutine)
	char *text_dir;  '.text'   (code segment)
	char *data_dir;  '.data'   (data segment)
	char *sdata_dir; '.sdata'  (small data segment)
	char *ent_dir;   '.ent'    (entry)
	char *frame_dir; '.frame'  (frame allocation)
	char *ext_dir;   '.extern' (global symbol decl)
	char *comm_dir   '.comm'   (common block decl)
	char *set_dir;   '.set'    (assembler flag)
	char *mask_dir;  '.mask'    (assembler flag)
	char *at_dir;    'at'
	char *noat_dir;  'noat'
	char *reorder_dir, *noreorder;
	char *macro_dir, *nomacro_dir;

    Version: float version_number;
             char *date_string;

    Flags:
        BOOL debug;      true => output debug messages
	BOOL verbose;    true => output status messages during augmentation

****************************************************************************
*   Copyright 1991                                                       
*   Eric A. Brewer  and  Chris N. Dellarocas                             
*   Massachusetts Institute of Technology                                
*                                                                        
*   Permission to use, copy, modify, and distribute this program         
*   for any purpose and without fee is hereby granted, provided          
*   that this copyright and permission notice appear on all copies       
*   and supporting documentation, the name of M.I.T. not be used         
*   in advertising or publicity pertaining to distribution of the        
*   program without specific prior permission, and notice be given       
*   in supporting documentation that copying and distribution is         
*   by permission of M.I.T.  M.I.T. makes no representations about       
*   the suitability of this software for any purpose.  It is pro-        
*   vided "as is" without express or implied warranty.		          
*****************************************************************************
****************************************************************************/


#include <stdio.h>
#include <string.h>
#include "lex.h"

GLOBAL float version_number = 2.4;
GLOBAL const char *date_string = "February 9, 1993";

static const char *copyright =
  "Copyright 1991, 1992,  Eric A. Brewer  and  Chris N. Dellarocas\nMassachusetts Institute of Technology\n";

extern void exit(int status);

/* symbol table declarations */
extern void InitTable();
extern const char *InsertSymbol(const char *name, int symbol_type);
extern void SymbolStats();

/* specific assembler directives */
const char *end_dir, *text_dir, *data_dir, *sdata_dir, *ent_dir, *frame_dir;
const char *ext_dir, *comm_dir, *set_dir, *mask_dir, *word_dir;
const char *at_dir, *noat_dir, *reorder_dir, *noreorder_dir;
const char *macro_dir, *nomacro_dir;

/* PMAX declartions */
extern void LoadInstructions();
extern void LoadMissingLibc();

/* Graph declarations */
extern void BuildGraph(FILE *fil);
extern void AugmentSource(FILE *fil, FILE *outfile);

BOOL debug = FALSE, verbose = FALSE, profiling = FALSE;
static FILE *outfile = stdout;

/** INTERNAL ROUTINES **/

static char *filename;



/****************************************************************************\
* interpret command line arguments
\****************************************************************************/

static void UnknownOption(char *s)
{
    if (s != NULL) fprintf(stderr, "Unknown option: `%s'\n\n", s);
    
    fprintf(stderr, "augment -- Version %.2f,  %s\n",
	    version_number, date_string);
    fprintf(stderr, "%s\n", copyright);

    fprintf(stderr, "Usage: augment [options] inputfile\n");
    fprintf(stderr,
	    "\t-o filename        Defines output file. [Default: standard output\n");
    fprintf(stderr,
	    "\t-p                 Generate profiling code.\n");
    fprintf(stderr,
	    "\t-d                 Print debugging messages.\n");
    fprintf(stderr,
	    "\t-v                 Print version number and date.\n");
    fprintf(stderr,
	    "\t-verb              Verbose mode.\n");

    fputc('\n', stderr);

    exit(1);
}


static void HandleArgs(int argc, char *argv[])
{
    int i;
    BOOL file_def;

    for (i=1, file_def=FALSE; i<argc; i++) {
	if (argv[i][0] == '-') {
	    switch (argv[i][1]) {
	      case 'd':
		if (strcmp(argv[i], "-d") != 0)
		  UnknownOption(argv[i]);
		debug = !debug;
		break;
	      case 'v':
		if (strcmp(argv[i], "-verb") == 0)
		  verbose = !verbose;
		else if (strcmp(argv[i], "-v") == 0) {
		    printf("augment -- Version %.2f,  %s\n",
			   version_number, date_string);
		    puts(copyright);
		}
		break;
	      case 'p':
		if (strcmp(argv[i], "-p") != 0)
		  UnknownOption(argv[i]);
		profiling = !profiling;
		break;
	      case 'o':
		if (strcmp(argv[i], "-o") != 0)
		  UnknownOption(argv[i]);
		if (++i == argc) {
		    fprintf(stderr,
			    "augment: Missing filename after '-o' option.\n");
		    exit(1);
		}
		outfile = fopen(argv[i], "w");
		if (outfile == NULL) {
		    fprintf(stderr,
			 "augment: Unable to open file \"%s\" for writing.\n",
			    argv[i]);
		    exit(1);
		}
		break;
	      default:
		UnknownOption(argv[i]);
	    }
	} else {
	    if (file_def)
	      fprintf(stderr,
	 "augment: Warning: input filename redefined from \"%s\" to \"%s\".\n",
		      filename, argv[i]);
	    filename = argv[i];
	    file_def = TRUE;
	}
    }

    if (!file_def) {
	fprintf(stderr, "augment: No input file defined.\n\n");
	UnknownOption(NULL);
    }
}


/*  Load (some) assembler directives into symbol table. */

static void LoadDirectives()
{
    end_dir = InsertSymbol(".end", SYM_DIR);
    text_dir = InsertSymbol(".text", SYM_DIR);
    data_dir = InsertSymbol(".data", SYM_DIR);
    sdata_dir = InsertSymbol(".sdata", SYM_DIR);
    ent_dir = InsertSymbol(".ent", SYM_DIR);
    frame_dir = InsertSymbol(".frame", SYM_DIR);
    ext_dir = InsertSymbol(".extern", SYM_DIR);
    comm_dir = InsertSymbol(".comm", SYM_DIR);
    set_dir = InsertSymbol(".set", SYM_DIR);
    mask_dir = InsertSymbol(".mask", SYM_DIR);
    word_dir = InsertSymbol(".word", SYM_DIR);
    at_dir = InsertSymbol("at", SYM_DIR);
    noat_dir = InsertSymbol("noat", SYM_DIR);
    reorder_dir = InsertSymbol("reorder", SYM_DIR);
    noreorder_dir = InsertSymbol("noreorder", SYM_DIR);
    macro_dir = InsertSymbol("macro", SYM_DIR);
    nomacro_dir = InsertSymbol("nomacro", SYM_DIR);
}

static void LoadLibraryRoutines()
{

#include "libc/counted.h"

#include "libm/counted.h"

}

void LoadMissingRoutines()
{
    /* missing floating point routines */
    InsertSymbol("c_abs_", SYM_NOCYC);
    InsertSymbol("facos", SYM_NOCYC);
    InsertSymbol("fasin", SYM_NOCYC);
    InsertSymbol("fatan", SYM_NOCYC);
    InsertSymbol("fatan2", SYM_NOCYC);
    InsertSymbol("fcabs", SYM_NOCYC);
    InsertSymbol("fceil", SYM_NOCYC);
    InsertSymbol("fcos", SYM_NOCYC);
    InsertSymbol("fcosh", SYM_NOCYC);
    InsertSymbol("fexp", SYM_NOCYC);
    InsertSymbol("fexpm1", SYM_NOCYC);
    InsertSymbol("ffloor", SYM_NOCYC);
    InsertSymbol("fhypot", SYM_NOCYC);
    InsertSymbol("flog", SYM_NOCYC);
    InsertSymbol("flog10", SYM_NOCYC);
    InsertSymbol("flog1p", SYM_NOCYC);
    InsertSymbol("fsin", SYM_NOCYC);
    InsertSymbol("fsinh", SYM_NOCYC);
    InsertSymbol("fsqrt", SYM_NOCYC);
    InsertSymbol("ftan", SYM_NOCYC);
    InsertSymbol("ftanh", SYM_NOCYC);
    InsertSymbol("ftrunc", SYM_NOCYC);
    InsertSymbol("gamma", SYM_NOCYC);
    InsertSymbol("swapINX", SYM_NOCYC);
    InsertSymbol("swapRM", SYM_NOCYC);
    InsertSymbol("trunc", SYM_NOCYC);
    InsertSymbol("z_abs_", SYM_NOCYC);
}


/** MAIN **/

/* unsigned long cycles_ = 1000000000;
long stackmin_ = 0;
void SimQuantum_() {}
void SimStack() {} */

void main(int argc, char *argv[])
{
    FILE *fil;
    /* unsigned long starttime = cycles_; */

    HandleArgs(argc, argv);

    InitTable();
    if (debug) printf("Symbol table initialized.\n");
    LoadDirectives();
    if (debug) printf("Assembler directives loaded.\n");
    LoadInstructions();
    if (debug) printf("PMAX instruction set loaded.\n");
    LoadLibraryRoutines();
    if (debug) printf("Library routines loaded.\n");
    LoadMissingRoutines();
    LoadMissingLibc();
    if (debug) printf("Missing library routines loaded.\n");

    fil = fopen(filename, "r");
    if (fil == NULL) {
	fprintf(stderr, "augment: Unable to read file \"%s\".\n", filename);
	exit(1);
    }

    BuildGraph(fil);
    rewind(fil);

    AugmentSource(fil, outfile);
    
    if (outfile != stdout) fclose(outfile);
    fclose(fil);

    if (verbose) SymbolStats();
}
