/****************************************************************************
*****************************************************************************

Parallel Architecture Simulator
Eric A. Brewer  and  Chris N. Dellarocas
Laboratory for Computer Science
Massachusetts Institute of Technology

Module:  Augment -- Graph Routines

Description:
    This module does the work of 'augment'.  The basic block graph is
    built by BuildGraph, which generates a list of roots, one for each
    subroutine in the file.  The graph is first built in a safe fashion,
    starting new blocks for every label.  When a subroutine is completed,
    branch references are filled in, and the graph is compacted.  Compaction
    merges two nodes when the latter was started because of an unreferenced
    label.  The basic block graphs for each subroutine are used by
    AugmentSource to add cycle counting code to the input file to generate
    the output file.

Last Modified:  4-26-90  (eab)

Global Functions:
    void BuildGraph(FILE *fil)
        Reads input file 'fil' and builds the basic block graph.

    void AugmentSource(FILE *fil, FILE *out)
        Rereads input file 'fil' and uses the basic block graph to
	build the augmented output file 'out'.

Global Variables:
    boolean noat_flag
        True when the INPUT has the "noat" flag set.

****************************************************************************
*   Copyright 1991                                                       
*   Eric A. Brewer  and  Chris N. Dellarocas                             
*   Massachusetts Institute of Technology                                
*                                                                        
*   Permission to use, copy, modify, and distribute this program         
*   for any purpose and without fee is hereby granted, provided          
*   that this copyright and permission notice appear on all copies       
*   and supporting documentation, the name of M.I.T. not be used         
*   in advertising or publicity pertaining to distribution of the        
*   program without specific prior permission, and notice be given       
*   in supporting documentation that copying and distribution is         
*   by permission of M.I.T.  M.I.T. makes no representations about       
*   the suitability of this software for any purpose.  It is pro-        
*   vided "as is" without express or implied warranty.		          
*****************************************************************************
****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include "lex.h"

extern void bzero(char *, int);
extern void bcopy(char *, char *, int);
extern void exit(int status), abort();
extern int atoi(char *);
extern long strtol(char *, char **, int);

/* lex declarations */
extern int yylex(), yyleng;
extern char yytext[];
extern char tmp[], *cur;

/* symbol table declarations */
char *LocateSymbol(char *name, int symbol_type);
char *LocateSymbolValue(char *name, int symbol_type, long *value);
 char *InsertSymbol(const char *name, int symbol_type);
void SetSymbolNode(const char *name, nodeType *node);
nodeType *GetSymbolNode(const char *name);
int ClearType(const char *sym, int styp);
extern const char *ent_dir, *data_dir, *sdata_dir;
extern const char *text_dir, *end_dir, *frame_dir;
extern const char *ext_dir, *comm_dir, *set_dir, *mask_dir;
extern const char *lw_inst_ptr; /* pointer to string "lw" */
extern const char *word_dir;
extern const char *at_dir, *noat_dir, *reorder_dir, *noreorder_dir;
extern const char *macro_dir, *nomacro_dir;

/* parsing modes */
#define TEXT_MODE    1
#define DATA_MODE    2
#define SUB_MODE     3

extern BOOL debug, verbose, profiling;
BOOL cycles_declared = 0, stackmin_declared = 0, noat_flag = 0;
extern float version_number;

static char *curProc = NULL;  /* name of the current procedure */

static short countStatus = 0;  /* counting on, positive for off,
				  negative is an error */
static BOOL noreorder = FALSE;  /* current status of reordering (parsing) */
static BOOL nomacro = FALSE;

static short gcc = 0;     /* source file compiled by gcc:
			      gcc=1 => version < 2,
			      gcc=2 => version 2+ */

/****************************************************************************
****************************************************************************/

/**  INTERNAL ROUTINES **/

/***************************************************************************/

static FILE *the_fil;
static int line_num = 0;

int NextLine()
{
    if (fgets(tmp, LINE_MAX, the_fil) == NULL) return 0;
    cur = tmp;
    line_num++;
    if (debug) printf("line %d\n", line_num);
    return 1;
}

static void catch()
{
    printf("In catch\n");
}

/***************************************************************************/
/* register accounting */

static BOOL regs[32];

static void InitRegisters()
{
    int i;

    for (i=0; i<16; i++) regs[i] = FALSE;
    for (i=16; i<24; i++) regs[i] = TRUE;
    for (i=24; i<26; i++) regs[i] = FALSE;
}

static void MarkReg(int reg)
{
    if (reg < 0 || reg > 31) {
	fprintf(stderr, "augment: INTERNAL3 MarkReg(%d)\n", reg);
	exit(1);
    }
    regs[reg] = TRUE;
}

/* Find a free register, returns -1 if there are none. */
static int FreeReg()
{
    int i;

    for (i=8; i<26; i++)
      if (regs[i] == FALSE) return i;
    return 1;
}


/***************************************************************************/
/* nodeType memory management */

static nodeType *free_nodes = NULL;

static nodeType *NewNode()
{
    nodeType *n;

    if (free_nodes != (nodeType *)NULL) {
	n = free_nodes;
	free_nodes = n->follow;
	bzero((char *)n, sizeof(nodeType));
    } else {
	n = (nodeType *) malloc(sizeof(nodeType));
    }
    bzero((char *)n, sizeof(nodeType));
    return n;
}

static void FreeNode(nodeType *node)
{
    node->follow = free_nodes;
    free_nodes = node;
}


/***************************************************************************/
/* track the root of each subroutine in a queue */

typedef struct rT {
    nodeType *node;
    char *proc;
    int useReg;
    short save31;
   /*  int prof_offset; */ /* profiling variable offset */
    struct rT *next;
} rootListType;

static rootListType *root_list = NULL;

static void EnqueueRoot(rootListType *root)
{
    rootListType **h;

    h = &root_list;
    while(*h != NULL) h = &((*h)->next);
    *h = (rootListType *) malloc(sizeof(rootListType));
    bcopy((char *)root, (char *)*h, sizeof(rootListType));
    (*h)->next = NULL;
}

static rootListType *DequeueRoot()
{
    rootListType *r;

    if (root_list == NULL) return NULL;
    r = root_list;
    root_list = root_list->next;
    return r;
}

static rootListType *root_gen = NULL;
static rootListType *InitRootGen()
{
    root_gen = root_list;
    if (root_gen != NULL) root_gen = root_gen->next;
    return(root_list);
}

static rootListType *NextRoot()
{
    rootListType *result = root_gen;
    if (root_gen != NULL) root_gen = root_gen->next;
    return(result);
}    


/***************************************************************************/
/* Track library calls that need to be changed to use the cycle counting
   library.

   For each reference to a library symbol, lex saves the symbol,
   the line number, and the character offset of the start of the symbol.
   When code is being output, each such symbol (located using
   the line number/character offset pair) is prepended with "cyc_".  This
   forces the reference to be linked with the version in the cycle-counting
   library (libcyc.a).  EnqueueLibCall() is called in yylex() (see asm.l)
   whenever an ID token is found that matches a library reference.  Library
   symbols are entered into the symbol table by LoadLibraryRoutines() in
   augment.c.  DequeueLibCall() is called by OutputLine().
*/

typedef struct libT {
    char *call;
    int line, ch;
    struct libT *next;
} libListType;

static libListType *lib_list = NULL;

void EnqueueLibCall(char *call)
{
    libListType **p;

    if (debug) printf("EnqueueLibCall(%s)\n", call);
    p = &lib_list;
    while(*p != NULL) p = &((*p)->next);
    *p = (libListType *) malloc(sizeof(libListType));
    (*p)->call = call;
    (*p)->line = line_num;
    (*p)->ch = (int)(cur - tmp) - yyleng;
    (*p)->next = NULL;
}

static libListType *DequeueLibCall()
{
    libListType *p;

    if (lib_list == NULL) return NULL;
    p = lib_list;
    lib_list = lib_list->next;
    if (debug) printf("DequeueLibCall() -> \"%s\", %d, %d.\n", p->call,
		      p->line, p->ch);
    return p;
}

/***************************************************************************/
/* find target label of branch instruction */
/* label is in argument number 'arg' */

static char *FindBranchTarget(int arg)
{
    int token, atoi();

    while (arg>1) {
	do {
	    token = yylex();
	    if (token==0) {
		fprintf(stderr,
			"augment: INTERNAL FindBranchTarget: Line %d, Insufficient arguments.\n",
			line_num);
		exit(2);
	    } else if (token == REG) {
		MarkReg(atoi(&yytext[1]));
	    }
	} while(token != (int)',');  /* skip argument */
	arg--;
    }
    if (yylex() != ID  && debug) printf("label `%s'\n", yytext);
    return(InsertSymbol(yytext, SYM_LABEL));
}


/***************************************************************************/
/* The alternate node for a branch (the non-fall-through node)
   is left blank until the entire subroutine has been parsed.
   The "alternate list" tracks these nodes.  'FillInAltNodes' fills them in.
*/

typedef struct aT {
    nodeType *node;
    const char *label;
    struct aT *next;
} listType;
static listType *alt_list = NULL;

static void AddAlternate(nodeType *node, const char *label)
{
    listType *a;
    
    a = (listType *) malloc(sizeof(listType));
    a->node = node;
    a->label = label;
    a->next = alt_list;
    alt_list = a;
}

static void FillInAltNodes()
{
    listType *a;
    nodeType *node;

    a = alt_list;
    while (a != NULL) {
	node = GetSymbolNode(a->label);
	assert(node != NULL);
	if (a->node != NULL)
	  (a->node)->alt = node;
	node->unref = FALSE;
	a = a->next;
    }
}


/***************************************************************************/
/* Print out the graph for a subroutine */

static void PrintGraph(nodeType *node, int block)
{
    if (node == NULL) return;
    printf("%d: %s(%d)", block,
	   (node->label == NULL)?"--":node->label, node->count);
    if (node->alt != NULL)
      printf(" alt:%s", node->alt->label);
    if (node->read_cycles > 0)
      printf(" read current time");
    putchar('\n');
    PrintGraph(node->follow, block + 1);
}

/***************************************************************************/
/* Compact graph.
   Given two nodes, the second of which is unrefenced, merge the two
   into one larger node.
*/

/* merge two nodes: 'first' and 'first->follow' into one */
static void MergeNodes(nodeType *first, nodeType *second)
{
    assert(first->alt == NULL);
    assert(second->unref == TRUE);

    first->end = second->end;
    first->alt = second->alt;
    first->read_cycles = second->read_cycles;
    first->count += second->count;
    first->follow = second->follow;
    FreeNode(second);
}

static void CompactGraph(nodeType *root)
{
    nodeType *second;

    if (root == NULL) return;
    second = root->follow;
    while (second != NULL) {
	if (root->alt == NULL && second->unref) {
	    MergeNodes(root, second);
	    second = root->follow;
	} else {
	    root = second;
	    second = root->follow;
	}
    }
}
    

/***************************************************************************/
/* parse a line at a time, and build the graph */

static nodeType *curNode = NULL;
static rootListType root;
static BOOL startProc = FALSE;

/* ParseData is used when in a data segment. */

static int ParseData()
{
    char *dir, *cmd;
    int token;

    if ((token = yylex()) == DIR) {
	dir = LocateSymbol(yytext, SYM_DIR);
	if (dir==text_dir) {
	    if (debug) printf("Parsing TEXT, line %d.\n", line_num);
	    while (yylex() != 0);
	    return(TEXT_MODE);
	} else if (dir==ext_dir || dir==comm_dir) {
	    yylex();
	    if (strcmp(yytext, "cycles_")==0) cycles_declared = TRUE;
	    else if (strcmp(yytext, "stackmin_")==0) stackmin_declared = TRUE;
	} else if (dir==set_dir) {
	    yylex();
	    cmd = LocateSymbol(yytext, SYM_DIR);
	    if (cmd==at_dir) noat_flag = FALSE;
	    else if (cmd==noat_dir) noat_flag = TRUE;
	    else if (cmd==reorder_dir) noreorder = FALSE;
	    else if (cmd==noreorder_dir) noreorder = TRUE;
	    else if (cmd==macro_dir) nomacro = FALSE;
	    else if (cmd==nomacro_dir) nomacro = TRUE;
	}
    } else if (token == LABEL) {
	if (strcmp(yytext, "gcc_compiled.:")==0) {
	    gcc = 1;
	    fprintf(stderr, "augment: Augmenting gcc source file.\n");
	} else if (strcmp(yytext, "gcc2_compiled.:")==0) {
	    gcc = 2;
	    fprintf(stderr, "augment: Augmenting gcc2 source file.\n");
	} else if (strcmp(yytext, "cycles_:")==0) {
	    cycles_declared = TRUE;
	} else if (strcmp(yytext, "stackmin_:")==0) {
	    stackmin_declared = TRUE; 
	}
    }
    while (yylex() != 0);
    return(DATA_MODE);
}

/* ParseText is used when in a code segment, but not in a subroutine */

static int ParseText()
{
    char *dir, *cmd;
    int token;

    if ((token = yylex()) == DIR) {
	dir = LocateSymbol(yytext, SYM_DIR);
	if (dir==data_dir || dir==sdata_dir) {
	    if (debug) printf("Parsing DATA, line %d.\n", line_num);
	    while (yylex() != 0);
	    return(DATA_MODE);
	}
	if (dir==ent_dir) {
	    token = yylex();
	    if (token != ID)
	      fprintf(stderr,
		      "augment: Parse error: token %d -- non-identifier after \".ent\".\n",
		      token);
	    curProc = InsertSymbol(yytext, SYM_IDENT);
	    if (debug) printf("Parsing subroutine \"%s\", line %d.\n",
			      curProc, line_num);
	    startProc = TRUE;
	    while (yylex() != 0);
	    return(SUB_MODE);
	} else if (dir==ext_dir || dir==comm_dir) {
	    yylex();
	    if (strcmp(yytext, "cycles_")==0) cycles_declared = 1;
	    else if (strcmp(yytext, "stackmin_")==0) stackmin_declared = 1;
	} else if (dir==set_dir) {
	    yylex();
	    cmd = LocateSymbol(yytext, SYM_DIR);
	    if (cmd==at_dir) noat_flag = FALSE;
	    else if (cmd==noat_dir) noat_flag = TRUE;
	    else if (cmd==reorder_dir) noreorder = FALSE;
	    else if (cmd==noreorder_dir) noreorder = TRUE;
	    else if (cmd==macro_dir) nomacro = FALSE;
	    else if (cmd==nomacro_dir) nomacro = TRUE;
	}
    } else if (token == LABEL) {
	if (strcmp(yytext, "gcc_compiled.:")==0) {
	    gcc = 1;
	    fprintf(stderr, "augment: Augmenting gcc source file\n");
	} else if (strcmp(yytext, "gcc2_compiled.:")==0) {
	    gcc = 2;
	    fprintf(stderr, "augment: Augmenting gcc2 source file\n");
	}
    }
    while (yylex() != 0);
    return(TEXT_MODE);
}


/* scan remainder of instruction for register usage */
static void ScanForRegisters()
{
    int token, atoi();

    while((token = yylex()) != 0) {
	if (token == REG) MarkReg(atoi(&yytext[1]));
    }
}

/* ParseSub is used when in a subroutine */

static int ParseSub(BOOL need_instruction)
{
    int token, typ;
    char *label, *id, *cmd;
    long weight;

    while((token=yylex()) != 0) {
	switch (token) {
	  case ID:
	    if (LocateSymbolValue(yytext, SYM_BRANCH, &weight) != NULL) {
		/** BRANCH INSTRUCTION **/
		assert(curNode != NULL);
		curNode->follow = NewNode();
		curNode->count += 1 + weight;  /* +1 for pipe delay. */
		if (LocateSymbol(yytext, SYM_INST3) != NULL)
		  label = FindBranchTarget(3);
		else if (LocateSymbol(yytext, SYM_INST2) != NULL)
		  label = FindBranchTarget(2);
		else
		  label = FindBranchTarget(1);
		AddAlternate(curNode, label);
		if (noreorder) { /* get delay slot instruction */
		    NextLine(); ParseSub(TRUE);
		}
		curNode->end = line_num;
		curNode = curNode->follow;
		curNode->start = line_num + 1;
		curNode->label = "#branch";
		curNode->noat = noat_flag;
		curNode->noreorder = noreorder;
		curNode->nomacro = nomacro;
		return(SUB_MODE);
	    } else if (LocateSymbolValue(yytext, SYM_CALL, &weight) != NULL) {
		/** CALL INSTRUCTION **/
		assert(curNode != NULL);
		token = yylex();
		if (token == ID) {
		    if (LocateSymbol(yytext, SYM_NOCYC) != NULL) {
			fprintf(stderr, "augment: Warning: library routine ");
			fprintf(stderr, "`%s' is not cycle counted.\n",
				yytext);
			ClearType(yytext, SYM_NOCYC);
		    }
		} else if (token == REG) MarkReg(atoi(&yytext[1]));
		ScanForRegisters();
		if (noreorder) { /* get delay slot instruction */
		    NextLine(); ParseSub(TRUE);
		}
		curNode->follow = NewNode();
		curNode->end = line_num;
		curNode->count += weight;
		curNode = curNode->follow;
		curNode->label = "#call";
		curNode->start = line_num + 1;
		curNode->noat = noat_flag;
		curNode->noreorder = noreorder;
		curNode->nomacro = nomacro;
		return(SUB_MODE);
	    } else if (LocateSymbolValue(yytext, SYM_JUMP, &weight) != NULL) {
		/** JUMP INSTRUCTION **/
		if (yylex() == ID)
		  AddAlternate(NULL, InsertSymbol(yytext, SYM_IDENT));
		ScanForRegisters();
		(curNode->count) += weight;
		curNode->last_jump = line_num;
		curNode->noreorder_jump = noreorder;
	    } else if (strcmp(yytext, lw_inst_ptr)==0) {

		/* check if lw is reading the cycle counter `current_time_',
		      `counting_on' or `counting_off' */
		/* don't need to Mark register, since either it's used
		   elsewhere or it can be overwritten */

		assert(yylex() == REG);
		typ = atoi(&yytext[1]); /* save register number */

		assert(yylex() == ',');  /* skip comma */

		token = yylex();   /* get lw source */

		if (token != ID) { /* no match, treat normally */
		    (curNode->count)++;
		    if (token == REG) MarkReg(atoi(&yytext[1]));
		    ScanForRegisters();
		} else if (strcmp(yytext, "current_time_")==0) {
		    /* reading cycle counter */
		    /* start new node, don't count this lw */
		    assert(curNode != NULL);
		    curNode->follow = NewNode();
		    curNode->end = line_num - 1;
		    curNode->optimize = TRUE;  /* can skip quantum check */
		    curNode->read_cycles = typ;

		    curNode = curNode->follow;
		    curNode->label = "#read current time";
		    curNode->start = line_num + 1;
		    curNode->noat = noat_flag;
		    curNode->noreorder = noreorder;
		    curNode->nomacro = nomacro;

		} else if (strcmp(yytext, "counting_on")==0) {

		    assert(curNode != NULL);
		    curNode->follow = NewNode();
		    curNode->end = line_num - 1;
		    curNode->optimize = TRUE;  /* can skip quantum check */

		    curNode = curNode->follow;
		    curNode->label = "#counting on";
		    curNode->change = ON;
		    curNode->start = line_num;
		    curNode->noat = noat_flag;
		    curNode->noreorder = noreorder;
		    curNode->nomacro = nomacro;

		} else if (strcmp(yytext, "counting_off")==0) {

		    assert(curNode != NULL);
		    curNode->follow = NewNode();
		    curNode->end = line_num - 1;
		    curNode->optimize = TRUE;  /* can skip quantum check */

		    curNode = curNode->follow;
		    curNode->label = "#counting off";
		    curNode->change = OFF;
		    curNode->start = line_num;
		    curNode->noat = noat_flag;
		    curNode->noreorder = noreorder;
		    curNode->nomacro = nomacro;

		} else { /* no match, treat normally */
		    (curNode->count)++;
		    if (token == REG) MarkReg(atoi(&yytext[1]));
		    ScanForRegisters();
		}
	    } else if (LocateSymbolValue(yytext, SYM_INST, &weight) != NULL) {
		/** OTHER INSTRUCTION **/
		(curNode->count) += weight;
		ScanForRegisters();
	    } else {
		/** NON-INSTRUCTION **/
		fprintf(stderr,
	       "augment: Unknown identifier \"%s\" at the start of line %d\n",
			yytext,	line_num);
		fprintf(stderr, "    ...treating as an instruction\n");
		(curNode->count)++;
		ScanForRegisters();
	    }
	    break;
	  case LABEL:
	    assert(!need_instruction);
	    yytext[strlen(yytext) - 1] = 0;  /* erase ':' */
	    label = InsertSymbol(yytext, SYM_LABEL);
	    if (curNode == NULL) {
		assert(startProc);
		root.node = curNode = NewNode();
		root.save31 = TRUE;  /* default to TRUE */
		startProc = curNode->unref = FALSE;
		/* root.prof_offset = -1; */
		root.proc = curProc;
		InitRegisters();
	    } else {
		curNode->end = line_num - 1;
		curNode->follow = NewNode();
		curNode = curNode->follow;
		curNode->unref = TRUE;
		curNode->label_block = TRUE;
	    }
	    curNode->start = line_num;
	    curNode->label = label;
	    curNode->noat = noat_flag;
	    curNode->noreorder = noreorder;
	    curNode->nomacro = nomacro;
	    SetSymbolNode(label, curNode);
	    break;
	  case DIR:
	    id = LocateSymbol(yytext, SYM_DIR);
	    if (id == end_dir) {
		curNode->end = line_num;
		curNode = NULL;
		FillInAltNodes();
		CompactGraph(root.node);
		root.useReg = FreeReg();
		EnqueueRoot(&root);
		if (verbose) {
		    printf("Subroutine \"%s\" completed.\n", curProc);
		    PrintGraph(root.node, 1);
		}
		while(yylex() != 0);
		return(TEXT_MODE);
	    } else if (id == frame_dir) {
		yylex(); /* skip stack pointer */
		yylex(); /* skip comma */
		assert(yylex() == OFFSET);
		/* recompute flags since code goes here */
		curNode->noat = noat_flag;
		curNode->noreorder = noreorder;
		curNode->nomacro = nomacro;
	    } else if (id == set_dir) {
		yylex();
		cmd = LocateSymbol(yytext, SYM_DIR);
		if (cmd==at_dir) noat_flag = FALSE;
		else if (cmd==noat_dir) noat_flag = TRUE;
		else if (cmd==reorder_dir) noreorder = FALSE;
		else if (cmd==noreorder_dir) noreorder = TRUE;
		else if (cmd==macro_dir) nomacro = FALSE;
		else if (cmd==nomacro_dir) nomacro = TRUE;
		if (curNode != NULL) {
		    if (curNode->start == line_num) {
			/* printf("Bumping start: %d\n", line_num); */
			curNode->start = line_num+1;
			curNode->noat = noat_flag;
			curNode->noreorder = noreorder;
			curNode->nomacro = nomacro;
		    }
		}
	    } else if (id == mask_dir) {
		assert(yylex() == NUM);
		root.save31 = !((strtol(yytext, NULL, 0) & 0x80000000));
	    } else if (id == word_dir) {
		if (yylex() == ID) {
		    AddAlternate(NULL, InsertSymbol(yytext, SYM_IDENT));
		}
	    }
	    while(yylex() != 0);
	    if (need_instruction) {
		NextLine(); ParseSub(TRUE);
	    }
	    return(SUB_MODE);
	  default:
	    printf("error: unexpected token: %d, line %d.\n", token,
		   line_num);
	    printf("yytext=\"%s\".\n", yytext);
	    exit(1);
	}
    }
    return(SUB_MODE);
}

    
/***************************************************************************/
/* output line of code */

static void OutputLine(char *buffer, FILE *out)
{
    static int next = 0;
    static libListType *lib;
    int i;

    /* fprintf(out, "line %d\n", line_num); */
    
    if (next == 0) {
	lib = DequeueLibCall();
	if (lib == NULL) {
	    next = -1;
	} else {
	    next = lib->line;
	}
    }

    if (next == line_num) {
	assert(strlen(buffer) >= lib->ch);
	for (i=0; i<lib->ch; i++) fputc(buffer[i], out);
	fprintf(out, "cyc_%s", lib->call);
	fputs(&buffer[lib->ch + strlen(lib->call)], out);
	next = 0;
    } else {
	fputs(buffer, out);
    }
}


/***************************************************************************/
/* output code to count cycles */

static int labelNum = 10000;

static void OutputCountCode(FILE *fil, FILE *out, nodeType *node,
			    int useReg, int save_reg_31, char *proc)
/* input file: fil
   output file: out
   basic block: node
   register to use: useReg   negative => must save/restore a register
                             -2 => do only restore, as save is already done
*/
{
    /* if no cycles or counting turned off exit */
    if (node->count == 0  ||  countStatus > 0) return;

    if (node->label_block) {  /* output label */
	fgets(tmp, LINE_MAX, fil);
	line_num++;
	OutputLine(tmp, out);
    }
    if (useReg == 1 && !node->noat) {
	fprintf(out, "\t.set noat\t\t# augment\n");
    } else if (useReg == 1) {
	fprintf(stderr,
		"augment: Warning: compiler using register $1, line %d\n",
		line_num);
    }

    if (node->nomacro) fprintf(out, "\t.set\tmacro\t\t# augment\n");
    if (node->noreorder) fprintf(out, "\t.set\treorder\t\t# augment\n");

    if (profiling) { /* update procedure cycle count */
	fprintf(out, "\tlw\t$%d, %s_P\t# augment\n", useReg, proc);
	fprintf(out, "\tadd\t$%d, $%d, %d\t# augment\n",
		useReg, useReg, node->count);
	fprintf(out, "\tsw\t$%d, %s_P\t# augment\n", useReg, proc);
    }
    
    fprintf(out, "\tlw\t$%d, cycles_\t# augment\n", useReg);
    fprintf(out, "\tsub\t$%d, $%d, %d\t# augment\n",
	    useReg, useReg, node->count);
    fprintf(out, "\tsw\t$%d, cycles_\t# augment\n", useReg);

    if (!node->optimize) {
	fprintf(out, "\tbgtz\t$%d, L%d\t# augment\n", useReg, labelNum);
	if (save_reg_31) fprintf(out, "\tsw\t$31, -4($sp)\t# augment\n");
	fprintf(out, "\tjal\tSimQuantum\t# augment\n");
	if (save_reg_31) fprintf(out, "\tlw\t$31, -4($sp)\t# augment\n");
	fprintf(out, "L%d:\t\t\t\t# augment\n", labelNum++);
    }
    if (node->noreorder) fprintf(out, "\t.set\tnoreorder\t# augment\n");
    if (node->nomacro) fprintf(out, "\t.set\tnomacro\t\t# augment\n");
    if (node->noreorder && !(node->optimize))
      fprintf(out, "\tnop\t\t\t# augment\n");

    if (useReg == 1 && !node->noat)
      fprintf(out, "\t.set at\t\t# augment\n");
}

/***************************************************************************/
/* output code to check stack */

static void OutputStackCheck(FILE *fil, FILE *out, int useReg, int save_reg_31)
{
    if (useReg == 1) useReg = 25;
    fprintf(out, "\tlw\t$%d, stackmin_\t# augment\n", useReg);
    if (save_reg_31) fprintf(out, "\tmove\t$24, $31\t# augment\n");
    fprintf(out, "\tsub\t$%d, $sp, $%d\t# augment\n", useReg, useReg);
    fprintf(out, "\tbgez\t$%d, L%d\t# augment\n", useReg, labelNum);
    fprintf(out, "\tjal\tSimStack\t# augment\n");
    if (save_reg_31) fprintf(out, "\tmove\t$31, $24\t# augment\n");
    fprintf(out, "L%d:\t\t\t\t# augment\n", labelNum++);
}
    
/***************************************************************************/
/* output profiling code for prologue/epilogue */

static void OutputProfilePrologue(FILE *out, int save_reg_31, char *proc)
{
    fprintf(out, "\tlw\t$25, %s_P + 4\t# augment\n", proc);
    fprintf(out, "\tbgtz\t$25, L%d\t# augment\n", labelNum);

    if (save_reg_31)
      fprintf(out, "\tmove\t$24, $31\t# augment: save reg $31\n");
    fprintf(out, "\tla\t$15, %s_P\t# augment\n", proc);
    fprintf(out, "\tjal\tInitProfile_\t# augment\n");
    if (save_reg_31)
      fprintf(out, "\tmove\t$31, $24\t# augment: restore reg $31\n");

    fprintf(out, "L%d:\taddiu\t$25, $25, 1\t# augment: inc number of calls\n",
	    labelNum++);
    fprintf(out, "\tsw\t$25, %s_P + 4\t# augment\n", proc);
}


/****************************************************************************
****************************************************************************/

/**  EXTERNAL ROUTINES  **/

/***************************************************************************/

void BuildGraph(FILE *fil)
{
    int mode;

    the_fil = fil;

    mode = TEXT_MODE;
    line_num = 0;
    while(NextLine()) {
	switch (mode) {
	  case TEXT_MODE:
	    mode = ParseText();
	    break;
	  case DATA_MODE:
	    mode = ParseData();
	    break;
	  case SUB_MODE:
	    mode = ParseSub(FALSE);
	    break;
	}
    }
}

   
/***************************************************************************/
/* output augmented file */


void AugmentSource(FILE *fil, FILE *out)
{
    nodeType *node;
    rootListType *curRoot;
    int block;
    BOOL prologue, found_frame;
    char *tok;

    line_num = 0;
    fprintf(out, " # created by augment version %.2f\n", version_number);
    fprintf(out, " # Eric A. Brewer  and  Chris N. Dellarocas\n");
    fprintf(out, " # MIT Laboratory for Computer Science\n\n");
    if (!cycles_declared) fprintf(out, "\t.extern cycles_ 4\n");
    if (!stackmin_declared) fprintf(out, "\t.extern stackmin_ 4\n");

    /* if profiling, output storage specification for profiling info */
    if (profiling) {
	fprintf(out, "\n # Specify storage for profiling info\n");
	for (curRoot = InitRootGen(); curRoot != NULL; curRoot = NextRoot()) {
	    fprintf(out, "\t.comm %s_P 8\n", curRoot->proc);
	}
	fprintf(out, " # End of profiling storage declarations\n\n");
    }

    curRoot = DequeueRoot();
    while (curRoot != NULL) {
	block = 1;
	prologue = TRUE;

	if (countStatus > 0)
	  fprintf(stderr,
		  "augment: Warning: Cycle-counting off at start of `%s'.\n",
		  curRoot->proc);

	for (node = curRoot->node;
	     (node != NULL)&&(!feof(fil));
	     node=node->follow) {
	    while(line_num < node->start - 1) {
		if (fgets(tmp, LINE_MAX, fil) == NULL) break;
		line_num++;
		OutputLine(tmp, out);
	    }
	    if (feof(fil)) break;
	    if (prologue) {
		prologue = FALSE;
		found_frame = FALSE;

		/* use next node's noat, since code added at end of block */
		if (node->follow != NULL) node->noat = node->follow->noat;

		while(line_num < node->end  && !found_frame) {
		    if (fgets(tmp, LINE_MAX, fil) == NULL) break;
		    line_num++;
		    OutputLine(tmp, out);
		    /* if (found_frame) break; */

		    tok = strtok(tmp, " \t\n");
		    if (tok == NULL) continue;

		    if (gcc == 1) { /* look for add to frame pointer */
			/* older versions use  "addiu  $30,$29,x",
			   newer versions use  "addu   $fp,$sp,x" */

			if (strcmp(tok, "addiu")==0) {
			    tok = strtok(NULL, " \t\n,");
			    assert(tok != NULL);
			    assert(tok[0] == '$');
			    if (strcmp(tok, "$30")==0) {
				found_frame = TRUE;
				if (profiling)
				  OutputProfilePrologue(out,
							curRoot->save31,
							curRoot->proc);
				OutputStackCheck(fil, out, curRoot->useReg,
						 curRoot->save31);
				OutputCountCode(fil, out, node,
						curRoot->useReg,
						curRoot->save31,
						curRoot->proc);
			    }
			} else if (strcmp(tok, "addu")==0) {
			    tok = strtok(NULL, " \t\n,");
			    assert(tok != NULL);
			    assert(tok[0] == '$');
			    if (strcmp(tok, "$fp")==0) {
				found_frame = TRUE;
				if (profiling)
				  OutputProfilePrologue(out,
							curRoot->save31,
							curRoot->proc);
				OutputStackCheck(fil, out, curRoot->useReg,
						 curRoot->save31);
				OutputCountCode(fil, out, node,
						curRoot->useReg,
						curRoot->save31,
						curRoot->proc);
			    }
			}
		    } else { /* cc source file -- look for .frame directive */
			if (strcmp(tok, frame_dir) == 0) {
			    found_frame = TRUE;
			    if (profiling)
			      OutputProfilePrologue(out, curRoot->save31,
						    curRoot->proc);
			    OutputStackCheck(fil, out, curRoot->useReg,
					     curRoot->save31);
			    OutputCountCode(fil, out, node, curRoot->useReg,
					    curRoot->save31, curRoot->proc);
			}
		    }
		}
		if (!found_frame) {
		    fprintf(stderr, "augment: Fatal Error:\n");
		    fprintf(stderr,
			    "\tUnable to find end of prologue in `%s'.\n",
			    curRoot->proc);
		    exit(1);
		}
	    } else {
		if (node->change != NO_CHANGE) {
		    if (node->change == ON) {
			if (countStatus == 0) {
			    fprintf(stderr,
				    "augment: Fatal Error:\n\t`counting_on' ");
			    fprintf(stderr,
				    "found in cycle-counted area, line %d\n",
				    node->start);
			    exit(1);
			}
			countStatus--;
			if (debug)
			  fprintf(stderr, "counting turned on, line %d\n",
				  node->start);
		    } else { /* turn counting off */
			countStatus++;
			if (debug)
			  fprintf(stderr, "counting turned off, line %d\n",
				  node->start);
		    }
		    
		    /* comment out "lw reg, counting_*" line */
		    fgets(tmp, LINE_MAX, fil);
		    fprintf(out, " # removed by augment: %s", tmp);
		    line_num++;
		}

		/* OUTPUT BASIC BLOCK */
		fprintf(out, " # block %d: %d instruction%c%s\n", block,
			node->count, (node->count != 1)?'s':' ',
			node->unref?" (unreferenced)":"");
		OutputCountCode(fil, out, node, curRoot->useReg,
				curRoot->save31, curRoot->proc);
	    }

	    if (node->follow == NULL) {
#ifdef DEBUG_EPILOGUE
		fprintf(stderr, "augmenting epilogue\n"); 
		fprintf(stderr, "\t jump line = %d, noreorder = %d\n",
			node->last_jump, node->noreorder_jump);
#endif
		/* output rest of block */
		while(line_num < node->end) {
		    if (fgets(tmp, LINE_MAX, fil) == NULL) break;
		    line_num++;
		    OutputLine(tmp, out);
#ifdef DEBUG_EPILOGUE
		    fprintf(stderr, "%d: ", line_num);
		    fputs(tmp, stderr);
#endif
		}
	    } else {
		/* output rest of block */
		while(line_num < node->end) {
		    if (fgets(tmp, LINE_MAX, fil) == NULL) break;
		    line_num++;
		    OutputLine(tmp, out);
		}
	    }

	    /* add code to read cycle counter if needed */
	    if (node->read_cycles) {
		fprintf(out,
			"\tlw\t$%d, base_time_\t\t# current time -> $%d\n",
			node->read_cycles, node->read_cycles);
		assert(node->follow != NULL);
		if (!(node->follow->noat)) fprintf(out, "\t.set noat\n");
		fprintf(out, "\tmove\t$1, $%d\n", node->read_cycles);
		fprintf(out, "\tlw\t$%d, cycles_\n", node->read_cycles);
		fprintf(out, "\tsubu\t$%d, $1, $%d\n",
			node->read_cycles, node->read_cycles);
		if (!(node->follow->noat)) fprintf(out, "\t.set at\n");
		
		/* skip "lw reg, current_time_" line */
		fgets(tmp, LINE_MAX, fil); 
		line_num++;
	    }
	    block++;
	}
	
	if (feof(fil)) break;
	curRoot = DequeueRoot();
    }
    if (curRoot != NULL) {
	fprintf(stderr,
		"augment: Unexpected end of file while augmenting subroutine \"%s\".\n",
		(curRoot->node)->label);
	exit(1);
    }
    while (fgets(tmp, LINE_MAX, fil) != NULL) {
	line_num++;
	OutputLine(tmp, out);
    }
}
