 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator                
 *       Eric A. Brewer, Chris N. Dellarocas, and Anthony D. Joseph
 *                     Laboratory for Computer Science                    
 *                  Massachusetts Institute of Technology                 
 *
 * Module: OS memory routines
 *
 * Description:                                                           
 *         Routines for simulated OS memory management
 *
 * Last Modified: $Date: 1996/05/13 20:33:01 $ ($Author: dfk $)
 * 
 * Originally written by Anthony D. Joseph 11-92
 *
 * Global Functions:                                                      
 *     char *_OS_getmem(ulong nbytes)			    [non-cycle-counted]
 *     char *_OS_getmemfrommodule(ulong nbytes, int module) [non-cycle-counted]
 *     int _OS_freemem(char *block, ulong nbytes)	    [non-cycle-counted]
 *     int _OS_free(char *block)			    [non-cycle-counted]
 *     char *OS_getmem(ulong nbytes)
 *     char *OS_getmemfrommodule(ulong nbytes, int module)
 *     int OS_freemem(char *block, ulong nbytes)
 *     int OS_free(char *block)
 *
 * Global Variables: None
 *     
 ****************************************************************************
 *   Copyright 1991, 1992                                                      
 *   Eric A. Brewer, Chris N. Dellarocas, and Anthony D. Joseph
 *   Massachusetts Institute of Technology                                
 *                                                                        
 *   Permission to use, copy, modify, and distribute this program         
 *   for any purpose and without fee is hereby granted, provided          
 *   that this copyright and permission notice appear on all copies       
 *   and supporting documentation, the name of M.I.T. not be used         
 *   in advertising or publicity pertaining to distribution of the        
 *   program without specific prior permission, and notice be given       
 *   in supporting documentation that copying and distribution is         
 *   by permission of M.I.T.  M.I.T. makes no representations about       
 *   the suitability of this software for any purpose.  It is pro-        
 *   vided "as is" without express or implied warranty.		          
 ****************************************************************************
 * $Header: /usr/wildcat/dfk/research/dmcache/src/proteus/RCS/OSmem.common.h,v 1.2 1996/05/13 20:33:01 dfk Exp $
 * $Log: OSmem.common.h,v $
 * Revision 1.2  1996/05/13 20:33:01  dfk
 * fixed precedence error
 *
 * Revision 1.1  1996/05/13 20:32:26  dfk
 * Initial revision
 *
 * Revision 1.1  92/11/25  17:45:43  brewer
 * Initial revision
 * 
 * 
 \**************************************************************************/

#include <malloc.h>
#include <stdlib.h> /* for qsort */

/* Debugging options:
   DEBUG  	Messages about (de)allocation amounts
   JOIN_DEBUG  	Messages about bucket coalescing
   SPLIT_DEBUG	Messages about bucket splitting
   ALLOC_DEBUG	Messages about allocation
*/   

/* comparison function to sort by address in ascending order */
static int addrcmp(const ShmemBlk *p1, const ShmemBlk *p2)
{
    if (p1 < p2) return(-1);
    if (p1 > p2) return(1);
    return(0);
}

/* Attempt to refill "bucket" by coalescing blocks in smaller buckets.
 *   First the bucket is sorted by increasing address. Next, the blocks
 *   in the bucket are scanned to form chains of adjacent blocks. If we
 *   can find a sufficient number of adjacent blocks, we coalesce them
 *   into one single block.
 */
static void coalesce_bucket(ShmemBlk **table, int bucket)
{
  int b, diff, mult, count, i;
  ShmemBlk *op, *chain, *prev, **entries;
  char *next;

#ifdef JOIN_DEBUG
  CCOFF;
  printf("(join) starting with bucket %d\n", bucket - 1);
  CCON;
#endif
  /* Current bucket is empty, try to refill it from smaller buckets */
  for (b = bucket - 1, mult = 2; b >= 0; b--, mult <<= 1) {
    if (INDIRECT (table + b) != NULL) {
      /* First, we sort bucket */
      op = INDIRECT (table + b);
      count = 0;
      while (op != NULL) {		/* Count the number of blocks */
	op = op GETELT ov_next;
	count++;
      }
#ifdef JOIN_DEBUG
      CCOFF;
      printf("Bucket %d has %d entries\n", b, count);
      CCON;
#endif
      /* Next, copy the elements from the bucket into an array */
      entries = (ShmemBlk **) malloc(count * sizeof(ShmemBlk *));
      op = INDIRECT (table + b);
      for (i = 0; i < count; i++, op = op GETELT ov_next)
	entries[i] = op;

      /* Now sort them */
      qsort((char *)entries, count, sizeof(ShmemBlk *), (int (*)())addrcmp);


      /* Finally, rechain them */
      INDIRECT (table + b) = op = entries[0];
      for (i = 1; i < count; i++) {
	op GETELT ov_next = entries[i];
	op = entries[i];
      }
      op GETELT ov_next = NULL;
      free(entries);

      /* Now we can try to form a big enough chain of adjacent blocks */
      op = chain = INDIRECT (table + b);
      prev = NULL;
      count = 1;
      while ((op != NULL) && (count < mult)) {
#ifdef JOIN_DEBUG
	CCOFF;
	printf("Count is %d\n", count);
	CCON;
#endif
	next = (char *) (op GETELT ov_next);
	diff = next - (char *) op;
	if (diff != BUCKET(b)) {
	  count = 1;
	  prev = op;
	  op = chain = (ShmemBlk *) next;
	  continue;
	} else
	  count++;
	op = (ShmemBlk *) next;
      }
      if (count >= mult) {	/* We found enough */
#ifdef JOIN_DEBUG
	CCOFF;
	printf("Found match, moving %d blocks from bucket %d to bucket %d\n",
	       count, b, bucket);
	CCON;
#endif
	INDIRECT (table + bucket) = chain;
	table[bucket] GETELT ov_next = NULL;
	if (prev == NULL) {
	  INDIRECT (table + b) = op GETELT ov_next;
	} else {
	  prev GETELT ov_next = op GETELT ov_next;
	}
	return;
      }
    }
  }
}

/*
 * Try to refill "bucket" by splitting blocks in larger buckets.
 *   Splitting a larger block yields multiple smaller blocks, at most
 *   "size" of which, we put into "bucket"
 */
static void split_bucket(ShmemBlk **table, int bucket, int size)
{
  ShmemBlk *op;
  int b, count, diff, i;

  /* Current bucket is empty, try to refill it from larger buckets */
#ifdef SPLIT_DEBUG
  CCOFF;
  printf("(split) Trying to fill bucket %d with %d * %d\n", bucket, size,
	 BUCKET(bucket));
  CCON;
#endif
  for (diff = BUCKET(bucket+1), b=bucket+1; b < NBUCKETS; b++, diff<<=1) {
    if ((op = INDIRECT (table + b)) != NULL) {	/* Look for first non-empty */
#ifdef SPLIT_DEBUG
      CCOFF;
      printf("Bucket %d has entry of size %d\n", b, BUCKET(b));
      CCON;
#endif
      INDIRECT (table + b) = op GETELT ov_next;
      INDIRECT (table + bucket) = op;
      if (b == (bucket+1)) {	/* Only two blocks, return them */
#ifdef SPLIT_DEBUG
	CCOFF;
	printf("Found two entries for %d, max %d, using 2\n", bucket, size);
	CCON;
#endif
	op = (op GETELT ov_next = (ShmemBlk *)((char *) op + BUCKET(bucket)));
	op GETELT ov_next = NULL;
	return;
      }
      if (size < (count = (1<<(b-bucket))))	/* Take only size at most */
	count = size;
#ifdef SPLIT_DEBUG
      CCOFF;
      printf("Found %d entries for %d, max %d, using %d\n",
	     (1<<(b-bucket)), bucket, size, count);
      CCON;
#endif
      /* Chain in new blocks */
      for (i = 1, size = BUCKET(bucket); i < count; i++) {
	op = (op GETELT ov_next = (ShmemBlk *)((char *) op + size));
      }
      diff -= (count * size);
      op GETELT ov_next = NULL;
      op = (ShmemBlk *)((char *) op + size);
#ifdef SPLIT_DEBUG
      CCOFF;
      printf("%d left for other buckets\n", diff);
      CCON;
#endif
      while (diff && (--b > bucket)) {	/* Split up the remainder */
	if ((size = BUCKET(b)) <= diff) {
	  INDIRECT (table + b) = op; 
	  diff -= size;
#ifdef SPLIT_DEBUG
	  CCOFF;
	  printf("Filling bucket %d, %d left\n", b, diff);
	  CCON;
#endif
	  while (size <= diff) {
#ifdef SPLIT_DEBUG
	    CCOFF;
	    printf("Another block for %d\n", b);
	    CCON;
#endif
	    op GETELT ov_next = (ShmemBlk *)((char *) op + size);
	    op = (ShmemBlk *)((char *) op + size);
	    diff -= size;
	  }
	  op GETELT ov_next = NULL;
	  op = (ShmemBlk *)((char *) op + size);
	}
      }
      CCOFF;
      if (diff != 0)	/* As long as size is a power of 2, diff will be 0 */
	fatal("split_bucket: %d left after all buckets filled\n", diff);
      CCON;
      return;
    }
  }
}

/*
 * OS_getmem procedure: Allocate shared memory.
 *   Attempts to allocate memory using the following algorithm:
 *	Memory is allocated from buckets (each a power of 2 minus some
 *	  overhead).
 *	1) Tries locally for exact match on bucket
 *	2) Tries locally to split a bucket
 *	3) Tries remotely for exact match on bucket
 *	4) Tries locally to coalesce a bucket
 *	5) Tries remotely to coalesce a bucket
 *
 *	The choice is made in terms of allocation cost. An alternative
 *	method would be to coalesce locally before trying to get an
 *	exact match or split remotely.
 *
 *   Assumes NO_OF_PROCESSORS == NO_OF_MODULES
 */
VISIBLE char *GETMEMFCN(ulong nbytes) 
{
  ShmemBlk *op = NULL;
  int bucket = 0;		/* first bucket */
  unsigned amt = BUCKET(0);	/* size of first bucket */
  int module, count;
  int over = nbytes + OVERHEAD;

  /*
   * Convert amount of memory requested into closest block size
   * stored in hash buckets which satisfies request.
   * Account for space used per block for accounting.
   */

#ifdef DEBUG
  printf("Allocating %d bytes\n", nbytes);
#endif
  while (over > amt) {
    amt <<= 1;
    bucket++;
    if (bucket >= NBUCKETS) {
#ifdef FATAL
      fatal("OS_getmem: cannot allocate %d bytes", nbytes);
#else
      return( (char *)NULL );
#endif
    }
  }

#ifdef ALLOC_DEBUG
  printf("Trying bucket %d\n", bucket);
#endif

  SEMP(CURR_PROCESSOR);
  if ((op = INDIRECT (nextf + bucket)) != NULL) {	/* Step 1 */
    /* remove from linked list */
    INDIRECT (nextf + bucket) = op GETELT ov_next;
    SEMV(CURR_PROCESSOR);
    op GETELT ov_magic = MAGIC;
    op GETELT ov_index = bucket;
    op GETELT ov_module = CURR_PROCESSOR;
#ifdef RCHECK
    CCOFF;
    /*
     * Record allocated size of block and
     * bound space with magic numbers.
     */
    op->ov_size = (nbytes + RSLOP - 1) & ~(RSLOP - 1);
    op->ov_rmagic = FRMAGIC;
    *(u_short *)((caddr_t)(op + 1) + op->ov_size) = RMAGIC;
    CCON;
#endif
#ifdef ALLOC_DEBUG
    printf("returning 0x%x\n", op);
#endif
    CCOFF;
    MemoryCount[CURR_PROCESSOR] += BUCKET(bucket);
    CCON;
    SEMV(CURR_PROCESSOR);
    return ((char *)(op + 1));
  } else {
    SEMV(CURR_PROCESSOR);
    
    /* Steps 2 and 3 */
    for (module = CURR_PROCESSOR, count = 0;
	 (count < NO_OF_MODULES) && (op == NULL);
	 count++, module = (++module)%NO_OF_MODULES) {
      SEMP(module);
      if ((op = INDIRECT (MemoryTable[module] + bucket)) != NULL) break;

      /* Current bucket is empty, try to refill it */
      split_bucket(MemoryTable[module], bucket, SPLITSIZE);
      if ((op = INDIRECT (MemoryTable[module] + bucket)) != NULL)
	break;
      else {
	  SEMV(module);
      }
    }
    
    if (op == NULL) { /* Steps 4 and 5 */
      for (module = CURR_PROCESSOR, count = 0;
	   (count < NO_OF_MODULES) && (op == NULL);
	   count++, module = (++module)%NO_OF_MODULES) {
	SEMP(module);
#ifdef ALLOC_DEBUG
	CCOFF;
	printf("No shared memory available for %d byte request in module %d, compacting...\n",
	       nbytes, module);
	CCON;
#endif
	coalesce_bucket(MemoryTable[module], bucket);
	if ((op = INDIRECT (MemoryTable[module] + bucket)) == NULL) {
	  SEMV(module);
	}
      }
      if (op == NULL) {
#ifdef FATAL
	fatal("OS_getmem: cannot allocate %d bytes from any module\n", nbytes);
#else
	return( (char *)NULL );
#endif
      }
#ifdef ALLOC_DEBUG
      CCOFF;
      printf("Successful\n");
      CCON;
#endif
    }

    /* remove from linked list */
    INDIRECT (MemoryTable[module] + bucket) = op GETELT ov_next;
    CCOFF;
    MemoryCount[module] += BUCKET(bucket);
    CCON;
    SEMV(module);
    op GETELT ov_magic = MAGIC;
    op GETELT ov_index = bucket;
    op GETELT ov_module = module;
#ifdef RCHECK
    CCOFF;
    /*
     * Record allocated size of block and
     * bound space with magic numbers.
     */
    op->ov_size = (nbytes + RSLOP - 1) & ~(RSLOP - 1);
    op->ov_rmagic = FRMAGIC;
    *(u_short *)((caddr_t)(op + 1) + op->ov_size) = RMAGIC;
    CCON;
#endif
#ifdef ALLOC_DEBUG
    CCOFF;
    printf("returning 0x%x\n", op);
    CCON;
#endif
    return ((char *)(op + 1));
  }
}

/*
 * OS_getmemfrommodule procedure: Allocate shared memory from current
 *   processor's memory module.
 *
 *   Attempts to allocate memory using the following algorithm:
 *	Memory is allocated from buckets (each a power of 2 minus some
 *	  overhead).
 *	1) Tries locally for exact match on bucket
 *	2) Tries locally to split a bucket
 *	3) Tries locally to coalesce a bucket
 *
 *
 *   Assumes NO_OF_PROCESSORS == NO_OF_MODULES
 */
VISIBLE char *GETMEMMDLFCN(ulong nbytes, int module) 
{
  ShmemBlk *op = NULL, **table;
  int bucket = 0;		/* first bucket */
  unsigned amt = BUCKET(0);	/* size of first bucket */
  int over = nbytes + OVERHEAD;

  /*
   * Convert amount of memory requested into closest block size
   * stored in hash buckets which satisfies request.
   * Account for space used per block for accounting.
   */

#ifdef DEBUG
  CCOFF;
  printf("Allocating %d bytes (%d w/ overhead) from module %d\n",
	 nbytes, nbytes + OVERHEAD, module);
  CCON;
#endif
  while (over > amt) {
    amt <<= 1;
    bucket++;
    CCOFF;
    if (bucket >= NBUCKETS) {
#ifdef FATAL
      fatal("OS_getmemfrommodule: cannot allocate %d bytes from module %d", nbytes,
	    module);
#else
      return( (char *)NULL );
#endif
    }
    CCON;
  }

#ifdef ALLOC_DEBUG
  CCOFF;
  printf("Trying bucket %d\n", bucket);
  CCON;
#endif

  SEMP(module);
  
  /* Step 1 */
  if ((op = INDIRECT ((table = MemoryTable[module]) + bucket)) == NULL) {
    /* Current bucket is empty, try to refill it */
#ifdef ALLOC_DEBUG
    CCOFF;
    printf("Bucket empty\n");
    CCON;
#endif
    split_bucket(table, bucket, SPLITSIZE);	/* Step 2 */
    if ((op = INDIRECT (table + bucket)) == NULL) {
#ifdef ALLOC_DEBUG
      CCOFF;
      printf("No shared memory available for %d byte request in module %d, compacting...\n",
	       nbytes, module);
      CCON;
#endif
      coalesce_bucket(table, bucket);	/* Step 3 */
      if ((op = INDIRECT (table + bucket)) == NULL) {
	SEMV(module);
#ifdef FATAL
	fatal("OS_getmemfrommodule: cannot allocate %d bytes in module %d (%u used / %u max\n",
	      nbytes, module, (unsigned int) MemoryCount[module],
	      MaxSharedMemory);
#else
	return( (char *)NULL );
#endif
      }
#ifdef ALLOC_DEBUG
      CCOFF;
      printf("Successful\n");
      CCON;
#endif
    }
  }

  /* remove from linked list */
  INDIRECT (table + bucket) = op GETELT ov_next;
  CCOFF;
  MemoryCount[module] += BUCKET(bucket);
  CCON;
#ifdef DEBUG
  CCOFF;
  printf("Using bucket of size %d, %d of %d used\n", BUCKET(bucket),
	 MemoryCount[module], MaxSharedMemory);
  CCON;
#endif
  SEMV(module);
  op GETELT ov_magic = MAGIC;
  op GETELT ov_index = bucket;
  op GETELT ov_module = module;
#ifdef RCHECK
  CCOFF;
  /*
   * Record allocated size of block and
   * bound space with magic numbers.
   */
  op->ov_size = (nbytes + RSLOP - 1) & ~(RSLOP - 1);
  op->ov_rmagic = FRMAGIC;
  *(u_short *)((caddr_t)(op + 1) + op->ov_size) = RMAGIC;
  CCON;
#endif
#ifdef ALLOC_DEBUG
  CCOFF;
  printf("returning 0x%x/0x%x\n", op, (char *)(op + 1));
  CCON;
#endif
  return ((char *)(op + 1));
}

/*
 * OS_freemem procedure: Return a block of shared memory to the memory pool.
 *   Returns a block of shared memory to the appropriate bucket. Performs
 *   validation checks to guarantee block is valid (avoids multiple frees
 *   frees of illegal pointers). Also verifies that size of block being
 *   returned matches nbytes, if range-checking is enabled. Can also perform
 *   range checking.
 */
VISIBLE int FREEMEMFCN(char *block, ulong nbytes) 
{
  int module;
  ShmemBlk *op, **bucket;

  CCOFF;
  if (block == NULL) {
#ifdef FATAL
    fatal("OS_freemem: Null free request");
#else
    return(ERROR);
#endif
  }
  CCON;

  op = (ShmemBlk *)((caddr_t)block - sizeof (ShmemBlk));

  CCOFF;
#ifdef DEBUG
  printf("Freeing memory item from bucket %d in module %d\n", op->ov_index,
	 op->ov_module);
#endif

  if (op->ov_magic != MAGIC) 		/* make sure it was in use */
#ifdef FATAL
    fatal("OS_freemem: Block being freed doesn't have valid magic number.\nProbable causes: Already freed block, overwrite error at location 0x%x\n",
	  &(op->ov_magic));
#else
    return(ERROR);
#endif

#ifdef RCHECK
  /* First validate front of block */
  if (op->ov_rmagic != FRMAGIC)
#ifdef FATAL
    fatal("OS_freemem: Block being freed doesn't have valid front range magic number.\nProbable cause: Overwrite error at location 0x%x\n",
	  &(op->ov_rmagic));
#else
    return(ERROR);
#endif

  /* Now, validate the end of the block */
  if (*(u_short *)((caddr_t)(op + 1) + op->ov_size) != RMAGIC)
#ifdef FATAL
    fatal("OS_freemem: Block being freed doesn't have valid end range magic number.\nProbable cause: Overwrite error at location 0x%x\n",
	  (u_short *)((caddr_t)(op + 1) + op->ov_size));
#else
    return(ERROR);
#endif

  /* Finally, validate the size of the block */
  if (op->ov_size != ((nbytes + RSLOP - 1) & ~(RSLOP - 1)))
#ifdef FATAL
    fatal("OS_freemem: Size of block being freed doesn't match computed size\nProbable causes: Program error, Overwrite error at location 0x%x\n",
	  &(op->ov_size));
#else
    return(ERROR);
#endif

#endif	/* RCHECK */

  if (op->ov_index >= NBUCKETS)
#ifdef FATAL
    fatal("OS_freemem: Block being freed doesn't have valid bucket index.\nProbable cause: Overwrite error at location 0x%x\n",
	  &(op->ov_index));
#else
    return(ERROR);
#endif
    
  CCON;

  module = op GETELT ov_module;

  SEMP(module);

  CCOFF;
  MemoryCount[module] -= BUCKET(op->ov_index);
  CCON;

  op GETELT ov_next =	/* also clobbers ov_magic */
    INDIRECT ((bucket = (MemoryTable[module] + op GETELT ov_index)));
  INDIRECT bucket = op;

  SEMV(module);
  return(OK);
}

/*
 * OS_free procedure: Return a block of shared memory to the memory pool.
 *   Returns a block of shared memory to the appropriate bucket. Performs
 *   validation checks to guarantee block is valid (avoids multiple frees
 *   frees of illegal pointers). Can also perform range checking.
 */
VISIBLE int FREEFCN(char *block) 
{
  int module;
  ShmemBlk *op, **bucket;

  CCOFF;
  if (block == NULL) {
#ifdef FATAL
    fatal("OS_freemem: Null free request");
#else
    return(ERROR);
#endif
  }
  CCON;

  op = (ShmemBlk *)((caddr_t)block - sizeof (ShmemBlk));

  CCOFF;
#ifdef DEBUG
  printf("Freeing memory item from bucket %d in module %d\n", op->ov_index,
	 op->ov_module);
#endif

  if (op->ov_magic != MAGIC) 		/* make sure it was in use */
#ifdef FATAL
    fatal("OS_freemem: Block being freed doesn't have valid magic number.\nProbable causes: Already freed block, overwrite error at location 0x%x\n",
	  &(op->ov_magic));
#else
    return(ERROR);
#endif

#ifdef RCHECK
  /* First validate front of block */
  if (op->ov_rmagic != FRMAGIC)
#ifdef FATAL
    fatal("OS_freemem: Block being freed doesn't have valid front range magic number.\nProbable cause: Overwrite error at location 0x%x\n",
	  &(op->ov_rmagic));
#else
    return(ERROR);
#endif

  /* Now, validate the end of the block */
  if (*(u_short *)((caddr_t)(op + 1) + op->ov_size) != RMAGIC)
#ifdef FATAL
    fatal("OS_freemem: Block being freed doesn't have valid end range magic number.\nProbable cause: Overwrite error at location 0x%x\n",
	  (u_short *)((caddr_t)(op + 1) + op->ov_size));
#else
    return(ERROR);
#endif

#endif	/* RCHECK */

  if (op->ov_index >= NBUCKETS)
#ifdef FATAL
    fatal("OS_freemem: Block being freed doesn't have valid bucket index.\nProbable cause: Overwrite error at location 0x%x\n",
	  &(op->ov_index));
#else
    return(ERROR);
#endif
    
  CCON;

  module = op GETELT ov_module;

  SEMP(module);

  CCOFF;
  MemoryCount[module] -= BUCKET(op->ov_index);
  CCON;

  op GETELT ov_next =	/* also clobbers ov_magic */
    INDIRECT ((bucket = (MemoryTable[module] + op GETELT ov_index)));
  INDIRECT bucket = op;

  SEMV(module);
  return(OK);
}

