 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator                
 *       Eric A. Brewer, Chris N. Dellarocas, and Anthony D. Joseph
 *                     Laboratory for Computer Science                    
 *                  Massachusetts Institute of Technology                 
 *
 * Module: OS memory internal routines
 *
 * Description:                                                           
 *         Routines for simulated OS memory management
 *
 * Last Modified: $Date: 92/12/09 15:19:06 $ ($Author: brewer $)
 * 
 * Originally written by Anthony D. Joseph 11-92
 *
 * Global Functions:                                                      
 *     void init_shmem(void)
 *     void initshared(void)
 *     void dump_module(FILE *out, int module)
 *     void view_shared_memory(FILE *out) 
 *     void print_memory_statistics(FILE *out)
 *     void diagnostic_shared_memory(void)
 *
 * Global Variables:                                                      
 *     ShmemBlk *shmem_pool
 *     Word memlist_mutex[NO_OF_MODULES]
 *     ShmemBlk **nextf
 *     ShmemBlk **MemoryTable[NO_OF_MODULES]
 *     int MemoryCount[NO_OF_MODULES]
 *     int MaxSharedMemory
 *     
 ****************************************************************************
 *   Copyright 1991, 1992                                                      
 *   Eric A. Brewer, Chris N. Dellarocas, and Anthony D. Joseph
 *   Massachusetts Institute of Technology                                
 *                                                                        
 *   Permission to use, copy, modify, and distribute this program         
 *   for any purpose and without fee is hereby granted, provided          
 *   that this copyright and permission notice appear on all copies       
 *   and supporting documentation, the name of M.I.T. not be used         
 *   in advertising or publicity pertaining to distribution of the        
 *   program without specific prior permission, and notice be given       
 *   in supporting documentation that copying and distribution is         
 *   by permission of M.I.T.  M.I.T. makes no representations about       
 *   the suitability of this software for any purpose.  It is pro-        
 *   vided "as is" without express or implied warranty.		          
 ****************************************************************************
 * $Header: /psg/proteus/RCS/OSmem_internal.c,v 1.4 92/12/09 15:19:06 brewer Exp $
 * $Log:	OSmem_internal.c,v $
 * Revision 1.4  92/12/09  15:19:06  brewer
 * updated output procedures to take FILE *
 * 
 * Revision 1.3  92/11/25  17:46:05  brewer
 * Anthony's revised version.
 * 
 * Revision 1.2  92/04/01  11:26:16  brewer
 * cleanup: fixed gcc warnings
 * 
 * Revision 1.1  92/02/11  13:54:17  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include "OSmem.h"

GLOBAL ShmemBlk **nextf;	/* local processor ptr to local bucket list */

/* base of shared memory (after semaphores) */
GLOBAL ShmemBlk *shmem_pool = (ShmemBlk *)&memory[MAX_SEMAPHORES*LINE_SIZE];

/* array of ptrs to bucket lists (treated as a list local to each proc) */
GLOBAL ShmemBlk **MemoryTable[NO_OF_MODULES]; 
GLOBAL int MemoryCount[NO_OF_MODULES];		/* space used in module */
GLOBAL int MaxSharedMemory;			/* Max useable space in mod. */
GLOBAL Word memlist_mutex[NO_OF_MODULES];	/* access ctrl semaphores */

static void initshared();

/* define access and cycle counting primitives */
#define GETELT 		->
#define INDIRECT	*
#define GETMEMFCN 	_OS_getmem
#define GETMEMMDLFCN	_OS_getmemfrommodule
#define FREEMEMFCN	_OS_freemem
#define FREEFCN		_OS_free
#define SEMP(MOD)
#define SEMV(MOD)
#define CCOFF
#define CCON
#define VISIBLE		static

#include "OSmem.common.h"

/* force references to _OS_getmem, _OS_getmemfrommodule, _OS_freemem,
   and _OS_free; this prevents bogus warnings when there are no
   "statically" allocated shared memory objects */
static void *dummy_ref[] = { (void *)_OS_getmem, (void *)_OS_getmemfrommodule,
			     (void *)_OS_freemem, (void *)_OS_free,
			     (void *)dummy_ref};

/*
 * Display the bucket lists for a module.
 *  For best results, the semaphore for the module should be acquired. 
 *  This will guarantee the lists are in a consistent state. Also, computes
 *  allocated memory and compares to expected allocation amount.
 */
GLOBAL void dump_module(FILE *fp, int module)
{
  ShmemBlk *op;
  int i, count, sanity;

  fprintf(fp, "Memory module for %d: %d bytes used of %d\n", module,
	 MemoryCount[module], MaxSharedMemory);
  for (sanity = i = 0; i < NBUCKETS; i++) {
    count = 0;
    op = MemoryTable[module][i];
    while (op != NULL) {
      if (count && ((count % 5) == 0)) fprintf(fp, "\n");
      fprintf(fp, "0x%lx -> ", (ulong)op);
      op = op->ov_next;
      count++;
    }
    if (count) fprintf(fp, "NIL\n");
    sanity += (count * BUCKET(i));
    fprintf(fp, "[%d] Bucket %d contains %d entries of size %u/%u\n",
	   module, i, count, BUCKET(i)-OVERHEAD, BUCKET(i));
  }
  sanity = MaxSharedMemory - sanity;
  if (sanity != MemoryCount[module])
    fprintf(fp, "Memory allocation check failed! %u used not %u\n", sanity,
	   MemoryCount[module]);	  
}


/* 
 * Initialize shared memory for all processors.
 *  Determines the largest bucket and allocates the shared memory in each
 *  module to the bucket lists.
 *  Initializes access control semaphores.
 */
GLOBAL void init_shmem() 
{
  char *start[NO_OF_MODULES], *base;
  ShmemBlk *op;
  char buf[30];
  int module, i, j, maxBucket, space, count;

  if (NO_OF_PROCESSORS != NO_OF_MODULES)
    fprintf(stderr, "\nWarning! Number of shared memory modules different from number of processors\n");

  /* Initialize the per processor local variable to the bucket list */
  define_local((char **)&nextf,(char *)shmem_pool, (ulong)MODULE_SIZE);

  /* Find the largest bucket size */
  for (i = maxBucket = 0; i < NBUCKETS; i++)
    if (BUCKET(i) > MAXSPACE)
      break;
    else
      maxBucket = i;
#ifdef DEBUG
    printf("Max bucket is %d Available memory %d\n", maxBucket, MAXSPACE);
#endif

  /* Set the starting point for all modules */
  base = (char *)shmem_pool;
  for (module = 0; module < NO_OF_MODULES; module++, base += MODULE_SIZE) {
    MemoryTable[module] = (ShmemBlk **) base;
    start[module] = (base + (NBUCKETS * sizeof(ShmemBlk *)));
#ifdef DEBUG
    printf("Memory table for %d at 0x%x Memory at 0x%x\n", module,
	   MemoryTable[module], start[module]);
#endif

    /* Zero out all buckets */
    for (i = 0; i < NBUCKETS; i++)
      MemoryTable[module][i] = NULL;
  }

  /* Now split up the memory */
  for (MaxSharedMemory = 0, space = MAXSPACE, i = maxBucket; i > 0; i--) {
    count = space / BUCKET(i);
    if (!count) continue;
    space -= (count * BUCKET(i));
    MaxSharedMemory += (count * BUCKET(i));
#ifdef DEBUG
    printf("Bucket %d contains %d entries of size %d\n", i, count,
	   BUCKET(i));
#endif
    for (module = 0; module < NO_OF_MODULES; module++) {
      MemoryTable[module][i] = op = (ShmemBlk *)(start[module]);
      start[module] += BUCKET(i);
      for (j = 1; j < count; j++) {
	op->ov_next = (ShmemBlk *)(start[module]);
	start[module] += BUCKET(i);
	op = (ShmemBlk *)(start[module]);
      }
      op->ov_next = NULL;
    }
  }
  
#ifdef DEBUG
  printf("MaxSharedMemory is %d\n", MaxSharedMemory);
  dump_module(stdout, 0);
#endif

  /* Allocate individual module semaphores */
  for (module = 0; module < NO_OF_MODULES; module++) {
    sprintf(buf, "SM Mod %d", module);
    memlist_mutex[module] = sem_open(1, buf, module);
  }
    
  init_array_stat(STAT_CACHE, NO_OF_PROCESSORS, 0, 100);
    
  /* "Statically" allocate memory for user program's shared variables */
  initshared();
}

/*
 * Display the allocation buckets for all memory modules 
 */
GLOBAL void view_shared_memory(FILE *fp) 
{
  int module;

  for (module = 0; module < NO_OF_MODULES; module++) {
    fprintf(fp, "Memory Module %d:\n", module);
    dump_module(fp, module);
    putc('\n', fp);
  }
}

/*
 * Display usage statistics for all memory modules
 */
GLOBAL void print_memory_statistics(FILE *fp)
{
  int module;

  for (module = 0; module < NO_OF_MODULES; module++) {
    fprintf(fp, "Memory module %d: %u bytes used of %u\n", module, 
	   (unsigned int) MemoryCount[module], MaxSharedMemory);

  }
}

/*
 * Verify that records of shared memory allocation match remaining shared
 *   memory. Done for all processors.
 */
GLOBAL void diagnostic_shared_memory()
{
  ShmemBlk *op;
  int module, i, count, sanity, failed;

  printf("\nPerforming sanity check to verify that the counter of allocated\n");
  printf("shared memory matches the actual amount of allocated memory.\n");
  printf("\n>>> This test may fail if the module is in use. <<<\n\n");

  for (module = failed = 0; module < NO_OF_MODULES; module++) {
    printf("Module %d...", module);
    for (sanity = i = 0; i < NBUCKETS; i++) {
      count = 0;
      op = MemoryTable[module][i];
      while (op != NULL) {
	op = op->ov_next;
	count++;
      }
      sanity += (count * BUCKET(i));
    }
    sanity = MaxSharedMemory - sanity;
    if (sanity != MemoryCount[module]) {
      printf("FAILED");
      failed = 1;
    } else
      printf("PASSED --- %u bytes used of %u max", 
	     (unsigned int) MemoryCount[module], MaxSharedMemory);
    if (S_VALUE(memlist_mutex[module]))
      printf(" (module NOT in use)\n");
    else
      printf(" (module in use)\n");
  }
  if (!failed)
    printf("All modules verified\n");
}


#include "initshared"
