
 /****************************************************************************
 * 
 *                     Parallel Architecture Simulator    
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science    
 *                  Massachusetts Institute of Technology 
 * 
 * Module: cache.h
 *
 * Description: defines cache constants and macros
 * 
 * Last Modified: $Date: 92/11/10 14:32:28 $  (eab)
 * 
 * Data Structures:
 *     CacheLine       storage for data of one cache line
 *     CacheSet        One set of associated lines
 * 
 * Constants:
 *     CACHE_SIZE
 *     SET_MASK
 *     TAG_BITS
 *     TAG_MASK
 *     LINE_SIZE
 *     LINE_MASK
 *
 * Macros:
 *     GetSetFromAddr(a)
 *     GetTagFromAddr(a)
 *     AddrInCache(p,a)           address of cached copy of address
 *     FebitInCache(p,a)          address of F/E bit for cached address
 *     CachedAddr(p, s, l)        address cached in proc=p, set=s, line=l
 *     CacheState(p, s, l)        state of the above line
 *     CacheTag(p, s, l)          tag of the above line
 *     RoundAddrToLine(a)         compute address of start of line for a
 *
 *     ReadCacheData(p, s, l)     the next 5 manipulated cached copies
 *     WriteCacheData(p, s, l, d)
 *     ReadCacheFebits(p, s, l)
 *     WriteCacheFebits(p, s, l, d)
 *     WriteCacheLine(a, d)
 *     ChooseLineToEvict(p, a)
 *
 * Variables:
 *     CacheSet cache[NO_OF_PROCESSORS][CACHE_SIZE]
 *
 ***************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.		
 ***************************************************************************
 * $Header: /psg/proteus/RCS/cache.h,v 1.3 92/11/10 14:32:28 brewer Exp $
 * $Log:	cache.h,v $
 * Revision 1.3  92/11/10  14:32:28  brewer
 * cleaned up header
 * Added LINE_MASK definition for NOCACHE case
 * 
 * Revision 1.2  92/04/01  17:01:57  brewer
 * cleaned up gcc warnings
 * 
 * Revision 1.1  92/02/11  14:19:15  brewer
 * Initial revision
 * 
 \**************************************************************************/

#ifndef CACHE_H
#define CACHE_H

#include "cache.param"

#ifndef NOCACHE


#ifdef WATCH_CACHE
#define CACHE_HIT \
      SUM_ARRAY_STAT(STAT_CACHE, processor_, 1, CURRENTTIME)
#define CACHE_MISS \
      SUM_ARRAY_STAT(STAT_CACHE, processor_, 0, CURRENTTIME)
#else
#define CACHE_HIT
#define CACHE_MISS
#endif /* ifdef WATCH_CACHE */


#define CACHE_SIZE  (1 << CACHE_BITS)
#define SET_MASK    (CACHE_SIZE-1)

#define TAG_BITS    (CACHE_BITS+LINE_BITS)
#define TAG_MASK    ((1 << (sizeof(Word)*8-TAG_BITS))-1)

#define LINE_SIZE   (1 << LINE_BITS)
#define LINE_MASK   (LINE_SIZE-1)


typedef struct {
          char d[LINE_SIZE];
	} CacheLine;

typedef struct {
          Word       ctag[SET_SIZE];
	  short      state[SET_SIZE];
#ifdef CACHEWITHDATA
	  CacheLine  data[SET_SIZE];
	  CacheLine  febit[SET_SIZE];
#endif
	} CacheSet;

CacheSet cache[NO_OF_PROCESSORS][CACHE_SIZE];

/* GetSetFromAddr returns the index within the cache */
#define GetSetFromAddr(a)      ( ((a) >> LINE_BITS) & SET_MASK )

/* GetTagFromAddr returns the tag to be compared with the set's tag fields */
#define GetTagFromAddr(a)      ( ((a) >> TAG_BITS ) & TAG_MASK  )

/* GetOffFromAddr returns the offset within a cache line */
#define GetOffFromAddr(a)      ( (a) & LINE_MASK )


/* AddrInCache returns the memory address where a particular shared 
 * memory address is cached in our processor's cache
 */
#ifdef CACHEWITHDATA
#define AddrInCache(p,a)  (&cache[p][GetSetFromAddr(a)].data[GetLineFromAddr((p),(a))].d[GetOffFromAddr(a)])
#define FebitInCache(p,a) (cache[p][GetSetFromAddr(a)].febit[GetLineFromAddr((p),(a))].d[GetOffFromAddr(a)])
#else
#define AddrInCache(p,a)  (&memory[a])
#define FebitInCache(p,a) (memtag[a])
#endif /* ifdef CACHEWITHDATA */


/* CachedAddr returns the address cached in set s, line l of processor p */
#define CachedAddr(p, s, l)       ((Word)(((cache[p][s].ctag[l]) << TAG_BITS) | ((s) << LINE_BITS)))


/* CacheState and CacheTag return the state and tag of a given cache line */
#define CacheState(p, s, l)       (cache[p][s].state[l])
#define CacheTag(p, s, l)         (cache[p][s].ctag[l])


#ifdef CACHEWITHDATA
#define ReadCacheData(p, s, l) \
    ((Word *)&cache[p][s].data[l])
#define WriteCacheData(p, s, l, d) \
    (cache[p][s].data[l] = *(CacheLine *)(d))
#define ReadCacheFebits(p, s, l) \
    ((Word *)&cache[p][s].febit[l])
#define WriteCacheFebits(p, s, l, d) \
    (cache[p][s].febit[l] = *(CacheLine *)(d))
#else
#define ReadCacheData(p, s, l)       NULL
#define WriteCacheData(p, s, l, d)
#define ReadCacheFebits(p, s, l)     NULL
#define WriteCacheFebits(p, s, l, d)
#endif /* ifdef CACHEWITHDATA */

#define CacheData                 ReadCacheData
#define CacheFebits               ReadCacheFebits


/* WriteCacheLine transfers a CacheLine from address d to address a */
#ifdef CACHEWITHDATA
#define WriteCacheLine(a, d) \
  (*(CacheLine *)(a) = *(CacheLine *)(d))
#else
#define WriteCacheLine(a, d)
#endif


/* ChooseLineToEvict returns the line number to be evicted from the cache
 * in order to make room for another line.
 */
#define ChooseLineToEvict(p, a)   (((p)^(a)) % SET_SIZE)


/* Round address a to the start address on the line that contains it */
#define RoundAddrToLine(a)         ((a) & (~LINE_MASK))

#else /* NOCACHE is defined */

/* pick line size for semaphore code */
#define LINE_SIZE    4
#define LINE_MASK    3

#endif /* ifndef NOCACHE */


void protocol_fence();
int processor_request_pre(Time *time, int reqtype, int processor, int tid,
			  ulong address);
int processor_request_post(Time *time, int reqtype, int processor,
			   int tid, ulong address);
#endif /* ifndef CACHE_H */
