
 /****************************************************************************
 *                                                        
 *                     Parallel Architecture Simulator    
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science    
 *                  Massachusetts Institute of Technology 
 *
 * Module: cache.net.c
 *
 * Description:  Chaiken cache-coherence protocol
 *
 * Last Modified: $Date: 94/01/24 00:39:06 $
 *
 * Global Functions:
 *     void init_cache(void) 
 *     void protocol_fence(void)
 *     int processor_request_pre(Time *time, int reqtype, int processor,
 *				 int tid, ulong address) 
 *     int processor_request_post(Time *time, int reqtype, int processor,
 *                                int tid, ulong address)
 *     void MtoC_Pkt(sendtid, time, sender, target, pkt) 
 *     void CtoM_Pkt(sendtid, time, sender, target, pkt)
 *     void Nocache_Pkt()
 *
 * Global Variables:
 *     BOOL monitor_cache
 *     char *protocol_msg_[]       Names for protocol messages
 *     
 ****************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.
 ****************************************************************************
 * $Header: /usr/wildcat/dfk/research/parallel/proteus/proteus-V3.01/engine/RCS/cache.net.c,v 1.2 94/01/24 00:39:06 dfk Time64bit Locker: dfk $
 * $Log:	cache.net.c,v $
 * Revision 1.2  94/01/24  00:39:06  dfk
 * Added Sanjay's support for 64-bit timers.
 * Added thread_sleep_until.
 * Fixed a few bugs in thread sleep/wakeup code (my bugs).
 * Fixed printf formats in many places.
 * 
 * Revision 1.1  94/01/23  11:34:13  dfk
 * Initial revision
 * 
 * Revision 1.6  92/11/19  14:15:56  brewer
 * Updated calls to build_new_packet_.
 * 
 * Revision 1.5  92/11/03  14:15:25  brewer
 * Changed message sending code to prevent multiple messages per cycle
 * from the same processor.
 * 
 * Revision 1.4  92/10/28  11:16:38  brewer
 * Added typecasts to reduce warnings.
 * 
 * Revision 1.3  92/05/01  17:25:30  brewer
 * Added substantial debugging information (ifdef CCDEBUG)
 * Cleaned up code
 * Added INVARIANT statements
 * Improved error messages
 * 
 * Revision 1.2  92/04/01  17:02:28  brewer
 * cleaned up gcc warnings, added ANSI prototypes
 * 
 * Revision 1.1  92/02/11  13:55:13  brewer
 * Initial revision
 * 
 \**************************************************************************/


/* #define CCDEBUG */
#define CC_SAFE

#ifdef CC_SAFE
#define INVARIANT2(x, y) \
  if (!(x)) { fprintf(stderr, "cache.net.c: address=0x%x\n", address); \
     fatal(y); }
#define INVARIANT3(x, y, z) \
  if (!(x)) { fprintf(stderr, "cache.net.c: address=0x%x\n", address); \
     fatal(y, z); }
#define INVARIANT4(x, y, z, w) \
  if (!(x)) { fprintf(stderr, "cache.net.c: address=0x%x\n", address); \
     fatal(y, z, w); }
#else
#define INVARIANT2(x, y)
#define INVARIANT3(x, y, z)
#define INVARIANT4(x, y, z, w)
#endif

#include "shmem.h"
#include "net.h"
#include "packet.h"
#include "simcalls.h"
#include "rt_thread_def.h"
#include "simreq.h"
#include "processor.h"


/* Cache States */

#define C_INVALID             0
#define C_READONLY_NETW       1
#define C_READONLY            2
#define C_READWRITE           3
#define C_READWRITE_NETW      4
#define C_INVALID_NETW        5
#define C_VALID_NETW          6
#define C_VALID               7
#define C_DIRTY               8
#define C_DIRTY_NETW          9

static char *cache_state_[] = {"INVALID", "READONLY_NETW", "READONLY",
				 "READWRITE", "READWRITE_NETW", "INVALID_NETW",
				 "VALID_NETW", "VALID", "DIRTY",
				 "DIRTY_NETW"};

/* Directory States */

#define D_ABSENT              0
#define D_READONLY            1
#define D_READWRITE           2
#define D_READTRANS           3
#define D_WRITETRANS          4

#define MULT_READERS         -1   /* directory.proc entry to indicate
				     multiple readers in READONLY state */

static char *directory_state_[] = {"ABSENT", "READONLY", "READWRITE",
				     "READTRANS", "WRITETRANS"};

/* Protocol Message Types */

#define M_RREQ                0
#define M_WREQ                1
#define M_MREQ                2
#define M_REPU                3
#define M_REPM                4
#define M_UPDATE              5
#define M_ACKC                6
#define M_INV                 7
#define M_RDATA               8
#define M_WDATA               9
#define M_MODG                10
#define M_BUSY                11
#define M_FETCH               12
#define M_FLUSH               13
#define M_FDATA               14
#define M_FACK                15

GLOBAL char *protocol_msg_[] = {"RREQ", "WREQ", "MREQ",
				  "REPU", "REPM", "UPDATE",
				  "ACKC", "INV", "RDATA",
				  "WDATA", "MODG", "BUSY",
				  "FETCH", "FLUSH", "FDATA",
				  "FACK"};

/* Macros for Protocol Messages */

#define CTOM_RREQ(t, p, a, l)     send_ctom_packet(t,M_RREQ,p,a,l,NULL,NULL)
#define CTOM_WREQ(t, p, a, l)     send_ctom_packet(t,M_WREQ,p,a,l,NULL,NULL)
#define CTOM_MREQ(t, p, a, l)     send_ctom_packet(t,M_MREQ,p,a,l,NULL,NULL)
#define CTOM_REPU(t, p, a, l)     send_ctom_packet(t,M_REPU,p,a,l,NULL,NULL)
#define CTOM_ACKC(t, p, a, l)     send_ctom_packet(t,M_ACKC,p,a,l,NULL,NULL)
#define CTOM_INV(t, p, a, l)      send_ctom_packet(t,M_INV ,p,a,l,NULL,NULL)
#define CTOM_MODG(t, p, a, l)     send_ctom_packet(t,M_MODG,p,a,l,NULL,NULL)
#define CTOM_BUSY(t, p, a, l)     send_ctom_packet(t,M_BUSY,p,a,l,NULL,NULL)
#define CTOM_FETCH(t, p, a, l)    send_ctom_packet(t,M_FETCH,p,a,l,NULL,NULL)
#define CTOM_FACK(t, p, a, l)     send_ctom_packet(t,M_FACK,p,a,l,NULL,NULL)
#ifdef CACHEWITHDATA
#define CTOM_REPM(t,p,a,l,d,f)   send_ctom_packet(t, M_REPM, p, a, l, d,  f  )
#define CTOM_UPDATE(t,p,a,l,d,f) send_ctom_packet(t, M_UPDATE, p, a, l,d, f  )
#define CTOM_RDATA(t,p,a,l,d,f)  send_ctom_packet(t, M_RDATA, p, a, l, d, f  )
#define CTOM_WDATA(t,p,a,l,d,f)  send_ctom_packet(t, M_WDATA, p, a, l, d, f  )
#define CTOM_FLUSH(t,p,a,l,d,f)  send_ctom_packet(t, M_FLUSH, p, a, l, d, f  )
#define CTOM_FDATA(t,p,a,l,d,f)  send_ctom_packet(t, M_FDATA, p, a, l, d, f  )
#else
#define CTOM_REPM(t,p,a,l,d,f)   send_ctom_packet(t, M_REPM,p,a,l,NULL,NULL )
#define CTOM_UPDATE(t,p,a,l,d,f) send_ctom_packet(t, M_UPDATE,p,a,l,NULL,NULL)
#define CTOM_RDATA(t,p,a,l,d,f)  send_ctom_packet(t, M_RDATA,p,a,l,NULL,NULL )
#define CTOM_WDATA(t,p,a,l,d,f)  send_ctom_packet(t, M_WDATA,p,a,l,NULL,NULL )
#define CTOM_FLUSH(t,p,a,l,d,f)  send_ctom_packet(t, M_FLUSH,p,a,l,NULL,NULL )
#define CTOM_FDATA(t,p,a,l,d,f)  send_ctom_packet(t, M_FDATA,p,a,l,NULL,NULL )
#endif

#define MTOC_RREQ(t, p, a, l)     send_mtoc_packet(t,M_RREQ,p,a,l,NULL,NULL)
#define MTOC_WREQ(t, p, a, l)     send_mtoc_packet(t,M_WREQ,p,a,l,NULL,NULL)
#define MTOC_MREQ(t, p, a, l)     send_mtoc_packet(t,M_MREQ,p,a,l,NULL,NULL)
#define MTOC_REPU(t, p, a, l)     send_mtoc_packet(t,M_REPU,p,a,l,NULL,NULL)
#define MTOC_ACKC(t, p, a, l)     send_mtoc_packet(t,M_ACKC,p,a,l,NULL,NULL)
#define MTOC_INV(t, p, a, l)      send_mtoc_packet(t,M_INV ,p,a,l,NULL,NULL)
#define MTOC_MODG(t, p, a, l)     send_mtoc_packet(t,M_MODG,p,a,l,NULL,NULL)
#define MTOC_BUSY(t, p, a, l)     send_mtoc_packet(t,M_BUSY,p,a,l,NULL,NULL)
#define MTOC_FETCH(t, p, a, l)    send_mtoc_packet(t,M_FETCH,p,a,l,NULL,NULL)
#define MTOC_FACK(t, p, a, l)     send_mtoc_packet(t,M_FACK,p,a,l,NULL,NULL)
#ifdef CACHEWITHDATA
#define MTOC_REPM(t,p,a,l,d,f)   send_mtoc_packet(t, M_REPM, p, a, l,  d, f  )
#define MTOC_UPDATE(t,p,a,l,d,f) send_mtoc_packet(t, M_UPDATE, p, a, l,d. f  )
#define MTOC_RDATA(t,p,a,l,d,f)  send_mtoc_packet(t, M_RDATA, p, a, l, d, f  )
#define MTOC_WDATA(t,p,a,l,d,f)  send_mtoc_packet(t, M_WDATA, p, a, l, d, f  )
#define MTOC_FLUSH(t,p,a,l,d,f)  send_mtoc_packet(t, M_FLUSH, p, a, l, d, f  )
#define MTOC_FDATA(t,p,a,l,d,f)  send_mtoc_packet(t, M_FDATA, p, a, l, d, f  )
#else
#define MTOC_REPM(t,p,a,l,d,f)   send_mtoc_packet(t,M_REPM,p,a,l,NULL,NULL )
#define MTOC_UPDATE(t,p,a,l,d,f) send_mtoc_packet(t,M_UPDATE,p,a,l,NULL,NULL )
#define MTOC_RDATA(t,p,a,l,d,f)  send_mtoc_packet(t,M_RDATA,p,a,l,NULL,NULL )
#define MTOC_WDATA(t,p,a,l,d,f)  send_mtoc_packet(t,M_WDATA,p,a,l,NULL,NULL )
#define MTOC_FLUSH(t,p,a,l,d,f)  send_mtoc_packet(t,M_FLUSH,p,a,l,NULL,NULL )
#define MTOC_FDATA(t,p,a,l,d,f)  send_mtoc_packet(t,M_FDATA,p,a,l,NULL,NULL )
#endif


/* Main Memory & Directory Structure */
#define DIRECTORY_SIZE            10
#define BLOCK_SIZE                LINE_SIZE

typedef struct {
    short state;
    short ackctr;
    short dircnt;
    short dir [DIRECTORY_SIZE];
    short line[DIRECTORY_SIZE];
} DirBlk;

static DirBlk directory[MEMSIZE/BLOCK_SIZE];


static short fencectr[NO_OF_PROCESSORS];
static int   waitsfence[NO_OF_PROCESSORS];

static BOOL tried_once[MAX_THREADS];
static BOOL thrash_wait[NO_OF_PROCESSORS];

extern int AddressToProcessor(ulong address);

static void cache_controller(), memory_controller();

static Time next_out_msg[NO_OF_PROCESSORS];

/*************************************************************************\
* debugging procedures
\*************************************************************************/

#ifdef CCDEBUG

GLOBAL BOOL monitor_cache = FALSE;
#define DEBUG_SET 2
#define CACHE_MSG(p, req, t) \
  if (p>0) IndexEvent(EV_STATE, p, req, t)
#define MEM_EVENT(t, s) \
  if (address==16) IndexEvent(EV_STATE, mproc, s, t)


static void ViewCacheLine(char *d)
{
    int i;
    printf("%lx: ", (ulong)d);
    for(i=0; i<LINE_SIZE;i++)
      printf("%02x ", *d++);
    printf("\n");
}

static void PrintDirectory(DirBlk *dir)
{
    int i;

    printf("Directory: ");
    for (i=0; i<dir->dircnt; i++) {
	printf("%d ", dir->dir[i]);
    }
    if (i==0) printf("empty\n");
    else printf("\n");
}

#else /* not CCDEBUG */
#define CACHE_MSG(p, r, t)
#define MEM_EVENT(t, s)
#endif


/*************************************************************************\
* initialize cache
\*************************************************************************/

GLOBAL void init_cache() 
{
    int i;
    
    for(i=0; i<NO_OF_PROCESSORS; i++)
      waitsfence[i] = -1;
}



/*************************************************************************\
* protocol fence
\*************************************************************************/

GLOBAL void protocol_fence() 
{
    extern int currtid;
    
    if (fencectr[processor_] > 0) {
	waitsfence[processor_] = currtid;
	TrapToSimulator();
    }
    waitsfence[processor_] = -1;
}


/*************************************************************************\
* send a message from cache controller to memory controller
\*************************************************************************/

static void send_ctom_packet(Time time, int msgtype, int processor,
			     Word address, int line, char *data, char *febits) 
{
    int pktlen = 1 + (data != NULL ? LINE_SIZE : 0);
    /* eventually will need protocol_pktlen */
    
    Packet *packet = build_new_packet_(PKT_MEM, pktlen, 5, CTOM_PKT, 
				       msgtype, processor, address, line);
    
    if (data != NULL ) {
	WriteCacheLine(&packet->data[5], data);
	WriteCacheLine(((char *)&packet->data[5])+LINE_SIZE, febits);
    }
    
    time = next_out_msg[processor] = MAX(time, next_out_msg[processor] + 1);

    MAKE_REQUEST_4(SC_SEND_PACKET, -1, time,
		   processor, AddressToProcessor(address), packet, FALSE);
    
#ifdef CCDEBUG
    if (monitor_cache && (GetSetFromAddr(address))==DEBUG_SET) {
	printf("Proc %d: sending %s to %d, address 0x%x, time %lu\n",
	       processor, protocol_msg_[msgtype],
	       AddressToProcessor(address), address, time);
    }
    CACHE_MSG(processor, msgtype+2, time);
    CACHE_MSG(processor, 0, time+4);
#endif
}



/*************************************************************************\
* send a message from memory controller to cache controller
\*************************************************************************/

static void send_mtoc_packet(Time time, int msgtype, int processor,
			     Word address, int line, char *data, char *febits) 
{
    int from = AddressToProcessor(address);
    int pktlen = 1 + (data != NULL ? LINE_SIZE : 0);
    /* eventually will need protocol_pktlen */
    
    Packet *packet = build_new_packet_(PKT_MEM, pktlen, 5, MTOC_PKT, 
				       msgtype, processor, address, line);
    
    if (data != NULL ) {
	WriteCacheLine(&packet->data[5], data);
	WriteCacheLine(((char *)&packet->data[5])+LINE_SIZE, febits);
    }    
    
    time = next_out_msg[from] = MAX(time, next_out_msg[from] + 1);

    MAKE_REQUEST_4(SC_SEND_PACKET, -1, time,
		   from, processor, packet, FALSE);
    
#ifdef CCDEBUG
    if (monitor_cache && (GetSetFromAddr(address))==DEBUG_SET) {
	printf("Proc %d: sending %s to %d, address 0x%x, time %lu\n",
	       from, protocol_msg_[msgtype],
	       processor, address, time);
    }
    if (address==16) {
	CACHE_MSG(1, msgtype+2, time);
	CACHE_MSG(1, 0, time+3);
    }
#endif
}


/*************************************************************************\
* processor_request_pre: handle request from processor to cache controller
\*************************************************************************/

#define switch_or_wait(p,t) \
  (  tried_once[t] ? \
   (thrash_wait[p] = TRUE, MEM_WAIT) : \
   (tried_once[t]  = TRUE, MEM_SWITCH) )

#define clear_thrash(p,t)         {tried_once[t] = thrash_wait[p] = FALSE;}

#ifdef PREDEBUG
#define RETURN(x) {if (monitor_cache && set==DEBUG_SET) printf("Proc %d: pre done, new state = %s, returning %s\n", processor, cache_state_[*state], cc_response_[x]); return x; }
#else
#define RETURN(x) return(x)
#endif


GLOBAL int processor_request_pre(Time *time, int reqtype, int processor,
				 int tid, ulong address) 
{
    int set = GetSetFromAddr(address);
    int line = GetLineFromAddr(processor, address);
    short *state;
    
#ifdef PREDEBUG
    if (monitor_cache && set==DEBUG_SET) {
	printf("Proc %d: request %s, %s, time %lu\n",
	       processor, mr_request_[reqtype],
	       line>=0 ? "Tag Match" : "Tag Mismatch", *time);
    }
#endif
    
    *time += CACHE_ACCESS_LATENCY;
    
    
    if (line >= 0 ) {
	/* Actions when the tag matches */
	state = &CacheState(processor, set, line);
	
#ifdef PREDEBUG
	if (monitor_cache && set==DEBUG_SET) {
	    printf("\tstate %s, address=0x%x, set=%d\n",
		   cache_state_[CacheState(processor, set, line)],
		   address, set);
	}
#endif
	
	switch( *state ) {
	  case C_INVALID:
	    switch( reqtype ) {
	      case MR_READ:
		if (!tried_once[tid]) CACHE_MISS;
		CTOM_RREQ(*time, processor, address, line);
		*state = C_INVALID_NETW;	    
		RETURN(switch_or_wait(processor, tid));
	      case MR_WRITE:
		if (!tried_once[tid]) CACHE_MISS;
		CTOM_WREQ(*time, processor, address, line);
		*state = C_INVALID_NETW;
		RETURN(switch_or_wait(processor, tid));
	      case MR_READSOFT:
	      case MR_WRITESOFT:
		if (!tried_once[tid]) CACHE_MISS;
		CTOM_FETCH(*time, processor, address, line);
		*state = C_INVALID_NETW;
		RETURN(MEM_SWITCH);
	      case MR_FLUSHSOFT:
		RETURN(MEM_READY);
	    }
	    break;
	  case C_INVALID_NETW:
	    switch( reqtype ) {
	      case MR_READ:
	      case MR_WRITE:
		if (!tried_once[tid]) CACHE_MISS;
		RETURN( thrash_wait[processor] ? MEM_WAIT : MEM_SWITCH );	    
	      case MR_READSOFT:
	      case MR_WRITESOFT:
		if (!tried_once[tid]) CACHE_MISS;
		RETURN(MEM_SWITCH);
	      case MR_FLUSHSOFT:
		RETURN(MEM_READY);
	    }
	    break;
	  case C_READONLY_NETW:
	    switch( reqtype ) {
	      case MR_READ:
		CACHE_HIT;
		clear_thrash(processor, tid);
		RETURN(MEM_READY);
	      case MR_WRITE:
		if (!tried_once[tid]) CACHE_MISS;
		RETURN( thrash_wait[processor] ? MEM_WAIT : MEM_SWITCH );
	      case MR_READSOFT:
		CACHE_HIT;
		clear_thrash(processor, tid);
		RETURN(MEM_READY);
	      default:
		RETURN(MEM_ERROR);
	    }
	    break;
	  case C_READONLY:
	    switch( reqtype ) {
	      case MR_READ:
		CACHE_HIT;
		clear_thrash(processor, tid);
		RETURN(MEM_READY);
	      case MR_WRITE:
		if (!tried_once[tid]) CACHE_MISS;
		CTOM_WREQ(*time, processor, address, line);
		*state = C_READONLY_NETW;
		RETURN( switch_or_wait(processor, tid) );
	      case MR_READSOFT:
		CACHE_HIT;
		clear_thrash(processor, tid);
		RETURN(MEM_READY);
	      default:
		RETURN(MEM_ERROR);
	    }
	    break;
	  case C_READWRITE:
	    switch( reqtype ) {
	      case MR_READ:
	      case MR_WRITE:
	      case MR_READSOFT:
		CACHE_HIT;
		clear_thrash(processor, tid);
		RETURN(MEM_READY);
	      default:
		RETURN(MEM_ERROR);
	    }
	    break;
	  case C_READWRITE_NETW:
	    switch( reqtype ) {
	      case MR_READ:
	      case MR_WRITE:
	      case MR_READSOFT:
		CACHE_HIT;
		clear_thrash(processor, tid);
		RETURN(MEM_READY);
	      default:
		RETURN(MEM_ERROR);
	    }
	    break;
	  case C_VALID_NETW:
	    switch( reqtype ) {
	      case MR_READ:
	      case MR_WRITE:
		if (!tried_once[tid]) CACHE_MISS;
		RETURN( thrash_wait[processor] ? MEM_WAIT : MEM_SWITCH );
	      case MR_READSOFT:
		clear_thrash(processor, tid);
		RETURN(MEM_READY);
	      case MR_WRITESOFT:
		*state = C_DIRTY_NETW;
		RETURN(MEM_WAIT);
	      case MR_FLUSHSOFT:
		*state = C_INVALID_NETW;
		RETURN(MEM_WAIT);
	    }
	    break;
	  case C_VALID:
	    switch( reqtype ) {
	      case MR_READ:
		if (!tried_once[tid]) CACHE_MISS;
		CTOM_RREQ(*time, processor, address, line);
		*state = C_VALID_NETW;
		RETURN( switch_or_wait(processor, tid) );
	      case MR_WRITE:
		if (!tried_once[tid]) CACHE_MISS;
		CTOM_WREQ(*time, processor, address, line);
		*state = C_VALID_NETW;
		RETURN( switch_or_wait(processor, tid) );
	      case MR_READSOFT:
		CACHE_HIT;
		clear_thrash(processor, tid);
		RETURN(MEM_READY);
	      case MR_WRITESOFT:
		*state = C_DIRTY;
		RETURN(MEM_WAIT);
	      case MR_FLUSHSOFT:
		*state = C_INVALID;
		RETURN(MEM_WAIT);
	    }
	    break;
	  case C_DIRTY:
	    switch( reqtype ) {
	      case MR_READSOFT:
	      case MR_WRITESOFT:
		CACHE_HIT;
		clear_thrash(processor, tid);
		RETURN(MEM_READY);
	      case MR_FLUSHSOFT:
		fencectr[processor]++;
		CTOM_FLUSH(*time, processor, address, line,
			   (char *) CacheData(processor, set, line),
			   (char *) CacheFebits(processor, set, line));
		*state = C_INVALID;
		RETURN(MEM_WAIT);
	      default:
		RETURN(MEM_ERROR);
	    }
	    break;
	  case C_DIRTY_NETW:
	    switch( reqtype ) {
	      case MR_READSOFT:
	      case MR_WRITESOFT:
		CACHE_HIT;
		clear_thrash(processor, tid);
		RETURN(MEM_READY);
	      case MR_FLUSHSOFT:
		fencectr[processor]++;
		CTOM_FLUSH(*time, processor, address, line,
			   (char *) CacheData(processor, set, line),
			   (char *) CacheFebits(processor, set, line));
		*state = C_INVALID_NETW;
		RETURN(MEM_WAIT);
	      default:
		RETURN(MEM_ERROR);
	    }
	}
    } else {
	/* Actions when the tag does not match */
	int victim = ChooseLineToEvict(processor, address);
	
	state = &CacheState(processor, set, victim);
	
	/*    CacheTag(processor, set, victim) = GetTagFromAddr(address); */
	
#ifdef PREDEBUG
	if (monitor_cache && set==DEBUG_SET) {
	    printf("\tstate %s, address=0x%x, set=%d\n",
		   cache_state_[CacheState(processor, set, victim)],
		   address, set);
	}
#endif
	
	switch( *state ) {
	  case C_INVALID:
	    switch( reqtype )
	      {
		case MR_READ:
		  if (!tried_once[tid]) CACHE_MISS;
		  CTOM_RREQ(*time, processor, address, victim);
		  *state = C_INVALID_NETW;	    
		  RETURN( switch_or_wait(processor, tid) );
		case MR_WRITE:
		  if (!tried_once[tid]) CACHE_MISS;
		  CTOM_WREQ(*time, processor, address, victim);
		  *state = C_INVALID_NETW;
		  RETURN( switch_or_wait(processor, tid) );
		case MR_READSOFT:
		case MR_WRITESOFT:
		  if (!tried_once[tid]) CACHE_MISS;
		  CTOM_FETCH(*time, processor, address, victim);
		  *state = C_INVALID_NETW;
		  RETURN(MEM_SWITCH);
		case MR_FLUSHSOFT:
		  RETURN(MEM_READY);
	      }
	    break;
	  case C_INVALID_NETW:
	    switch( reqtype )
	      {
		case MR_READ:
		case MR_WRITE:
		  if (!tried_once[tid]) CACHE_MISS;
		  RETURN( thrash_wait[processor] ? MEM_WAIT : MEM_SWITCH);
		case MR_READSOFT:
		case MR_WRITESOFT:
		  if (!tried_once[tid]) CACHE_MISS;
		  RETURN(MEM_SWITCH);
		case MR_FLUSHSOFT:
		  RETURN(MEM_READY);
	      }
	    break;
	  case C_READONLY_NETW:
	    switch( reqtype )
	      {
		case MR_READ:
		case MR_WRITE:
		  if (!tried_once[tid]) CACHE_MISS;
		  RETURN( thrash_wait[processor] ? MEM_WAIT : MEM_SWITCH );
		case MR_READSOFT:
		case MR_WRITESOFT:
		  if (!tried_once[tid]) CACHE_MISS;
		  RETURN( MEM_SWITCH );
		case MR_FLUSHSOFT:
		  RETURN(MEM_READY);
	      }
	    break;
	  case C_READONLY:
	    switch( reqtype )
	      {
		case MR_READ:
		  if (!tried_once[tid]) CACHE_MISS;
		  CTOM_RREQ(*time, processor, address, victim);
		  *state = C_READONLY_NETW;
		  RETURN( switch_or_wait(processor, tid) );
		case MR_WRITE:
		  if (!tried_once[tid]) CACHE_MISS;
		  CTOM_WREQ(*time, processor, address, victim);
		  *state = C_READONLY_NETW;
		  RETURN( switch_or_wait(processor, tid) );
		case MR_READSOFT:
		case MR_WRITESOFT:
		  if (!tried_once[tid]) CACHE_MISS;
		  CTOM_FETCH(*time, processor, address, victim);
		  *state = C_READONLY_NETW;
		  RETURN(MEM_SWITCH);
		case MR_FLUSHSOFT:
		  RETURN(MEM_READY);
	      }
	    break;
	  case C_READWRITE:
	    switch( reqtype )
	      {
		case MR_READ:
		  if (!tried_once[tid]) CACHE_MISS;
		  CTOM_RREQ(*time, processor, address, victim);
		  *state = C_READWRITE_NETW;	    
		  RETURN( switch_or_wait(processor, tid) );
		case MR_WRITE:
		  if (!tried_once[tid]) CACHE_MISS;
		  CTOM_WREQ(*time, processor, address, victim);
		  *state = C_READWRITE_NETW;
		  RETURN( switch_or_wait(processor, tid) );
		case MR_READSOFT:
		case MR_WRITESOFT:
		  if (!tried_once[tid]) CACHE_MISS;
		  CTOM_FETCH(*time, processor, address, victim);
		  *state = C_READWRITE_NETW;
		  RETURN(MEM_SWITCH);
		case MR_FLUSHSOFT:
		  RETURN(MEM_READY);
	      }
	    break;
	  case C_READWRITE_NETW:
	    switch( reqtype )
	      {
		case MR_READ:
		case MR_WRITE:
		  if (!tried_once[tid]) CACHE_MISS;
		  RETURN( thrash_wait[processor] ? MEM_WAIT : MEM_SWITCH);
		case MR_READSOFT:
		case MR_WRITESOFT:
		  if (!tried_once[tid]) CACHE_MISS;
		  RETURN(MEM_SWITCH);
		case MR_FLUSHSOFT:
		  RETURN(MEM_READY);
	      }
	    break;
	  case C_VALID_NETW:
	    switch( reqtype )
	      {
		case MR_READ:
		case MR_WRITE:
		  if (!tried_once[tid]) CACHE_MISS;
		  RETURN(MEM_SWITCH);
		case MR_READSOFT:
		case MR_WRITESOFT:
		  if (!tried_once[tid]) CACHE_MISS;
		  RETURN( thrash_wait[processor] ? MEM_WAIT : MEM_SWITCH);
		case MR_FLUSHSOFT:
		  RETURN(MEM_READY);
	      }
	    break;
	  case C_VALID:
	    switch( reqtype )
	      {
		case MR_READ:
		  if (!tried_once[tid]) CACHE_MISS;
		  CTOM_RREQ(*time, processor, address, victim);
		  *state = C_VALID_NETW;	    
		  RETURN( switch_or_wait(processor, tid));
		case MR_WRITE:
		  if (!tried_once[tid]) CACHE_MISS;
		  CTOM_WREQ(*time, processor, address, victim);
		  *state = C_VALID_NETW;
		  RETURN( switch_or_wait(processor, tid));
		case MR_READSOFT:
		case MR_WRITESOFT:
		  if (!tried_once[tid]) CACHE_MISS;
		  CTOM_FETCH(*time, processor, address, victim);
		  *state = C_VALID_NETW;
		  RETURN(MEM_SWITCH);
		case MR_FLUSHSOFT:
		  RETURN(MEM_READY);
	      }
	    break;
	  case C_DIRTY:
	    switch( reqtype )
	      {
		case MR_READ:
		  if (!tried_once[tid]) CACHE_MISS;
		  CTOM_RREQ(*time, processor, address, victim);
		  *state = C_DIRTY_NETW;	    
		  RETURN( switch_or_wait(processor, tid));
		case MR_WRITE:
		  if (!tried_once[tid]) CACHE_MISS;
		  CTOM_WREQ(*time, processor, address, victim);
		  *state = C_DIRTY_NETW;
		  RETURN( switch_or_wait(processor, tid));
		case MR_READSOFT:
		case MR_WRITESOFT:
		  if (!tried_once[tid]) CACHE_MISS;
		  CTOM_FETCH(*time, processor, address, victim);
		  *state = C_DIRTY_NETW;
		  RETURN(MEM_SWITCH);
		case MR_FLUSHSOFT:
		  RETURN(MEM_READY);
	      }
	    break;
	  case C_DIRTY_NETW:
	    switch( reqtype )
	      {
		case MR_READ:
		case MR_WRITE:
		  if (!tried_once[tid]) CACHE_MISS;
		  RETURN( thrash_wait[processor] ? MEM_WAIT : MEM_SWITCH );
		case MR_READSOFT:
		case MR_WRITESOFT:
		  if (!tried_once[tid]) CACHE_MISS;
		  RETURN(MEM_SWITCH);
		case MR_FLUSHSOFT:
		  RETURN(MEM_READY);
	      }
	}
    }
    fatal("ERROR in processor_request_pre in cache.net.c\n");
    RETURN(ERROR);  /* unreachable */
}


/*************************************************************************\
* handle cleanup after processor request completes -- no action
\*************************************************************************/

GLOBAL int processor_request_post(Time *time, int reqtype, int processor,
				  int tid, ulong address) 
{
    return(MEM_READY);
}


/*************************************************************************\
* handler for messages from memory to cache controller
\*************************************************************************/

GLOBAL void MtoC_Pkt(sendtid, time, sender, target, pkt) 
     int sendtid ;
     Time time ;
     int sender ;
     int target ;
     CCPacket * pkt ;
{
    CACHE_MSG(target, pkt->reqtype+2, time-2);
    CACHE_MSG(target, 0, time);

    cache_controller(time, target, sender,
		     pkt->reqtype, pkt->address, pkt->line, 
		     &pkt->data,
		     (char *)&pkt->data+LINE_SIZE);
#ifdef CCDEBUG
    if (monitor_cache && GetSetFromAddr(pkt->address)==DEBUG_SET) {
	printf("Proc %d: done with %s, New cache state: %s\n",
	       target,
	       protocol_msg_[pkt->reqtype],
	       cache_state_[CacheState(target,
				       GetSetFromAddr(pkt->address),
				       pkt->line)]);
    }
#endif
}


/*************************************************************************\
* handler for non cache messages -- shouldn't be any
\*************************************************************************/

GLOBAL void Nocache_Pkt()
{
    fatal("Nocache_Pkt called with caches on!\n");
}


/*************************************************************************\
* Implement cache controller message dispatch
\*************************************************************************/

/* cproc is the processor where the cache controller resides 
 * (the recipient of messages handled by this function)
 * mproc is the processor where address resides
 */

static void cache_controller(Time time, int cproc, int mproc, int reqtype,
			     Word address, int line, char *data, char *febits) 
{
    int set = GetSetFromAddr(address);
    int tag = GetTagFromAddr(address);
    short *state = &CacheState(cproc, set, line);
    
#ifdef CCDEBUG
    if (monitor_cache && set==DEBUG_SET) {
	printf("Proc %d: cache handling %s in state %s\n",
	       cproc, protocol_msg_[reqtype], cache_state_[*state]);
	printf("\taddress=0x%x, set=%d, tag=%d, time %lu\n",
	       address, set, tag, time);
    }
#endif
    
    time += CACHE_ACCESS_LATENCY;
    
    if (reqtype == M_FACK) {
	fencectr[cproc]--;
	if (fencectr[cproc] == 0 )
	  if (waitsfence[cproc] >= 0)
	    MAKE_RESUME_REQUEST(waitsfence[cproc], time, OK);
	return;
    }
    
    if (tag == CacheTag(cproc, set, line) ) {
	/* Actions when the tag matches */
	switch( *state ) {
	  case C_INVALID:
	    switch( reqtype )
	      {
		case M_INV:
		  CTOM_ACKC(time, cproc, address, line);
		  return;
	      }
	    break;
	  case C_INVALID_NETW:
	    switch( reqtype )
	      {
		case M_INV:
		  CTOM_ACKC(time, cproc, address, line);
		  return;
		case M_WDATA:
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_READWRITE;
		  return;
		case M_RDATA:
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_READONLY;
		  return;
		case M_BUSY:
		  *state = C_INVALID;
		  return;
		case M_FDATA:
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_VALID;
		  return;
	      }
	    break;
	  case C_READONLY_NETW:
	    switch( reqtype )
	      {
		case M_INV:
		  CTOM_ACKC(time, cproc, address, line);
		  *state = C_INVALID_NETW;
		  return;
		case M_BUSY:
		  *state = C_READONLY;
		  return;
		case M_WDATA:
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_READWRITE;
		  return;
	      }
	    break;
	  case C_READONLY:
	    switch( reqtype )
	      {
		case M_INV:
		  CTOM_ACKC(time, cproc, address, line);
		  *state = C_INVALID;
		  return;
	      }
	    break;
	  case C_READWRITE:
	    switch( reqtype )
	      {
		case M_INV:
		  CTOM_UPDATE(time, cproc, address, line,
			      (char *) CacheData(cproc, set, line),
			      (char *) CacheFebits(cproc, set, line) );
		  *state = C_INVALID;
		  return;
	      }
	    break;
	  case C_READWRITE_NETW:
	    switch( reqtype )
	      {
		case M_INV:
		  CTOM_UPDATE(time, cproc, address, line,
			      (char *) CacheData(cproc, set, line),
			      (char *) CacheFebits(cproc, set, line) );
		  *state = C_INVALID_NETW;
		  return; 
	      }
	    break;
	  case C_VALID_NETW:
	    switch( reqtype )
	      {
		case M_INV:
		  CTOM_ACKC(time, cproc, address, line);
		  return;
		case M_RDATA:
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_READONLY;
		  return;
		case M_WDATA:
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_READWRITE;
		  return;
	      }
	    break;
	  case C_VALID:
	    switch( reqtype )
	      {
		case M_INV:
		  CTOM_ACKC(time, cproc, address, line);
		  return;
	      }
	    break;
	  case C_DIRTY:
	    switch( reqtype )
	      {
	      }
	    break;
	  case C_DIRTY_NETW:
	    switch( reqtype )
	      {
	      }
	    break;
	}
	fatal("cache_controller: state = %s, illegal message %s, address = %d",
	      cache_state_[*state], protocol_msg_[reqtype], address);
    } else {
	/* Actions when the tag does not match */
	switch( *state ) {
	  case C_INVALID:
	    switch( reqtype )
	      {
		case M_INV:
		  CTOM_ACKC(time, cproc, address, line);
		  return;
	      }
	    break;
	  case C_INVALID_NETW:
	    switch( reqtype )
	      {
		case M_INV:
		  CTOM_ACKC(time, cproc, address, line);
		  return;
		case M_WDATA:
		  CacheTag(cproc, set, line) = tag;
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_READWRITE;
		  return;
		case M_RDATA:
		  CacheTag(cproc, set, line) = tag;
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_READONLY;
		  return;
		case M_BUSY:
		  *state = C_INVALID;
		  return;
		case M_FDATA:
		  CacheTag(cproc, set, line) = tag;
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_VALID;
		  return;
	      }
	    break;
	  case C_READONLY_NETW:
	    switch( reqtype )
	      {
		case M_INV:
		  CTOM_ACKC(time, cproc, address, line);
		  return;
		case M_BUSY:
		  *state = C_READONLY;
		  return;
		case M_WDATA:
		  CTOM_REPU(time, cproc, CachedAddr(cproc, set, line), line);
		  CacheTag(cproc, set, line) = tag;
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_READWRITE;
		  return;
		case M_RDATA:
		  CTOM_REPU(time, cproc, CachedAddr(cproc, set, line), line);
		  CacheTag(cproc, set, line) = tag;
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_READONLY;
		  return;
		case M_FDATA:
		  CTOM_REPU(time, cproc, CachedAddr(cproc, set, line), line);
		  CacheTag(cproc, set, line) = tag;
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_VALID;
		  return;
	      }
	    break;
	  case C_READONLY:
	    switch( reqtype )
	      {
		case M_INV:
		  CTOM_ACKC(time, cproc, address, line);
		  return;
	      }
	    break;
	  case C_READWRITE:
	    switch( reqtype )
	      {
		case M_INV:
		  CTOM_ACKC(time, cproc, address, line);
		  return;
	      }
	    break;
	  case C_READWRITE_NETW:
	    switch( reqtype )
	      {
		case M_INV:
		  CTOM_ACKC(time, cproc, address, line);
		  return;
		case M_BUSY:
		  *state = C_READWRITE;
		  return;
		case M_RDATA:
		  CTOM_REPM(time, cproc, CachedAddr(cproc, set, line), line,
			    (char *) CacheData(cproc, set, line),
			    (char *) CacheFebits(cproc, set, line) );
		  CacheTag(cproc, set, line) = tag;
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_READONLY;
		  return;
		case M_WDATA:
		  CTOM_REPM(time, cproc, CachedAddr(cproc, set, line), line,
			    (char *) CacheData(cproc, set, line),
			    (char *) CacheFebits(cproc, set, line) );
		  CacheTag(cproc, set, line) = tag;
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_READWRITE;
		  return;
		case M_FDATA:
		  CTOM_REPM(time, cproc, CachedAddr(cproc, set, line), line,
			    (char *) CacheData(cproc, set, line),
			    (char *) CacheFebits(cproc, set, line) );
		  CacheTag(cproc, set, line) = tag;
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_VALID;
		  return;
	      }
	    break;
	  case C_VALID_NETW:
	    switch( reqtype )
	      {
		case M_INV:
		  CTOM_ACKC(time, cproc, address, line);
		  return;
		case M_RDATA:
		  CacheTag(cproc, set, line) = tag;
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_READONLY;
		  return;
		case M_WDATA:
		  CacheTag(cproc, set, line) = tag;
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_READWRITE;
		  return;
		case M_FDATA:
		case M_BUSY:
		  *state = C_VALID;
		  return;
	      }
	    break;
	  case C_VALID:
	    switch( reqtype )
	      {
		case M_INV:
		  CTOM_ACKC(time, cproc, address, line);
		  return;
	      }
	    break;
	  case C_DIRTY:
	    switch( reqtype )
	      {
		case M_INV:
		  CTOM_ACKC(time, cproc, address, line);
		  return;
	      }
	    break;
	  case C_DIRTY_NETW:
	    switch( reqtype )
	      {
		case M_INV:
		  CTOM_ACKC(time, cproc, address, line);
		  return;
		case M_BUSY:
		  *state = C_DIRTY;
		  return;
		case M_RDATA:
		  fencectr[cproc]++;
		  CTOM_FLUSH(time, cproc, CachedAddr(cproc, set, line), line,
			     (char *) CacheData(cproc, set, line),
			     (char *) CacheFebits(cproc, set, line) );
		  CacheTag(cproc, set, line) = tag;
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_READONLY;
		  return;
		case M_WDATA:
		  fencectr[cproc]++;
		  CTOM_FLUSH(time, cproc, CachedAddr(cproc, set, line), line,
			     (char *) CacheData(cproc, set, line),
			     (char *) CacheFebits(cproc, set, line) );
		  CacheTag(cproc, set, line) = tag;
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_READWRITE;
		  return;
		case M_FDATA:
		  fencectr[cproc]++;
		  CTOM_FLUSH(time, cproc, CachedAddr(cproc, set, line), line,
			     (char *) CacheData(cproc, set, line),
			     (char *) CacheFebits(cproc, set, line) );
		  CacheTag(cproc, set, line) = tag;
		  WriteCacheData(cproc, set, line, data);
		  WriteCacheFebits(cproc, set, line, febits);
		  *state = C_VALID;
		  return;
	      }
	    break;
	}
	fatal("cache_controller: state = %s, illegal message %s, address = %d",
	      cache_state_[*state], protocol_msg_[reqtype], address);
    }
}


/*************************************************************************\
* handler for cache to memory messages
\*************************************************************************/

GLOBAL void CtoM_Pkt(int sendtid, Time time,
		     int sender, int target, CCPacket *pkt) 
{
#ifdef CCDEBUG
    if (pkt->address==16) {
	CACHE_MSG(1, pkt->reqtype+2, time-2);
	CACHE_MSG(1, 0, time);
    }
#endif

    memory_controller(time, sender, target,
		      pkt->reqtype, pkt->address, pkt->line,
		      &pkt->data,
		      (char *)&pkt->data+LINE_SIZE);
    
#ifdef CCDEBUG
    if (monitor_cache && GetSetFromAddr(pkt->address)==DEBUG_SET) {
	printf("Proc %d: done with %s, New directory state: %s\n",
	       target, protocol_msg_[pkt->reqtype],
	       directory_state_[directory[GetBlockFromIndex(pkt->address)]
				.state]);
    }
#endif
}


/*************************************************************************\
* memory_controller: implements protocol for messages from cache to memory
\*************************************************************************/

static void memory_controller(Time time, int cproc, int mproc, int reqtype, 
			      Word address, int line,
			      char *data, char *febits) 
{
    int block = GetBlockFromIndex(address);
    DirBlk *d = &directory[block];
    short *state = &d->state;
    short i;
    
#ifdef CCDEBUG
    if (monitor_cache && GetSetFromAddr(address)==DEBUG_SET) {
	printf("Proc %d: memory handling %s from %d, address=0x%x, time %lu\n",
	       mproc, protocol_msg_[reqtype], cproc, address, time);
	printf("\tDirectory state %s, dircnt = %d, dir[0]=%d, ackcnt = %d\n",
	       directory_state_[*state], d->dircnt, d->dir[0], d->ackctr);
	PrintDirectory(d);
	if (mproc != AddressToProcessor(address))
	  printf("WARNING: mproc=%d, AddressToProcessor(0x%x)=%d\n",
		 mproc, address, AddressToProcessor(address));
    }
#endif
    
    time += MEM_ACCESS_LATENCY;  /* mem_free_again??? */
    
    switch( *state ) {
      case D_ABSENT:
	switch( reqtype ) {
	  case M_REPU:
	    return;
	  case M_ACKC:
	    --d->ackctr;
	    return;
	  case M_FETCH:
	    MTOC_FDATA(time, cproc, address, line, 
		       &memory[address], &memtag[address]);
	    return;
	  case M_FLUSH:
	    WriteCacheLine(&memory[address], data);
	    WriteCacheLine(&memtag[address], febits);
	    MTOC_FACK(time, cproc, address, line);
	    return;
	  case M_RREQ:
	    d->dir[0] = cproc; d->line[0] = line;
	    d->dircnt = 1;
	    MTOC_RDATA(time, cproc, address, line, 
		       &memory[address], &memtag[address]);
	    *state = D_READONLY;
	    MEM_EVENT(time, *state);
	    return;
	  case M_WREQ:
	    if (d->ackctr == 0 ) {
		d->dir[0] = cproc; d->line[0] = line;
		d->dircnt = 1;
		MTOC_WDATA(time, cproc, address, line, 
			   &memory[address], &memtag[address]);
		*state = D_READWRITE;
		MEM_EVENT(time, *state);
	    } else {
		d->dir[0] = cproc; d->line[0] = line;
		d->dircnt = 1;
		*state = D_WRITETRANS;
		MEM_EVENT(time, *state);
	    }
	    return;
	}
	break;
      case D_READONLY:
	switch( reqtype ) {
	  case M_ACKC:
	    --d->ackctr;
	    return;
	  case M_FETCH:
	    MTOC_FDATA(time, cproc, address, line, 
		       &memory[address], &memtag[address]);
	    return;
	  case M_RREQ:
	    for(i=0; i<d->dircnt; i++)
	      if (d->dir[i] == cproc ) {
		  /* case 1: cproc in set P => return data */
		  MTOC_RDATA(time, cproc, address, line, 
			     &memory[address], &memtag[address]);
		  return;
	      }
	    if (d->dircnt < DIRECTORY_SIZE ) {
		/* case 2, cproc not in P, n < p => add cproc */
		d->dir[d->dircnt] = cproc; d->line[d->dircnt] = line;
		d->dircnt++;
		MTOC_RDATA(time, cproc, address, line, 
			   &memory[address], &memtag[address]);
	    } else {
		/* case 3, cproc not in P, n == p => replace randomly */
		int victim = random() % DIRECTORY_SIZE;
		MTOC_INV(time, d->dir[victim], address, d->line[victim]);
		d->ackctr++;
		d->dir[victim] = cproc; d->line[victim] = line;
		MTOC_RDATA(time, cproc, address, line, 
			   &memory[address], &memtag[address]);
	    }
	    return;
	  case M_WREQ:
	    INVARIANT2(d->dircnt != 0, "READONLY:WREQ dircnt==0\n");
	    if (d->dircnt == 1 && d->dir[0] == cproc ) {
		if (d->ackctr == 0 ) {
		    /* case 1: P = {cproc}, ackctr==0 */
		    MTOC_WDATA(time, cproc, address, line, 
			       &memory[address], &memtag[address]);
		    *state = D_READWRITE;
		    MEM_EVENT(time, *state);
		} else {
		    /* case 2: P = {cproc}, ackctr != 0 */
		    *state = D_WRITETRANS;
		    MEM_EVENT(time, *state);
		}
	    } else {
		/* cases 3 & 4: invalidate all but cproc */
		for(i=0; i<d->dircnt; i++)
		  if (d->dir[i] != cproc) {
		      /* case 3: P = {..., cproc, ...} */
		      MTOC_INV(time, d->dir[i], address, d->line[i]);
		      d->ackctr++;
		  }
		d->dircnt = 1;
		d->dir[0] = cproc; d->line[0] = line;
		*state = D_WRITETRANS;
		MEM_EVENT(time, *state);
	    }
	    return;
	  case M_REPU:
	    for(i=0; i<d->dircnt; i++)
	      if (d->dir[i] == cproc) {
		  if (i < d->dircnt-1) {
		      /* if not last, copy last to here */
		      d->dir[i] = d->dir[d->dircnt-1];
		      d->line[i] = d->line[d->dircnt-1];
		  }
		  d->dircnt--;
		  if (d->dircnt == 0) {
		      *state = D_ABSENT;
		      MEM_EVENT(time, *state);
		  }
		  return;
	      }
	    return;
	}
	break;
      case D_READWRITE:
	INVARIANT3(d->dircnt == 1, "READWRITE: dircnt=%d\n", d->dircnt);
	switch( reqtype ) {
	  case M_ACKC:
	    --d->ackctr;
	    return;
	  case M_REPM:
	    INVARIANT4(d->dir[0] == cproc,
		       "READWRITE: REPM, (dir[0]=%d) != (cproc=%d)\n",
		       d->dir[0], cproc);
	    WriteCacheLine(&memory[address], data);
	    WriteCacheLine(&memtag[address], febits);
	    d->dircnt = 0;
	    *state = D_ABSENT;
	    MEM_EVENT(time, *state);
	    return;
	  case M_RREQ:
	  case M_FETCH:
	    MTOC_INV(time, d->dir[0], address, d->line[0]);
	    d->ackctr++;
	    d->dir[0] = cproc; d->line[0] = line;
	    *state = D_READTRANS;
	    MEM_EVENT(time, *state);
	    return;
	  case M_WREQ:
	    MTOC_INV(time, d->dir[0], address, d->line[0]);
	    d->ackctr++;
	    d->dir[0] = cproc; d->line[0] = line;
	    *state = D_WRITETRANS;
	    MEM_EVENT(time, *state);
	    return;
	}
	break;
      case D_READTRANS:
	INVARIANT3(d->dircnt == 1, "READTRANS: dircnt=%d\n", d->dircnt);
	switch( reqtype )
	  {
	    case M_ACKC:
	      --d->ackctr;
	      return;
	    case M_RREQ:
	    case M_WREQ:
	      MTOC_BUSY(time, cproc, address, line);
	      return;
	    case M_UPDATE:
	      INVARIANT3(d->dir[0] != cproc,
			 "READTRANS, UPDATE: dir[0] == cproc (=%d)\n",
			 cproc);
	      WriteCacheLine(&memory[address], data);
	      WriteCacheLine(&memtag[address], febits);
	      --d->ackctr;
	      MTOC_RDATA(time, d->dir[0], address, d->line[0], 
			 &memory[address], &memtag[address]);
	      *state = D_READONLY;
	      MEM_EVENT(time, *state);
	      return;
	    case M_REPM:
	      INVARIANT3(d->dir[0] != cproc,
			 "READTRANS, REPM: dir[0] == cproc (=%d)\n",
			 cproc);
	      WriteCacheLine(&memory[address], data);
	      WriteCacheLine(&memtag[address], febits);
	      MTOC_RDATA(time, d->dir[0], address, d->line[0],
			 &memory[address], &memtag[address]);
	      *state = D_READONLY;
	      MEM_EVENT(time, *state);
	      return;
	  }
	break;
      case D_WRITETRANS:
	INVARIANT3(d->dircnt == 1, "WRITETRANS: dircnt=%d\n", d->dircnt);
	switch( reqtype ) {
	  case M_RREQ:
	  case M_WREQ:
	    MTOC_BUSY(time, cproc, address, line);
	    return;
	  case M_REPU:
	    INVARIANT3(d->dir[0] != cproc,
		       "WRITETRANS, REPU: dir[0] == cproc (=%d)\n",
		       cproc);
	    return;
	  case M_UPDATE:
	    INVARIANT3(d->dir[0] != cproc,
		       "WRITETRANS, UPDATE: dir[0] == cproc (=%d)\n",
		       cproc);
	    WriteCacheLine(&memory[address], data);
	    WriteCacheLine(&memtag[address], febits);
	    --d->ackctr;
	    if (d->ackctr == 0) {
		MTOC_WDATA(time, d->dir[0], address, d->line[0],
			   &memory[address], &memtag[address]);
		*state = D_READWRITE;
		MEM_EVENT(time, *state);
	    }
	    return;
	  case M_ACKC:
	    --d->ackctr;
	    if (d->ackctr == 0) {
		MTOC_WDATA(time, d->dir[0], address, d->line[0],
			   &memory[address], &memtag[address]);
		*state = D_READWRITE;
		MEM_EVENT(time, *state);
	    }
	    return;
	  case M_REPM:
	    INVARIANT3(d->dir[0] != cproc,
		       "WRITETRANS, REPM: dir[0] == cproc (=%d)\n",
		       cproc);
	    WriteCacheLine(&memory[address], data);
	    WriteCacheLine(&memtag[address], febits);
	    if (d->ackctr == 1 ) {
		MTOC_WDATA(time, d->dir[0], address, d->line[0],
			   &memory[address], &memtag[address]);
		*state = D_READWRITE;
		MEM_EVENT(time, *state);
	    }
	    return;
	}
	break;		
    }
    fatal("memory_controller: state %s, Illegal message %s, address = 0x%x",
	  directory_state_[*state], protocol_msg_[reqtype], address);
}
