 
 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: debug.c
 *
 * Description: Routines to aid debugging
 *
 * Last Modified:  12-Oct-92 (adj)
 *
 * Global Functions:
 *     void AddOverwriteCheck(long *address, char *name)
 *     void ListOverwriteChecks()
 *     void OverwriteCheck()
 *     void bus_error_(int sig, int code, struct sigcontext *scp)
 *     void debug_printf_(format_string, args...)
 *     void bomb_()
 *     void catch_()
 *
 * Global Variables:
 *     BOOL overwrite_check
 *
 ****************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.
 ****************************************************************************
 * $Header: /usr/wildcat/dfk/research/parallel/proteus/proteus-V3.01/engine/RCS/debug.c,v 1.2 94/01/24 00:39:08 dfk Time64bit Locker: dfk $
 * $Log:	debug.c,v $
 * Revision 1.2  94/01/24  00:39:08  dfk
 * Added Sanjay's support for 64-bit timers.
 * Added thread_sleep_until.
 * Fixed a few bugs in thread sleep/wakeup code (my bugs).
 * Fixed printf formats in many places.
 * 
 * Revision 1.1  94/01/23  11:34:15  dfk
 * Initial revision
 * 
 * Revision 1.4  92/12/09  15:15:35  brewer
 * Updated calls to trace_stack_
 * 
 * Revision 1.3  92/10/28  11:18:28  brewer
 * adj improved signal handling
 * 
 * Revision 1.2  92/09/23  11:55:37  brewer
 * Updated args to trace_stack_
 * 
 * Revision 1.1  92/02/11  13:55:18  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include <signal.h>
#include <syms.h>
#include <mips/cpu.h>
#include <malloc.h>
#undef NET
#include "sim.h"
#include "conf.param"
#include "rt_thread_def.h"
#include "simreq.h"
#include "processor.h"
#include "thread.h"


typedef struct aT {
    const char *name;
    long *address;
    long value;
    struct aT *next;
} addressType;

/* `overwrite_check' flag TRUE iff at least one overwrite check is being
    performed */
GLOBAL BOOL overwrite_check = FALSE;

static addressType *checks = NULL;


GLOBAL void AddOverwriteCheck(long *address, const char *name)
{
    addressType *adr;
    extern long _fdata; /* &_fdata = start of data area; see "man ld" */

    if (address < &_fdata || address >= (long *)0x80000000) {
	printf("Address out of range; not installing check.\n");
	return;
    }

    adr = (addressType *) malloc(sizeof(addressType));
    adr->address = address;
    adr->name = name;
    adr->value = *address;
    adr->next = checks;
    checks = adr;

    overwrite_check = TRUE;
}


GLOBAL void ListOverwriteChecks()
{
    addressType *adr;

    if (checks == NULL) {
	printf("There are currently no overwrite checks.\n");
    } else {
	printf("Overwrite Checks:\n");
    
	for (adr = checks; adr != NULL; adr = adr->next)
	  printf("\t%10lu = 0x%-10lx   %s\n", (ulong) adr->address,
		 (ulong) adr->address, adr->name);
	putchar('\n');
    }
}


GLOBAL void OverwriteCheck()
{
    addressType *adr;

    if (checks == NULL) return;

    adr = checks;
    while (adr != NULL) {
	if (*(adr->address) != (adr->value)) {
	    fprintf(stderr, "Overwrite warning: `%s' (%lu = 0x%lx)\n",
		    adr->name, (ulong) adr->address,
		    (ulong) adr->address);
	    fprintf(stderr,
		    "\told value = %ld = 0x%lx, new value = %ld = 0x%lx\n",
		    adr->value, (ulong)adr->value,
		    *(adr->address), (ulong) *(adr->address));
	    fprintf(stderr, "\tThread %d (%s) resumed at %s\n",
		    currtid, thread_table_[currtid].t_name,
		    time_print(thread_table_[currtid].t_time));
	    fprintf(stderr, "\tProcessor %d, (Current Time = %s)\n",
		    processor_, time_print(CURRENTTIME));
	    trace_stack_(stderr,
			 (ulong) thread_table_[currtid].t_regs[31],
			 (ulong *) thread_table_[currtid].t_regs, currtid);
	    adr->value = *(adr->address);
	}
	adr = adr->next;
    }
}


/***************************************************************************/

GLOBAL void bus_error_(sig, code, scp)
int sig, code;
struct sigcontext *scp;
{
    unsigned long exception_instruction;

    fprintf(stderr, "\n\n");
    
    switch (sig) {
      case SIGSEGV:
	fprintf(stderr, "Segmentation Violation\n");
	break;
      case SIGBUS:
	switch (code) {
	  case EXC_DBE:
	    fprintf(stderr, "Data Bus Error\n");
	    break;
	  case EXC_IBE:
	    fprintf(stderr, "Instruction Bus Error\n");
	    break;
	  case EXC_RADE:
	    fprintf(stderr, "Read Address Error\n");
	    break;
	  case EXC_WADE:
	    fprintf(stderr, "Write Address Error\n");
	    break;
	  default:
	    fprintf(stderr, "Bus Error\n");
	}
	break;
      case SIGILL:
	fprintf(stderr, "Illegal Instruction\n");
	break;
      default:
	fprintf(stderr, "Unknown Bus Error\n");
    }
    
    if (scp->sc_cause & CAUSE_BD)
      exception_instruction = (ulong)(scp->sc_pc + 4);
    else
      exception_instruction = (ulong)(scp->sc_pc);

    putchar('\n'); /* flush out stdout buffer */
    fprintf(stderr, "In thread %d at time %s (mode = %s)\n",
	    currtid, time_print(CURRENTTIME), mode_?"ENGINE":"USER");
    fprintf(stderr, "Processor %d\n", processor_);
    fprintf(stderr, "(quantum_ = %s, cycles_ = %lu)\n", 
	    time_print(quantum_), cycles_);

    trace_stack_(stderr,
		 exception_instruction, (ulong *) scp->sc_regs, currtid);
    Disassemble(exception_instruction - 40, exception_instruction);

    snapshot();
    exit(2);
}

/***************************************************************************/

/* Generic bomb routine */
GLOBAL void bomb_()
{
    fprintf(stderr, "Bomb!  --  entered routine bomb()\n");
    exit(2);
}

/* generic debugging routine */
/* We use this to set breakpoints at compile time.  Simply add a call to 
   'catch_' at the desired breakpoint, then use "stop in catch_" in dbx */
GLOBAL void catch_()
{
    fprintf(stderr, "Reached \"catch_\".\n");
}
