
/**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: ihandler.c
 *
 * Description: Defines interrupt routines
 *
 * Last Modified:  $Date: 92/12/09 15:21:27 $ (eab)
 *
 * Global Functions:
 *     SimRequest *ResumeInterruptedThread(SimRequest *cont)
 *     void ihandler(int proc)
 *     void make_interrupt_request(target, sender, sendtid, time,
 *                                 priority, type, argc, argv)
 *     void insert_new_interrupt(target, sender, sendtid, iprio,
 *                               itype, itime, argc, argv)
 *     BOOL PendingInterrupts(void) 
 *     void CheckForInterrupts(void) 
 *     void schedule_intr_handler_(SimRequest *rptr)
 *     void ServiceInterrupt(int processor) 
 *
 * Global Variables: none
 *    
 * Referenced parameters:
 *     INTR_LATENCY
 *
 ***************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.		
 ****************************************************************************
 * $Header: /psg/proteus/RCS/ihandler.c,v 1.5 92/12/09 15:21:27 brewer Exp $
 * $Log:	ihandler.c,v $
 * Revision 1.5  92/12/09  15:21:27  brewer
 * Added call to PassControl_no_interrupts to PendingInterrupts
 * 
 * Revision 1.4  92/11/12  12:56:30  brewer
 * Changed T_INTERRUPTED from state to flag -- see rt_thread.ca log for
 * details.
 * 
 * Revision 1.3  92/10/28  13:35:25  brewer
 * commented out early return in CheckForInterrupts to ensure that
 * even the ihandler checks for interrupts.
 * 
 * Revision 1.2  92/07/06  11:22:21  brewer
 * fixed last arg to insert_new_interrupt in schedule_intr_handler_
 * changed from "&rptr->iargv" to "(Word *)rptr->iargv"
 * 
 * Revision 1.1  92/02/11  13:55:52  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include "sim.h"
#include "conf.h"
#include "simreq.h"
#include "rt_thread_def.h"
#include "processor.h"
#include "thread.h"
#include "mem.param"
#include "sema.h"
#include "intreq.h"
#include "simcalls.h"
#include "ipi.param"
#include "event.h"

/**************************************************************************/

GLOBAL void ServiceInterrupt(int processor) 
{
    IntrRequest *iptr = next_intr_request(processor);
    
    /* assumes interrupts are turned off for this thread */
    if (currosptr->t_intrflag)
      fatal("Service Interrupt called with interrupts ON");

#ifdef FNPTRDEBUG
    printf("FNPTR Will start intr_handler_ at %08ld\n", intr_handler_[iptr->itype]);
#endif
    
    (void)(*intr_handler_[iptr->itype])(iptr->argc, iptr->argv);
    
    free_intr_request(iptr);
    
    cycles_ -= INTR_LATENCY;
}


/**************************************************************************/

/* User threads periodically check for pending interrupts. If a pending
 * interrupt is detected from within a user thread, its handler is
 * executed in the context of that thread. 
 * There are two cases where this strategy does not work:
 * (1) an interrupt is received by a processor that is currently idle
 * (2) an interrupt is received by a processor that is currently executing
 *     a local intruction block, and is therefore not checking for interrupts
 *
 * To handle interrupts in those cases, each processor has a "fake"
 * interrupt handler (ihandler) thread. Ihandler threads are created
 * during system initialization and execute function ihandler.
 * Normally, they sleep inside ResumeInterruptedThread.
 * When an interrupt arrives on an idle processor, the corresponding
 * ihandler is resumed. Every resume request has an optional parameter
 * which is passed to the resumed thread via global variable result_
 * When ihandler is resumed for an idle processor, result_ is always
 * NULL.
 * When an interrupt arrives during execution of a long local instruction
 * block, it will be detected the next time the simulator quantum expires
 * for that thread. In that case (see resume.c) a resume request is
 * built for the interrupted thread, the thread's state is changed to
 * INTERRUPTED, and the corresponding ihandler thread is resumed.
 * The parameter passed to the resumed thread is now equal to the
 * address of the resume request for the interrupted thread.
 * After handling the interrupts, ihandler enqueues this request
 * and traps to the engine.
 */

GLOBAL SimRequest *ResumeInterruptedThread(SimRequest *cont) 
{
    /*
      printf("` Resuming intr tid #%d on processor %d\n",
      cont == NULL ? -1 : cont->tid, processor_);
      */

    currosptr->t_state = T_SUSPENDED;

    /* cont is NULL whenever the processor is idle when ihandler is called */
    if (cont == (SimRequest *)NULL) { 

	_thread_yieldToScheduler();   /* ihandler sleeps here */    
        
    } else { 
	/* cont is not NULL when there is a pending interrupt
         * when the simulator quantum expires
	 */
	Thread *introsptr = T_OSBLOCK(cont->h.tid);

	cont->h.timestamp = CURRENTTIME;

	/* if the interrupted thread is still current, resume it */
	if (introsptr != NULL && (introsptr->t_state & T_INTERRUPTED)) {
	    introsptr->t_state &= ~T_INTERRUPTED; /* remove intr. flag */
	    enqueue_request_(cont);
	    TrapToSimulator();  /* ihandler sleeps here */
	} else {
	    free_request_(cont);
	    _thread_yieldToScheduler(); /* ihandler sleeps here */
	}
    }
    /* result_ contains the parameter passed to the resume request that
     * woke up ihandler. It is copied to cont in ihandler
     */
    return((SimRequest *)result_);
}


/**************************************************************************/

GLOBAL void ihandler(int proc) 
{
    SimRequest *cont = (SimRequest *)result_;
    ProcBlk *pptr = &proc_table_[proc];
    
    for (;;) {
	for (;;) {
	    if( (cont == NULL || T_OSBLOCK(cont->h.tid)->t_preempt) &&
	       ReadTimer(processor_) <= 0 ) {
		(*proc_table_[processor_].p_timer_ihandler)
		  (processor_,currtid,CURRENTTIME);
		/* Very convoluted code !
		 * Timer_handler usually calls thread_yieldToScheduler
		 * which places a resume request for the next current thread.
		 * The request stored in cont is now superfluous and must be
		 * freed. Whenever ihandler is called again, it will resume
		 * execution from this point in code. result_ will contain the
		 * new value of cont.
		 */
		if (cont != (SimRequest *)result_) {
		    if (cont != NULL)
		      free_request_(cont);
		    cont = (SimRequest *)result_;
		    /* goto top; */
		} else break;
	    } else break;
	}
	
	pptr->p_ihandleractive = TRUE;
	
	while(intr_pending[proc] > 0)
	  ServiceInterrupt(proc);
	
	pptr->p_ihandleractive = FALSE;
	/* Ihandler will now sleep, waiting for another interrupt 
	   When it is woken up again, cont will either be NULL, or equal
	   to the address of the resume request for the interrupted thread */
	cont = ResumeInterruptedThread(cont);
    }
}


/**************************************************************************/

GLOBAL void make_interrupt_request(target, sender, sendtid, time,
				   priority, type, argc, argv) 
int target;
int sender;
int sendtid;
Time time;
int priority;
int type;
int argc;
Word *argv;
{
    IntrRequest *iptr;
    int i;
    
    iptr = new_intr_request();
    iptr->itime = time;
    iptr->itype = type;
    iptr->isender = sender;
    iptr->isendtid = sendtid;
    iptr->iprio = priority;
    iptr->argc = argc;
    
    for(i=0; i<argc; i++)
      iptr->argv[i] = argv[i];
    
    enqueue_intr_request(iptr, target);
}


/**************************************************************************/

GLOBAL void insert_new_interrupt(target, sender, sendtid, iprio,
				 itype, itime, argc, argv) 
int target;
int sender;
ulong sendtid;
int iprio;
ulong itype;
Time itime;
int argc;
Word *argv;
{
    ProcBlk *pptr = &proc_table_[target];
    
    make_interrupt_request(target, sender, sendtid, itime, iprio,
			   itype, argc, argv);
    
    /* printf("Target %d is executing %d\n", target, pptr->p_tid); */
    
    if (Idle(target)) {

	if (itime < pptr->p_time) {
	    /* this case occurs if the interrupt arrived while
	       the last thread on the processor was being killed */
	    itime = pptr->p_time;
	}

	/* printf("Will resume ihandler@%d\n", target); */
	pptr->p_tid = pptr->p_ihandler;
	pptr->p_interrupted = ERROR;
	
	T_OSBLOCK(pptr->p_ihandler)->t_state = T_CURRENT;
	
	idle_processors--;
	CancelPendingTimerInterrupt(pptr);
	
	EVENT_BUSY(itime, target);
	
	MAKE_RESUME_REQUEST(pptr->p_ihandler, itime, NULL);
    }
}


/**************************************************************************/

GLOBAL BOOL PendingInterrupts()
{
    extern void PassControl_no_interrupts(void);

    PassControl_no_interrupts();
    return(intr_pending[processor_] > 0);
}

GLOBAL void CheckForInterrupts() 
{
    BOOL prevp;
    
/*    if (currtid == currpptr->p_ihandler)
      return; */

    if (currosptr->t_preempt && ReadTimer(processor_) <= 0) {
	/*  printf("Timer interrupt on processor %d\n", processor_); */
	(*proc_table_[processor_].p_timer_ihandler)
	  (processor_,currtid,CURRENTTIME);
    }
    
    /* Check if thread can accept interrupts */
    if (!(currosptr->t_intrflag))
      return;
    
    currpptr->p_ihandleractive = TRUE;
    currpptr->p_interrupted = currtid;
    
    prevp = currosptr->t_preempt;
    currosptr->t_preempt = PREEMPT_OFF;
    currosptr->t_intrflag = INTR_OFF;
    
    while (intr_pending[processor_] > 0
	   /* && first_intr_request(processor_)->itime <= CURRENTTIME */) {
	ServiceInterrupt(processor_);
    }
    
    currosptr->t_preempt = prevp;
    currosptr->t_intrflag = INTR_ON;
    
    currpptr->p_ihandleractive = FALSE;
}


/**************************************************************************/

typedef struct {
    SimHeader  h;
    Word       processor;  /* argv[0] */
    Word       itype;      /* argv[1] */
    Word       iprio;      /* argv[2] */
    Word       iargc;
    Word       iargv;
} SchedIntrRequest;


GLOBAL void schedule_intr_handler_(rptr)
SchedIntrRequest *rptr;
{
    insert_new_interrupt(rptr->processor, rptr->processor, (ulong)-1,
			 rptr->iprio, rptr->itype, rptr->h.timestamp,
			 rptr->iargc, (Word *)rptr->iargv);
}



