
/****************************************************************************
 *                                                                          *
 *                     Parallel Architecture Simulator                      *
 *                Eric A. Brewer  and  Chris N. Dellarocas                  *
 *                     Laboratory for Computer Science                      *
 *                  Massachusetts Institute of Technology                   *
 *                                                                          *
 * Module:  Interrupt Request Priority Queue                                *
 *                                                                          *
 * Description:  The module uses a singly-linked list to implement a        *
 *    priority queue.  Five operations are provided:  1) Initialize the     *
 *    queue,  2) Extract the minimum element (i.e. the lowest-numbered      *
 *    iprio),  3) Enqueue a request,  4) Allocate a new request block,      *
 *    and 5) Deallocate a request block.                                    *
 *                                                                          *
 * Last Modified:  6-13-90 (cnd)                                            *
 *                                                                          *
 * Global Functions:                                                        *
 *        void init_intr_request_queue()                                    *
 *        IntrRequest *next_intr_request(int processor)                     *
 *        void enqueue_intr_request(IntrRequest *request, int processor)    *
 *        IntrRequest *new_intr_request()                                   *
 *        void free_intr_request(IntrRequest *request)                      *
 *                                                                          *
 * Global Variables:                                                        *
 *        none                                                              *
 *                                                                          *
 ****************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.
 ****************************************************************************
 * $Header: /psg/proteus/RCS/intr_queue.c,v 1.1 92/02/11 13:55:54 brewer Exp $
 * $Log:	intr_queue.c,v $
 * Revision 1.1  92/02/11  13:55:54  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include "net.param"
#include <stdio.h>
#include "sim.h"
#include "conf.h"
#include "rt_thread_def.h"
#include "intreq.h"
#include "thread.h"

/***************************************************************************/

IntrRequest *InterruptQueue[NO_OF_PROCESSORS];


IntrRequest * first_intr_request ( processor) 
int processor ;
{
  return InterruptQueue[processor];
}


IntrRequest * next_intr_request ( processor) 
int processor ;
{
    IntrRequest *ptr;

    if (InterruptQueue[processor] == NULL)
      return (NULL);
    
    ptr = InterruptQueue[processor];
    InterruptQueue[processor] = ptr->inextreq;

    intr_pending[processor]--;

    return(ptr);
}


void enqueue_intr_request ( request, processor) 
IntrRequest * request ;
int processor ;
{
    IntrRequest **handle;


    for(handle = &InterruptQueue[processor]; *handle != NULL;
	handle = &((*handle)->inextreq)) {
      if (request->itime < (*handle)->itime ||
	  ((request->itime == (*handle)->itime) &&
	  request->iprio > (*handle)->iprio)) {
	request->inextreq = (*handle);
	*handle = request;
	intr_pending[processor]++;
	return;
      }
    }
    (*handle) = request;
    request->inextreq = NULL;

    intr_pending[processor]++;
}


/***************************************************************************/
/* allocation/deallocation of request blocks */

static IntrRequest intr_request_blocks[MAX_INTR_REQS];

static IntrRequest *intr_free_list = NULL;


void init_intr_request_queue ( ) 
{
    int i;

    intr_request_blocks[0].inextreq = NULL;

    for (i=1; i<MAX_INTR_REQS; i++) 
	intr_request_blocks[i].inextreq = &intr_request_blocks[i-1];


    for(i=0; i<NO_OF_PROCESSORS; i++) {
	InterruptQueue[i] = NULL;
	intr_pending[i] = 0;
    }

    i = MAX_INTR_REQS - 1;
    intr_free_list = &intr_request_blocks[ i ];

}

int numintrs=0;

IntrRequest * new_intr_request ( ) 
{
    IntrRequest *new;

    if (intr_free_list == NULL) {  /* fatal error, exit */
	fatal("out of interrupt request blocks");
    }
    new = intr_free_list;
    intr_free_list = intr_free_list->inextreq;

#ifdef INTRDEBUG
    printf("Allocated intrreqs : %d\n", ++numintrs);
#endif

    return(new);
}


void free_intr_request ( request) 
IntrRequest * request ;
{
    request->inextreq = intr_free_list;
    intr_free_list = request;

#ifdef INTRDEBUG
    printf("Allocated intrreqs : %d\n", --numintrs);
#endif


}


BOOL TrapPending ( p) 
int p;
{
  return( intr_pending[p] > 0 ? 
	  InterruptQueue[p]->iprio == IPRIO_TRAP :
	  FALSE );
}
