
 /****************************************************************************
 *
 *                     Parallel Architecture Simulator      
 *                Eric A. Brewer  and  Chris N. Dellarocas  
 *                     Laboratory for Computer Science      
 *                  Massachusetts Institute of Technology   
 *
 * Module: ipi.c
 *
 * Description:                                             
 *         Code to implement interprocessor interrupts
 *
 * Last Modified: $Date: 94/07/02 14:58:50 $
 *
 * Global Functions:
 *     void define_ipi(int type, FuncPtr handler, const char *name)
 *     int define_new_ipi(FuncPtr handler, const char *name)
 *     void send_ipi(processor, priority, type, length, argc, args) 
 *     void send_ipiV(processor, priority, type, length, argc, argv)
 *     void ipi_bus_handler_(rptr)
 *
 * Global Variables: none
 *
 ****************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.
 ****************************************************************************
 * $Header: /usr/wildcat/dfk/research/parallel/proteus/proteus-V3.01/engine/RCS/ipi.c,v 1.2 94/07/02 14:58:50 dfk Exp $
 * $Log:	ipi.c,v $
 * Revision 1.2  94/07/02  14:58:50  dfk
 * changed INTR_LATENCY to IPI_LATENCY, as it ought to be
 * 
 * Revision 1.1  94/07/02  14:58:20  dfk
 * Initial revision
 * 
 * Revision 1.4  92/11/19  14:18:34  brewer
 * Changed interface for build_new_packet_ to use stdarg convention.
 * Added support for explicit send-receive communication model.
 * 
 * Revision 1.3  92/05/08  12:47:25  brewer
 * Added procedure define_new_ipi
 * Fixed redefinition check in define_ipi; it no longer gives a warning
 * when you redefined a handler that was set to null_handler_
 * 
 * Revision 1.2  92/05/07  12:35:13  brewer
 * Added warning message to indicate when define_ipi redefines a type.
 * 
 * Revision 1.1  92/02/11  13:55:56  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include "sim.h"
#include "conf.h"
#include "rt_thread_def.h"
#include "simreq.h"
#include "processor.h"
#include "simcalls.h"
#include "cache.h"
#include "shmem.h"
#include "sema.h"
#include "thread.h"
#include "intreq.h"
#include "net.h"
#include "event.h"
#include "packet.h"
#include "ipi.param"


#define InvalidIpi(t)         (((t)<0)||(intr_handler_[(t)]==(FuncPtr)NULL))

#define MIN_USER_IPI          100

/* Bitmasks for separating interrupt type from interupt mode info */
#define MODEMASK              0xffff0000 /* Top 16 bits encode mode */
#define TYPEMASK              0x0000ffff /* Low 16 bits encode type */


/*************************************************************************\
* define_ipi: register a handler for a specific ipi
\*************************************************************************/

#ifdef SEND_RECEIVE

GLOBAL FuncPtr define_ipi(int type, FuncPtr handler, const char *name)
{
    fatal("define_ipi called while communication style is send-receive");
    return(handler); /* unreachable, but makes compiler happy */
}

#else

static void name_error()
{
    fatal("define_ipi called with invalid `name' argument");
}

GLOBAL FuncPtr define_ipi(int type, FuncPtr handler, const char *name) 
{
    FuncPtr previous;
    void (*f1)(), (*f2)();
    
    if (type < 0 || type >= MAX_INTR_TYPES)
      fatal("define_ipi failed: Attempt to redefine illegal interrupt type %d",
	    type);
    
    if (name == NULL) {
	fprintf(stderr,
		"Warning: define_ipi called with 3rd arg `name' == NULL\n");
	name = "User-defined ipi";
    }

    /* check validity of name argument
     *   this was added because the addition of the name argument caused
     *   existing software to fail with a segmentation violation
     */
    f1 = signal(SIGSEGV, name_error);
    f2 = signal(SIGBUS, name_error);
    strlen(name); /* if name invalid, this should cause a seg violation */
    signal(SIGSEGV, f1);
    signal(SIGBUS, f2);
    /* assume name is valid */

    previous = intr_handler_[type];
    
    if (handler != NULL) {
	fprintf(stderr, "Defining ipi %d (%s)\n", type, name);
	intr_handler_[type] = handler;
	if (type > MAX_DEFINED_IPI) intr_name_[type] = name;
    }

    if (previous != NULL  &&  previous != (FuncPtr)(null_handler_)) {
	fprintf(stderr, "WARNING: redefined ipi %d with %s\n",
		type, name);
    }
    
    return(previous);
}

#endif /* ifdef SEND_RECEIVE else */


/*************************************************************************\
* define_new_ipi: register a handler, assign an ipi number
\*************************************************************************/

#ifdef SEND_RECEIVE

GLOBAL int define_new_ipi(FuncPtr handler, const char *name)
{
    fatal("define_new_ipi called while communication style is send-receive");
    return(0); /* unreachable, but keeps compiler happy */
}

#else

GLOBAL int define_new_ipi(FuncPtr handler, const char *name)
{
    int i;

    for (i=MIN_USER_IPI; i<MAX_INTR_TYPES; i++) {
	if (intr_handler_[i] == NULL) break;
    }
    if (i==MAX_INTR_TYPES)
      fatal("Unable to define new ipi handler, out of ipi types.");

    define_ipi(i, handler, name);
    return(i);
} 

#endif /* ifdef SEND_RECEIVE else */


/*************************************************************************\
* send_ipi: generate an active message (also used for send-receive)
\*************************************************************************/

GLOBAL void send_ipi(processor, priority, type, length, argc, args) 
int processor;
int priority;
int type;
int length;
int argc;
Word args;
{
    int i, mode;
    
    mode = type & MODEMASK; 
    type = type & TYPEMASK; 

#ifndef SEND_RECEIVE
    if (InvalidIpi(type))
      fatal("IPI: Invalid ipi. type = %d", type);
#endif    

#ifdef BUS
    if ( InvalidProc(processor) || argc+5 > REQ_MAX_ARGS )
      fatal("send_ipi failed: Invalid processor %d or too many arguments %d",
	    processor, argc);
#else /* NET */
    if ( InvalidProc(processor)  || argc+3 > MAX_PACKET_WORDS )
      fatal("send_ipi failed: Invalid processor %d or too many argunments %d",
	    processor, argc);
#endif
    
    
    if ( processor == processor_ ) {
	/* Local interrupts (traps) are delivered immediately */
	insert_new_interrupt(processor, processor, currtid, priority,
			     type, CURRENTTIME, argc, &args);
	CheckForInterrupts();
    } else {
#ifdef BUS
	SimRequest *rptr, *build_new_request_();
	
	rptr = build_new_request_(SC_IPI_BUS, currtid, CURRENTTIME,
				  5, processor, processor_, priority, type, argc);
	
	for(i=0; i<argc; i++)
	  rptr->argv[5+i] = *(&args + i);
	rptr->h.argc += argc;
	
	enqueue_request_(rptr);
	TrapToSimulator();
#endif /* BUS */
#ifdef NET
	Packet *packet;
	
	packet = build_new_packet_(PKT_IPI, length, 3, type, priority, argc);
	
	for(i=0; i<argc; i++)
	  packet->data[3+i] = *(&args + i);
	
	SEND_PACKET_RQ(processor_, processor, packet, mode | RESUME_CALLER);

#endif /* NET */
    }
}



GLOBAL void send_ipiV(processor, priority, type, length, argc, argv) 
int processor;
int priority;
int type;
int length;
int argc;
Word *argv;
{
    int i, mode;
    
    mode = type & MODEMASK; 
    type = type & TYPEMASK; 

    if ( InvalidIpi(type) )
      fatal("IPI: Invalid ipi. type = %d", type);
    
    
#ifdef BUS
    if ( InvalidProc(processor) || argc+5 > REQ_MAX_ARGS )
      fatal("send_ipiV failed: Invalid processor %d or too many argunments %d",
	    processor, argc);
#else /* NET */
    if ( InvalidProc(processor)  || argc+3 > MAX_PACKET_WORDS )
      fatal("send_ipiV failed: Invalid processor %d or too many argunments %d",
	    processor, argc);
#endif
    
    
    if ( processor == processor_ ) {
	/* Local interrupts (traps) are delivered immediately */
	insert_new_interrupt(processor, processor, currtid, priority,
			     type, CURRENTTIME, argc, argv);
	CheckForInterrupts();
    } else {
#ifdef BUS
	SimRequest *rptr, *build_new_request_();
	
	rptr = build_new_request_(SC_IPI_BUS, currtid, CURRENTTIME,
				  5, processor, processor_, priority, type, argc);
	
	for(i=0; i<argc; i++)
	  rptr->argv[5+i] = *(argv + i);
	rptr->h.argc += argc;
	
	enqueue_request_(rptr);
	TrapToSimulator();
#endif
#ifdef NET
	Packet *packet;
	
	packet = build_new_packet_(PKT_IPI, length, 3, type, priority, argc);
	
	for(i=0; i<argc; i++)
	  packet->data[3+i] = *(argv + i);
	
	SEND_PACKET_RQ(processor_, processor, packet, mode | RESUME_CALLER);

#endif /* NET */
    }
}


#ifdef BUS

static Time ipi_free_again;

typedef struct {
    SimHeader  h;
    Word       target;   /* argv[0] */
    Word       sender;   /* argv[1] */
    Word       priority; /* argv[2] */
    Word       ipi_type; /* argv[3] */
    Word       iargc;    /* argv[4] */
    Word       iargv;    /* argv[5] */
} IpiRequest;


GLOBAL void ipi_bus_handler_(rptr) 
IpiRequest *rptr;
{
    
    ProcBlk *pptr = &proc_table_[ rptr->target ];
    ThreadInfo *tptr;
    Time itime;
    
    if ( InvalidIpi(rptr->ipi_type) )
      fatal("IPI_BUS: Invalid ipi. type = %d\n", rptr->ipi_type);
    
    ipi_free_again = MAX(ipi_free_again, rptr->h.timestamp) + rptr->iargc;
    
    /* todo: Implement completion interrupt for bus */

    /* Resume thread that invoked ipi */
    MAKE_RESUME_REQUEST(rptr->h.tid, ipi_free_again, OK);
    
    itime = MAX(ipi_free_again,
		proc_table_[rptr->target].p_time) + IPI_LATENCY; 
    
    insert_new_interrupt(rptr->target, rptr->sender, rptr->h.tid,
			 rptr->priority, rptr->ipi_type, itime,
			 rptr->iargc, &rptr->iargv);
}

#endif /* BUS */
