
 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: net.model.c
 *
 * Description:
 *         Network module. Receives packets, introduces appropriate delay
 *         and delivers them to the destination node.
 *
 * Last Modified:  $Date: 94/05/12 00:00:39 $
 *
 * Global Functions:
 *     void send_packet_handler_(SimRequest *rptr) 
 *     void receive_packet_handler_(SimRequest *rptr) 
 *     void route_packet_handler_(SimRequest *rptr) 
 *     void init_network_() 
 *     
 * Global Variables: none
 *
 ****************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.		
 ****************************************************************************
 * $Header: /usr/wildcat/dfk/research/parallel/proteus/proteus-V3.01/engine/RCS/net.model.c,v 1.3 94/05/12 00:00:39 dfk Exp $
 * $Log:	net.model.c,v $
 * Revision 1.3  94/05/12  00:00:39  dfk
 * added PARAGON flag
 * 
 * Revision 1.2  94/01/27  00:47:05  dfk
 * gcc bug when adding floats to unsigned long longs
 * I changed all floats to doubles to fix that.
 * 
 * Revision 1.1  94/01/26  23:40:07  dfk
 * Initial revision
 * 
 * Revision 1.5  92/11/13  17:03:47  brewer
 * Added adj's hash code that prevent message reordering.
 * 
 * Revision 1.4  92/11/13  16:35:45  brewer
 * Now calls dispatch_packet_ to handle packets.
 * 
 * Revision 1.3  92/05/01  17:23:46  brewer
 * Removed  rptr->h.timestamp -= SWITCH_DELAY from send_request_handler_
 * It sometimes caused CC messages to arrive out of order.
 * 
 * Revision 1.2  92/04/01  17:02:51  brewer
 * cleaned up gcc warnings, added ANSI prototypes
 * removed dispatch to controller/processor
 * 
 * Revision 1.1  92/02/11  13:56:03  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include "simcalls.h"
#include "thread.h"
#include "processor.h"
#include "cache.h"
#include "shmem.h"
#include "sema.h"
#include "event.h"
#include "packet.h"
#include <malloc.h>

static Time last_net_request;
static double net_req_prob = 0;
#define TABLE_SIZE  0x200
#define TABLE_MASK  0x1ff
typedef struct netEntrytype {
    int from, to;
    Time time;
    struct netEntrytype *next;
} NetEntryType;

static NetEntryType *table[TABLE_SIZE];

 /**************************************************************************\
 *   Hops in one dimension for bidirectional and unidirectional networks
 \**************************************************************************/

#ifdef N_BIDIR
#define hops_per_dim(dfrom, dto)   \
  MIN((abs(dto-dfrom)), (Nk - abs(dto-dfrom)))
#endif

#ifdef N_UNIDIR
#define hops_per_dim(dfrom, dto)   \
  ( dto >= dfrom ? dto-dfrom : Nk-dfrom+dto )
#endif

static int net_hops(int from, int to) 
{
    static int hopstab[NO_OF_PROCESSORS][NO_OF_PROCESSORS];
    unsigned hops = 0, dfrom, dto;
    int i, *hp;
    
    if ( *(hp = &hopstab[from][to]) )
      return( *hp );
    
    for(i=0; i<Ndim; i++) {
	dfrom = from % Nk;
	dto   = to   % Nk;
	from  = (from - dfrom) / Nk;
	to    = (to   - dto  ) / Nk;
	hops += hops_per_dim(dfrom, dto);
	/*
	  printf("Hops per dim (%d, %d) = %d\n", dfrom, dto,
	  hops_per_dim(dfrom, dto));
	  */
    }
    return( *hp = hops );
}


 /**************************************************************************\
 *  Routines for managing processor message receive times
 \**************************************************************************/

inline unsigned int NETHASH(int from, int to)
{
  register int code;
/*  code = ((from <<8)|to)/511; */
/*  code = (from^(from<<5)^(to<<1))&TABLE_MASK; */
  code = (to^(from<<5)^(from>>5))&TABLE_MASK;
  return(code);
}

NetEntryType *netLocate(int from, int to)
{
  NetEntryType *e;

  e = table[NETHASH(from,to)];
  while (e != NULL) {
    if ((e->from == from) && (e->to == to)) return e;
    e = e->next;
  }
  return NULL;
}

static void netInsert(int from, int to, Time time)
{
  NetEntryType *ptr, *e;
  
  e = (NetEntryType *) malloc(sizeof(NetEntryType));
  e->from = from;
  e->to = to;
  e->time = time;

  ptr = netLocate(from,to);
  if (ptr != NULL) fatal("Tried to insert duplicate entry in insert\n");

  e->next = table[NETHASH(from,to)];
  table[NETHASH(from,to)] = e;
}

void print_net_table_statistics()
{
  unsigned int i,j,count,bcount, start, ocount;
  unsigned int total, ecount, tmp;
  unsigned int hist[21], opt[21];
  unsigned int sumsq, sum;
  double stdev, optimal, ostdev;
  NetEntryType *ptr;
  extern double pow(double, double);

  for (i=0;i<21;i++)
    hist[i] = opt[i] = 0;

  sumsq = sum = 0;
  for (count=i=0;i<TABLE_SIZE;i++) {
    ecount = 0;
    ptr = table[i];
    while (ptr) {
      ecount++;
      ptr = ptr->next;
    }
    sumsq += ecount*ecount;
    sum += ecount;
    count += ecount;
    if (ecount >= 20)
      ecount = 20;
    hist[ecount]++;
  }
  stdev = pow(((double)sumsq/(double)TABLE_SIZE) - 
	      ((double)(sum*sum)/(double)(TABLE_SIZE*TABLE_SIZE)),0.5);


  start = count/TABLE_SIZE + 1;
  ocount = tmp = count % TABLE_SIZE;
  if (start < 20)
    opt[start] = tmp;
  optimal = start * start * tmp;
  start--;
  tmp = TABLE_SIZE - (count % TABLE_SIZE);
  ocount += tmp;
  if (start < 20)
    opt[start] = tmp;
  optimal += (start * start * tmp);
  optimal /= (double) ocount;

  sumsq = sum = 0;
  for (i=0;i<21;i++) {
    ecount = opt[i];
    for (j = 0; j < ecount; j++) {
      sumsq += i*i;
      sum += i;
    }
  }
  ostdev = pow(((double)sumsq/(double)TABLE_SIZE) - 
	       ((double)(sum*sum)/(double)(TABLE_SIZE*TABLE_SIZE)),0.5);

  printf("Chain Length:\tCount\tOptimal\n");
  bcount = total = 0;
  for (i=1;i<20;i++) {
    total += i * i * hist[i];
    bcount += hist[i];
    printf("%d:\t\t%d\t%d\n",i,hist[i],opt[i]);
  }
  total += 20 * 20 * hist[20];
  printf("20+:\t\t%d\n",hist[20]);
  printf("Total inserts: %d\n", count);
  printf("Std Dev. %f (optimal: %f)\n", stdev, ostdev);
  printf("Chain length factor %f (optimal: %f)\n",
	 ((double) total) / ((double) bcount), optimal);

}


 /**************************************************************************\
 *  Calculate network delay based on Agarwal's model
 \**************************************************************************/

static Time net_delay(int from, int to, Time sendtime, Packet *packet) 
{
    int hops;
    Time k;
    double chan_util;
    double base_delay, contention_delay, tot_delay;
    static double cum_contention_delay = 0;
    
    k = sendtime - last_net_request;
    
    last_net_request = sendtime;
    
    net_req_prob = ((double)(NETWINDOW - k) *net_req_prob + 1) / NETWINDOW;
    if ( net_req_prob < 0 ) net_req_prob = 1/NETWINDOW;
    
    chan_util = (double) net_req_prob * CHAN_UTIL_FACTOR *
      packet->length / NO_OF_PROCESSORS;
    if (chan_util >= MAX_CHAN_UTIL) chan_util = MAX_CHAN_UTIL;
    
#ifdef N_DIRECT
    hops = net_hops(from, to);
#endif
#ifdef N_INDIRECT
    hops = NET_STAGES;
#endif
    
#ifdef PARAGON		      /* dfk */
    base_delay = (double)(packet->length) * SWITCH_DELAY
      + (double)hops * (SWITCH_DELAY + WIRE_DELAY);
#else
    base_delay = (double)(packet->length + hops) * (SWITCH_DELAY + WIRE_DELAY);
#endif
    
    contention_delay = (double)chan_util * packet->length
      * CONTENTION_FACTOR / (1 - chan_util);
    
    cum_contention_delay += contention_delay * hops;
    if (cum_contention_delay > 1) {
	tot_delay = base_delay + (int)cum_contention_delay;
	cum_contention_delay -= (int)cum_contention_delay;
    }
    else
      tot_delay = base_delay;
    
    
#ifdef NETDEBUG
    printf("MODEL DELAY: T%s From %d to %d length %d receive time %s\n",
	   time_print(sendtime), from, to, packet->length,
	   time_print( (Time)(sendtime + tot_delay + 0.5)));
    printf("net_req_prob %1.5f hops %d tot_dly %2.2f\n",
	   net_req_prob, hops, tot_delay);
    printf("cont.factor = %f chan_util = %f base_delay = %f cont_delay = %f\n",
	   CONTENTION_FACTOR, chan_util, base_delay, contention_delay);
#endif
    
    return( (Time)(sendtime + tot_delay + 0.5) );
}


typedef struct {
    SimHeader  h;
    Word       from;        /* argv[0] */
    Word       to;          /* argv[1] */
    Packet     *packet;     /* argv[2] */
    Word       flags;       /* argv[3] */
    Word       previous;    /* argv[4] */
    Word       current;     /* argv[5] */
    Word       line;        /* argv[6] */
} SendRequest;

 /**************************************************************************\
 *  Handler to send a packet
 \**************************************************************************/

GLOBAL void send_packet_handler_(SendRequest *rptr) 
{
    
    Time receivetime;
    int from, to;
    NetEntryType *entry;
    
#ifdef NETDEBUG
    printf("#%d Net will send packet from %lu to %lu T%s\n",
           rptr->h.tid, rptr->from, rptr->to,
	   time_print(rptr->h.timestamp));
    DisplayPacket(stdout, rptr->packet);
#endif
    
    from = rptr->from;
    to   = rptr->to;
    
    if (InvalidProc(to))
      fatal("Attempt to send packet to nonexistent processor %d", to);
    
    if (rptr->flags & RESUME_CALLER) {
	/* Sending processor becomes free at `processor_ready' */
	/* this version computes time based on length of packet */
	Time processor_ready =
	  rptr->h.timestamp + rptr->packet->length;
	MAKE_RESUME_REQUEST(rptr->h.tid, processor_ready, OK);
    }
    
    /* Special case -- if dest == source then packet is immediately dispatched 
     * This feature is used in the cache coherence protocol to deliver memory
     * requests to a processor's own shared memory module
     */
    if (from == to) {
	rptr->current = from;               /* Make sure Route will dispatch */
	route_packet_handler_(rptr);
	return;
    }
    
    /* compute arrival time of packet based on Agarwal's model */
    receivetime = net_delay(from, to, rptr->h.timestamp, rptr->packet);
    receivetime -= SWITCH_DELAY;

    entry = netLocate(from,to);
    if (entry != NULL) {
      if (receivetime < entry->time) {
#ifdef NETDEBUG
	fprintf(stderr, ">>> Reordered message from %d to %d <<<\n", from, to);
#endif
	receivetime = entry->time;
      }
      entry->time = receivetime + rptr->packet->length;
    } else
      netInsert(from,to,receivetime + rptr->packet->length);
    
    MAKE_REQUEST_6(SC_ROUTE_PACKET, rptr->h.tid, receivetime,
		   from, to, rptr->packet, FALSE, from, to);
}


 /**************************************************************************\
 *  Handler for receiving from the network: receive_packet_handler_
 \**************************************************************************/

GLOBAL void receive_packet_handler_(rptr) 
     SendRequest *rptr;
{
    /* no intermediate hops in network model */
    fatal("receive_packet_handler_ called in modeled network version.");
}


 /**************************************************************************\
 *  Handler for routing: route_packet_handler_
 \**************************************************************************/

GLOBAL void route_packet_handler_(SendRequest *rptr) 
{
    
    Time receivetime;
    
    receivetime = rptr->h.timestamp;
    
    if (rptr->current == rptr->to) {
	
#ifdef NEWNETDEBUG
	printf("%d. (%d --> %d ) Packet received.\n",
	       receivetime, rptr->from, rptr->to);
#endif
	dispatch_packet_(rptr->h.tid, receivetime + SWITCH_DELAY, 
			 rptr->from, rptr->to, rptr->packet);
    } else {
	fatal("Packet (%d->%d) at intermediate node %d\n\tin modeled network.",
	      rptr->from, rptr->to, rptr->current);
    }
}


/***************************************************************************/

GLOBAL void init_network_() 
{
    net_req_prob = 0.0;
#ifdef N_DIRECT
    printf("(Modeled network simulation)\n");
#else
    printf("(Indirect network simulation)\n");
#endif
}


