
 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: packet.c
 *
 * Description:
 *         Packet module
 *
 * Last Modified:  $Date: 94/01/24 00:39:15 $
 *
 * Global Functions:
 *     void init_packet_(void)
 *     int define_new_packet(FuncPtr handler, char *name, int type)
 *     void dispatch_packet_(sendtid, itime, sender, target, packet) 
 *     Packet *build_new_packet_(class, length, argc, dummy, args) 
 *     void DisplayPacket(FILE *out, Packet *packet)
 *     
 * Global Variables: none
 *
 ****************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.		
 ****************************************************************************
 * $Header: /usr/wildcat/dfk/research/parallel/proteus/proteus-V3.01/engine/RCS/packet.c,v 1.2 94/01/24 00:39:15 dfk Time64bit Locker: dfk $
 * $Log:	packet.c,v $
 * Revision 1.2  94/01/24  00:39:15  dfk
 * Added Sanjay's support for 64-bit timers.
 * Added thread_sleep_until.
 * Fixed a few bugs in thread sleep/wakeup code (my bugs).
 * Fixed printf formats in many places.
 * 
 * Revision 1.1  94/01/23  16:19:07  dfk
 * Initial revision
 * 
 * Revision 1.6  92/11/19  14:20:24  brewer
 * Changed interface to build_new_packet_ to use stdarg convention.
 * Added support for SEND_RECEIVE communication model.
 * 
 * Revision 1.5  92/11/13  16:36:33  brewer
 * Replaced dispatch_processor and dispatch_controller with
 * dispatch_packet_.
 * Changed free_packet to static.
 * 
 * Revision 1.4  92/08/24  12:47:16  brewer
 * Rewrote new_packet, free_packet, and init_packet_ to use a linked list
 * of free packets instead of searching for a packet with a loop.
 * Also, improved the information provided by DisplayPacket -- it now
 * lists the first sixs args for an IPI packet.
 * 
 * Revision 1.3  92/05/04  11:46:51  brewer
 * Added PROTOCOL_MESSAGE abstraction to support simulations without networks
 * and caching.
 * 
 * Revision 1.2  92/05/01  17:24:11  brewer
 * Improved output of DisplayPacket for cache protocol messages.
 * 
 * Revision 1.1  92/02/11  13:56:04  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include "sim.h"
#include "simcalls.h"
#include "conf.param"
#include "cache.h"
#include "net.h"
#include "packet.h"

#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

static FuncPtr packet_handler_[MAX_PKT_TYPES];
static char *packet_name_[MAX_PKT_TYPES];
static Packet packet_table[MAX_PACKETS];
static Packet *free_head, *free_tail;

extern void Nocache_Pkt();
extern void CtoM_Pkt();
extern void MtoC_Pkt();


 /**************************************************************************\
 *  Packet list management
 \**************************************************************************/

static Packet *new_packet(void) 
{
    Packet *pkt;
    
    pkt = free_head;
    if (pkt == NULL)
      fatal("Out of packets (increase MAX_PACKETS ?)\n");

    free_head = pkt->next;
    return(pkt);
}


static void free_packet(packet) 
Packet *packet;
{
    if (free_tail->next != NULL)
      fatal("Free packet list tail is invalid");
    
    free_tail->next = packet;
    packet->next = (Packet *)NULL;
    free_tail = packet;
}


 /**************************************************************************\
 *  Define a new packet type with handler and name
 \**************************************************************************/

GLOBAL int define_new_packet(FuncPtr handler, char *name, int type)
{
    int i;

    if (type < -1 || type >= MAX_PKT_TYPES) {
	fatal("Requested packet handler number %d out of range.", type);
	return(0); /* unreachable, but helps compiler */
    }

    if (type >= 0) {
	if (packet_handler_[type] == NULL) {
	    packet_handler_[type] = handler;
	    packet_name_[type] = name;
	    return(type);
	}
    }

    for (i=0; i<MAX_PKT_TYPES; i++) {
	if (packet_handler_[i] == NULL) {
	    packet_handler_[i] = handler;
	    packet_name_[i] = name;
	    return(i);
	}
    }
    fatal("Out of packet types -- unable to define \"%s\".", name);
    return(0); /* unreachable, but pacifies compiler */
}


GLOBAL void init_packet_(void) 
{
    int i, type;;
    
    for(i=0; i<MAX_PACKETS-1; i++)
      packet_table[i].next = &packet_table[i+1];

    packet_table[MAX_PACKETS-1].next = (Packet *)NULL;
    
    free_head = packet_table;
    free_tail = &packet_table[MAX_PACKETS - 1];

#ifdef NOCACHE
    type = define_new_packet((FuncPtr)Nocache_Pkt, "NoCache", NOCACHE_PKT);
    assert(type == NOCACHE_PKT);
#else
#ifdef YESCACHE
    type = define_new_packet((FuncPtr)CtoM_Pkt, "C to M", CTOM_PKT);
    assert(type == CTOM_PKT);
    type = define_new_packet((FuncPtr)MtoC_Pkt, "M to C", MTOC_PKT);
    assert(type == MTOC_PKT);
#else
    error: neither NOCACHE nor YESCACHE defined
#endif
#endif
}



 /**************************************************************************\
 *  Network buffer code for networks that queue arriving messages, for
 *  the synchronous send-receive model
 \**************************************************************************/

#ifdef SEND_RECEIVE

GLOBAL Packet *packet_heads_[NO_OF_PROCESSORS];
GLOBAL Packet *packet_tails_[NO_OF_PROCESSORS];
GLOBAL int packet_count_[NO_OF_PROCESSORS];
GLOBAL Packet *previous_packet_[NO_OF_PROCESSORS];

static void enqueue_packet(Packet *p, int proc)
{
    int i;

    assert(proc>=0 && proc<NO_OF_PROCESSORS && p!=NULL);
    p->next = NULL;
    if (packet_tails_[proc] == NULL) {
	packet_heads_[proc] = packet_tails_[proc] = p;
    } else {
	packet_tails_[proc]->next = p;
	packet_tails_[proc] = p;
    }
    packet_count_[proc]++;

    for (p = packet_heads_[proc], i=0; p != NULL; i++, p = p->next);
    if (packet_count_[proc] != i)
      fatal("enqueue_packet: packet count off, proc %d", proc);
}

static Packet *dequeue_packet(int proc)
{
    Packet *p, *q; int i;

    p = packet_heads_[proc];
    if (p == NULL) return(NULL);
    
    if ((packet_heads_[proc] = p->next) == NULL) {
	packet_tails_[proc] = NULL;
    }
    packet_count_[proc]--;

    for (q = packet_heads_[proc], i=0; q != NULL; i++, q = q->next);
    if (packet_count_[proc] != i)
      fatal("enqueue_packet: packet count off, proc %d", proc);

    return(p);
}


GLOBAL IpiPacket *ReceivePacket()
{
    Packet *p;

    int proc = processor_;
    p = previous_packet_[proc];
    if (p != NULL) free_packet(p);
    p = dequeue_packet(proc);
    previous_packet_[proc] = p;
    cycles_ -= 10;
    return((IpiPacket *)p);
}


GLOBAL IpiPacket *SpinPollReceive()
{
    IpiPacket *p;
    while ((p = ReceivePacket()) == NULL) {
	cycles_ -= 4;
	PassControl();
    }
    return(p);
}

GLOBAL void DisplayMessageQueue(FILE *fp, int proc)
{
    Packet *p = packet_heads_[proc];

    fprintf(fp, "Message Queue for Processor %d: %d message%s\n",
	    proc, packet_count_[proc], PLURAL(packet_count_[proc]));

    while(p != NULL) {
	fprintf(fp, "    ");
	DisplayPacket(fp, p);
	p = p->next;
    }
}

#else /* not SEND_RECEIVE */

GLOBAL IpiPacket *ReceivePacket()
{
    fatal("ReceivePacket called in active-messages model.");
    return(NULL); /* unreachable, but keeps compiler happy */
}

GLOBAL IpiPacket *SpinPollReceive()
{
    fatal("SpinPollReceive called in active-messages model.");
    return(NULL); /* unreachable, but keeps compiler happy */
}

GLOBAL void DisplayMessageQueue(FILE *fp, int proc)
{
    fprintf(fp, "Message Queue: invalid in Active-Messages model.\n");
}

#endif


 /**************************************************************************\
 *  Dispatch a packet to either the processor or the cache controller
 \**************************************************************************/

GLOBAL void dispatch_packet_(ulong sendtid, Time time, int sender, int target,
			     Packet *packet)
{
    MemPacket *mp;

    switch(packet->class) {
      case PKT_IPI: {
#ifdef SEND_RECEIVE
	enqueue_packet(packet, target);
#else /* interrupt processor */
	IpiPacket *ip = (IpiPacket *) packet;
	insert_new_interrupt(target, sender, sendtid, ip->iprio,
			     ip->itype, time, ip->argc, &ip->argv);
	free_packet(packet);
#endif
	break; }
      case PKT_MEM:
	mp = (MemPacket *) packet;
    
	(*packet_handler_[mp->type])(sendtid, time, sender, target,
				     &mp->rest);
	free_packet(packet);
	break;
      default:
	fatal("Node %d: Received invalid packet (class : %d)",
	      target, packet->class);
    }	
}




 /**************************************************************************\
 *  Build a packet
 \**************************************************************************/

/* dummy is inserted to ensure that args is stored on the stack
 * (first 4 args of any func. are passed in registers)
 * At the time this was written, there was no format for variable number
 * of args that both cc and gcc understood.
 */

#ifdef __STDC__
GLOBAL Packet *build_new_packet_(int class, int length, int argc, ...)
#else
GLOBAL Packet *build_new_packet_(va_alist)
va_dcl
#endif
{
    va_list ap;
    int i;
    Packet *packet;
    
#ifdef __STDC__
    va_start(ap, argc);
#else
    int class, length, argc;
    va_start(ap);
    class = va_arg(ap, int);
    length = va_arg(ap, int);
    argc = va_arg(ap, int);
#endif

    packet = new_packet();
    
    packet->class = class;
    packet->length = length;
    for (i=0; i<argc; i++) packet->data[i] = va_arg(ap, Word);
    
    return(packet);
}



 /**************************************************************************\
 *  Display a packet in human-readable form
 \**************************************************************************/

#define PROTOCOL_MESSAGE(x) "Invalid Message Type!"

#ifdef YESCACHE
#ifdef NET
#undef PROTOCOL_MESSAGE(x)
extern char *protocol_msg_[];
#define PROTOCOL_MESSAGE(x) protocol_msg_[x]
#endif
#endif

GLOBAL void DisplayPacket(FILE *out, Packet *packet) 
{
    int itype, i, argc;
    Word *argv;

    switch(packet->class) {
      case PKT_IPI:
	itype = ((IpiPacket *)packet)->itype;
	argc = ((IpiPacket *)packet)->argc;
	if (argc > 0) {
	    argv = &(((IpiPacket *)packet)->argv);
	    if (intr_name_[itype] != NULL)
	      fprintf(out, "%s(%d", intr_name_[itype], (int)argv[0]);
	    else
	      fprintf(out, "type=%d, args=(%d", itype, (int)argv[0]);

	    for (i=1; i<MIN(argc, 6); i++)
	      fprintf(out, ", %d", (int)argv[i]);

	    if (i==argc) /* printed all args */
	      fprintf(out, ")\n");
	    else /* additional args not shown */
	      fprintf(out, ", ...)\n");
        } else { /* no args */
	    if (intr_name_[itype] != NULL) 
	      fprintf(out, "%s()\n", intr_name_[itype]);
	    else
	      fprintf(out, "type=%d, no args\n", itype);
	}
	break;
      case PKT_MEM:
	switch(packet->data[0]) {
	  case CTOM_PKT:
	    fprintf(out, "Cache to Mem: type %s, Proc %lu, 0x%lx, line %lu\n",
		    PROTOCOL_MESSAGE(packet->data[1]),
		    packet->data[2], packet->data[3], packet->data[4]);
	    break;
	  case MTOC_PKT:
	    fprintf(out, "Mem to Cache: type %s, Proc %lu, 0x%lx, line %lu\n",
		    PROTOCOL_MESSAGE(packet->data[1]),
		    packet->data[2], packet->data[3], packet->data[4]);
	    break;
	  case NOCACHE_PKT:
	    fprintf(out, "Shared Memory: %lu, address = 0x%lx\n",
		    ((NOCPacket *)(packet))->valaddr,
		    ((NOCPacket *)(packet))->address);
	    break;
	  default:
	    fprintf(out, "Invalid memory packet type: %ld\n",
		    (long) packet->data[0]);
	}
	break;
      default:
	fprintf(out, "Invalid packet class.\n");
    }
}

