
 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: processor.c
 *
 * Description: Processor management
 *
 * Last Modified:  $Date: 94/01/24 00:39:16 $  (eab)
 *
 * Global Functions:
 *     void init_processors_(void)
 *     void RecordProcessorUtilization(time, proc, state)
 *     void ExitProcUtilStat(void)
 *     void define_local(char **address, char *base, ulong length)
 *     void set_local_pointers(int proc)
 *
 * Global Variables:
 *     ProcBlk proc_table_[NO_OF_PROCESSORS]
 *     ProcBlk *currpptr
 *     int idle_processors
 *
 *     Single ready list:
 *         int idle_head, ready_head, ready_tail, ready_threads
 *
 * Referenced Parameters:
 *     NO_OF_PROCESSORS, WATCH_UTIL, SINGLE_RL, OS_QUANTUM
 *
 ****************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.
 ****************************************************************************
 * $Header: /usr/wildcat/dfk/research/parallel/proteus/proteus-V3.01/engine/RCS/processor.c,v 1.2 94/01/24 00:39:16 dfk Time64bit Locker: dfk $
 * $Log:	processor.c,v $
 * Revision 1.2  94/01/24  00:39:16  dfk
 * Added Sanjay's support for 64-bit timers.
 * Added thread_sleep_until.
 * Fixed a few bugs in thread sleep/wakeup code (my bugs).
 * Fixed printf formats in many places.
 * 
 * Revision 1.1  94/01/23  11:35:01  dfk
 * Initial revision
 * 
 * Revision 1.2  92/11/19  13:50:07  brewer
 * Imported UserPointer typedef from processor.h; added set_local_pointers
 * to provide the same capability with improved modularity.
 * 
 * Revision 1.1  92/02/11  13:56:05  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include "sim.h"
#include "conf.h"
#include "event.h"
#include "events.h"
#include "rt_thread_def.h"
#include "simreq.h"
#include "processor.h"
#include "thread.h"
#include "simcalls.h"
#include "q.h"

#define MAX_LOCAL_POINTERS 20

typedef struct {
    char **address;
    char *base;
    ulong length;
} UserPointer;

GLOBAL ProcBlk proc_table_[NO_OF_PROCESSORS];
GLOBAL ProcBlk *currpptr;
GLOBAL int idle_processors;

#ifdef SINGLE_RL
GLOBAL int idle_head;
GLOBAL int ready_head, ready_tail;
GLOBAL int ready_threads;
#endif

/*************************************************************************/

void init_processors_() 
{
    int i;
    char iname[20];
    Thread *iptr;
    extern void ihandler();

#ifdef SINGLE_RL
    ready_tail = 1 + (ready_head = newqueue() );
    ready_threads = 0;
#endif
    
    for(i = NO_OF_PROCESSORS - 1; i >= 0; i--) {
	ProcBlk *pptr = &proc_table_[i];
	
	pptr->p_time = 0;
	pptr->p_timer = TIMER_PERIOD;
	define_timer_handler(i, (FuncPtr)I_DEFAULT);
	define_switch_handler(i,(FuncPtr) I_DEFAULT);
	define_mismatch_handler(i, (FuncPtr)I_DEFAULT);
	define_error_handler(i, (FuncPtr)I_DEFAULT);
	
#ifdef MULT_RL
	pptr->p_ready_tail = 1 + ( pptr->p_ready_head = newqueue() );
	pptr->p_numthreads = pptr->p_ready_threads = 0;
#endif
	
	MakeIdle(i);
	WriteTimer(i, (Time) TIMER_PERIOD);
	sprintf(iname,"ihandler@%d", i);
	processor_ = i;
	pptr->p_ihandler = simulator_tid();
	iptr = &processor_private[i].thread_table[0];
	if ( (create_thread(pptr->p_ihandler, (FuncPtr)ihandler, 4,
			    (const char *)iname, (Time) 0,
			    (void *)iptr, 1, (Word *)&processor_)) == ERROR )
	  fatal("cannot spawn ihandler for processor %d", i);
	
	iptr->t_tid = 0;
	iptr->t_stid = pptr->p_ihandler;
	iptr->t_state = T_SUSPENDED;
	iptr->t_processor = i;
	iptr->priority = OS_CALL_PRIORITY;
	iptr->t_preempt  = PREEMPT_OFF;
	iptr->t_intrflag = INTR_OFF;
    }
}


/***************************************************************************/

/* initialize per-processor local pointer */

static UserPointer user_pointer_table_[MAX_LOCAL_POINTERS];


GLOBAL void define_local(char **address, char *base, ulong length)
{
    int i;

    if (address == 0)
      fatal("`define_local' called with `address == NULL'");

    for (i = 0; i < MAX_LOCAL_POINTERS; i++) {
	if (user_pointer_table_[i].address == 0) break;
    }

    if (i == MAX_LOCAL_POINTERS)
      fatal("Out of local pointers in `define_local'; (adjust MAX_LOCAL_POINTERS");

    user_pointer_table_[i].address = address;
    user_pointer_table_[i].base = base;
    user_pointer_table_[i].length = length;

    /* initialize pointer */
    *address = base + length*processor_;
}


GLOBAL void set_local_pointers(int proc)
{
    UserPointer *p;

    for (p = user_pointer_table_; p->address != 0; p++)
      *(p->address) = p->base + proc * p->length;
}
