 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: prototypes.h
 *
 * Description: Function prototypes for all GLOBAL procedures
 *
 * Last Modified: $Date: 94/01/24 00:39:20 $
 *
 * Data Structures: none
 *
 * Macros:
 *     UNREACHABLE()   -- indicates code that should be unreachable,
 *                        and generates fatal error if reached
 *
 * Constants:
 *     GLOBAL, LIBRARY
 *
 ****************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.
 ***************************************************************************
 * $Header: /usr/wildcat/dfk/research/parallel/proteus/proteus-V3.01/engine/RCS/prototypes.h,v 1.3 94/01/24 00:39:20 dfk Time64bit Locker: dfk $
 * $Log:	prototypes.h,v $
 * Revision 1.3  94/01/24  00:39:20  dfk
 * Added Sanjay's support for 64-bit timers.
 * Added thread_sleep_until.
 * Fixed a few bugs in thread sleep/wakeup code (my bugs).
 * Fixed printf formats in many places.
 * 
 * Revision 1.2  93/02/28  11:18:43  dfk
 * >  * changed the sleeping-thread system to be much more efficient.
 * >  * If a processor is idle, it will skip ahead to the wakeup time
 * >  * for the first sleeping thread.
 * >  * It runs MUCH faster than spinning on the TIMER_PERIOD method.
 * >  * Questions:
 * >  *   how does it interact with preemptive scheduling? It should work.
 * >  *   how does it interact with semaphores, which also use timer requests
 * >  *   does it work with multiple sleeping threads? it should
 * >  *   does it work when a new thread is creating on a sleeping processor? should.
 * 
 * Revision 1.1  93/02/28  10:22:54  dfk
 * Initial revision
 * 
 * Revision 1.19  92/12/09  15:16:51  brewer
 * Update tracing routines to take FILE *
 * 
 * Revision 1.18  92/12/02  12:31:23  brewer
 * Added fast_random, vfast_random
 * 
 * Revision 1.17  92/11/25  17:48:03  brewer
 * added shared-memory primitives
 * 
 * Revision 1.16  92/11/19  14:21:43  brewer
 * Added Send-Receive procedures.
 * Added thread_suspend_end_atomic.
 * 
 * Revision 1.15  92/10/07  16:42:47  brewer
 * removed OutputProfData
 * 
 * Revision 1.14  92/10/05  12:25:14  brewer
 * Removed 'void' arg types for thread_create and thread_create_and_wakeup
 * for the non __STDC__ case.
 * 
 * Revision 1.13  92/09/23  15:48:41  brewer
 * Removed redundant LIBRARY prototypes
 * 
 * Revision 1.12  92/09/23  12:20:10  brewer
 * Updated trace_stack_
 * Changed prototypes to declare arg 'void' if they take no arguments.
 * 
 * Revision 1.11  92/07/06  13:10:39  brewer
 * Placed ifdef STDC around declaration of define_new_simcall
 * 
 * Revision 1.10  92/07/06  11:24:03  brewer
 * Added define_new_simcall
 * 
 * Revision 1.9  92/05/08  12:48:39  brewer
 * Added define_new_ipi
 * 
 * Revision 1.8  92/04/28  11:21:00  brewer
 * Added set_prof_
 * 
 * Revision 1.7  92/04/09  11:47:20  brewer
 * Added  void PassControl()
 * 
 * Revision 1.6  92/04/02  16:48:17  brewer
 * Changed prototypes of ReadTimer, WriteTimer
 * 
 * Revision 1.5  92/04/02  13:20:52  brewer
 * added UNREACHABLE macro
 * 
 * Revision 1.4  92/04/01  11:25:09  brewer
 * added shared memory access procedures
 * 
 * Revision 1.3  92/03/09  12:22:19  brewer
 * Updated for gcc 2.0; changed return type of `exit' to void
 * 
 * Revision 1.2  92/02/12  12:22:03  brewer
 * Added thread_create_and_wakeup
 * 
 * Revision 1.1  92/02/11  14:19:54  brewer
 * Initial revision
 * 
 \**************************************************************************/

#ifndef _PROTOTYPES_H
#define _PROTOTYPES_H
  
#include "common.h"

/**************************************************************************\
* Usermain
\**************************************************************************/

void usermain(int argc, char **argv);



/**************************************************************************\
* Context-switching procedures
\**************************************************************************/

#ifdef __STDC__  /* cc2.1 can't handle ellipses */
Word SimCall(int reqtype, unsigned argc,                      /* simreq.c */
	     Word dummy1, Word dummy2,
	     Word argv, ...);
#else
Word SimCall();
#endif

void thread_top_level(void);                                  /* thread.c */
void TrapToSimulator(void);                                   /* resume.c */
void PassControl(void);                                       /* resume.c */
void ctxsw(Word *, Word *);                                   /* ctxsw.s */



/**************************************************************************\
* Simulator threads-package routines   (thread.c)
\**************************************************************************/

void init_threads(void);
void init_processors_(void);
ulong simulator_tid(void);
ulong create_thread(ulong sim_tid, FuncPtr func, unsigned stacksize,
		    const char *name, Time timestamp,
		    void *OS_block, unsigned argc, Word *argv);
int resume_thread(ulong sim_tid, Time timestamp);
void release_simulator_tid(ulong sim_tid);



/**************************************************************************\
* Interrupt procedures   (ihandler.c, ipi.c)
\**************************************************************************/

void CheckForInterrupts(void);
BOOL PendingInterrupts(void);
void insert_new_interrupt(int, int, ulong, int, ulong, Time, int, Word *);
Word lock(void);
void unlock(Word previous);
void begin_atomic(void);
void end_atomic(void);
void ASSERT_ATOMIC(void);
void ASSERT_NOT_ATOMIC(void);

#ifdef __STDC__  /* cc2.1 can't handle ellipses */
void send_ipi(int processor, int priority, int type, int length,
	      int argc, Word args, ...);
#else
void send_ipi();
#endif

void send_ipiV(int processor, int priority, int type, int length,
	       int argc, Word *argv);

/* `define_ipi' sets the interrupt handler for interrupts of type `type'
    to `handler'.  If it is a user-defined ipi, the `name' sets the name
    of the interrupt for debugging.  define_ipi returns the previous handler
    for this interrupt type.  If `handler' is NULL, the current handler is
    returned and the handler and name do not change. */
   
FuncPtr define_ipi(int type, FuncPtr handler, const char *name);
int define_new_ipi(FuncPtr handler, const char *name);

/**************************************************************************\
* Send-Receive procedures   (ipi.c)
\**************************************************************************/

#define IpiPacket struct ipiStruct
IpiPacket *ReceivePacket(void);
IpiPacket *SpinPollReceive(void);
#undef IpiPacket

/**************************************************************************\
* Statistics/Event procedures (debug.c)
\**************************************************************************/

void user_event_(ulong, ulong);
void user_time_event_(ulong, ulong, Time);
void user_index_event_(ulong, ulong, ulong);
void user_index_time_event_(ulong, ulong, ulong, Time);

/* others located in event.h */



/**************************************************************************\
* Debugging procedures  (profile.c, debug.c) 
\**************************************************************************/

#ifdef __STDC__
void fatal(const char *format, ...);
#else
void fatal();                                               /* fatal.c */
#endif

void snapshot(void);                                        /* snapshot.c */

void Disassemble(ulong address, ulong pc);
void DisassembleSymbol(const char *symbol);
void trace_thread_(FILE *fp, ulong tid);
void trace_stack_(FILE *fp, ulong pc, ulong *tregs, ulong tid);
ulong GetAddress_(void);
ulong GetStack_(void);
void set_prof_(ulong pc, profileInfo *prof);
void AddOverwriteCheck(long *address, const char *name);
void ListOverwriteChecks(void);
void OverwriteCheck(void);

void bomb_(void);
void catch_(void);
void control_C_();                  /* snapshot.c */
void bus_error_();                  /* debug.c */

/* `UserDebug' is called on command from the snapshot menu.  The default
   routine simply returns, but users can define their own version that
   does whatever they like.  It is intended as a way to hook in
   application-specific debugging routines, such as printing out a
   particular data structure. */

LIBRARY void UserDebug(void);



/**************************************************************************\
* Memory routines  (mem.c)
\**************************************************************************/

/* `define_local' takes the address of a pointer, an array of NO_OF_PROCESSORS
    elements, and a length for each element.  The type of the pointer should
    be *elementType, where elementType is the type of the array element.  After
    define_local is executed, the pointer will *always* be set to the element
    of the array whose index is the current processor, or
         pointer = &array[processor_].
    This is useful for declaring variables that are global, but unique for
    each processor. */

void define_local(char **address_of_pointer,        /* processor.c */
		  char *array_base,
		  ulong element_length);

int allocstk(unsigned nblks);
int freestk(unsigned n);
int extendstk(unsigned n);
char *stktop(unsigned n);
char *stkbase(unsigned n);
int makepool(unsigned maxblks);



/**************************************************************************\
* Profiling procedures  (profile.c)
\**************************************************************************/

void init_profile(char *cmd);
const char *sym_name_(ulong address);


/**************************************************************************\
* Semaphore procedures  (sema.c)
\**************************************************************************/

typedef Word Sem;

int sem_P(Sem sem);
int sem_V(Sem sem);
Sem sem_open(Word value, const char *name, int processor);
int sem_close(Sem sem);
void init_semaphores(void);


/**************************************************************************\
* Shared-Memory procedures  (shmem.calls.c)
\**************************************************************************/
Word Shared_Memory_Read(void *address, Word mode);
Word Shared_Memory_Write(void *address, Word value, Word mode);
Word Shared_Memory_ReadTag(void *address, Word mode);
Word Shared_Memory_WriteTag(void *address, Word value, Word mode);
double Shared_Memory_Read_Fl(void *address, Word mode);
double Shared_Memory_Write_Fl(void *address, double value, Word mode);
int Flush(void *address);
void Fence(void);


/**************************************************************************\
* Runtime kernel procedures  (rt_thread.ca, rt_thread.aux.c)
\**************************************************************************/

int thread_createV(FuncPtr func, unsigned stacksize, Word priority,
		   unsigned argc, Word *argv);

#ifdef __STDC__   /* cc2.1 can't handle ellipses */ 
int thread_create(FuncPtr func, unsigned stacksize, Word priority,
		  unsigned argc, Word args, ...);
int thread_create_and_wakeup(FuncPtr func, unsigned stacksize, Word priority,
		  unsigned argc, Word args, ...);
#else
int thread_create();
int thread_create_and_wakeup();
#endif

int thread_destroy(int runtime_tid);
int thread_set_priority(int runtime_tid, Word newprio);
void thread_yield_to_scheduler(void);
void _thread_yieldToScheduler(void);
int thread_suspend(int runtime_tid);
void thread_suspend_end_atomic(void);
int thread_wakeup(int runtime_tid);
int thread_sleep(int runtime_tid, int duration); /* in TIMER_PERIODs */
int thread_sleep_until(int runtime_tid, Time wakeup_time);
int thread_start_forwarding(int runtime_tid, int newproc, Word newtidaddr);

ulong fast_random(void);
ulong vfast_random(ulong state);

Time ReadTimer(int p);
void WriteTimer(int p, Time t);
void _timer_handler(int processor, int currtid, Time currenttime);
FuncPtr define_timer_handler(int processor, FuncPtr handler);
FuncPtr define_switch_handler(int processor, FuncPtr handler);
FuncPtr define_mismatch_handler(int processor, FuncPtr handler);
FuncPtr define_error_handler(int processor, FuncPtr handler);

/* memory management routines */
void init_shmem(void);
char *OS_getmem(ulong nbytes);
char *OS_getmemfrommodule(ulong nbytes, int module);
int OS_freemem(char *block, ulong nbytes);
int OS_free(char *block);
void dump_module(FILE *fp, int module);
void view_shared_memory(FILE *fp); 
void print_memory_statistics(FILE *fp);
void diagnostic_shared_memory(void);


/**************************************************************************\
* Generic queue procedures  (q.c)
\**************************************************************************/

void init_Q(void);
int enqueue(int item, int tail);
int dequeue(int item);
void insert(int tid, int head, Time key);
int getfirst(int head);
int getlast(int tail);
int newqueue(void);



/**************************************************************************\
* Priority Heap procedures  (req_queue.c)
\**************************************************************************/

#define SimRequest struct simreq

void init_request_queue_(void);
SimRequest *next_request_(void);
void enqueue_request_(SimRequest *request);
void enqueue_snapshot_(Time time);
SimRequest *new_request_(void);
SimRequest *InitGenRequest_(void);
SimRequest *NextGenRequest_(void);
void free_request_(SimRequest *request);
#ifdef __STDC__
int define_new_simcall(char *name,
		       void (*handler)(SimRequest *),
		       void (*print)(FILE *, SimRequest *));
#else
int define_new_simcall(char *name, void (*handler)(), void (*print)());
#endif

#undef SimRequest

void make_timer_request_(int processor,           /* simreq.c */
			 ulong sim_tid,
			 Time time);

#endif /* _PROTOTYPES_H */
