
/**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module:  q.c
 *
 * Description: General thread queue routines
 *
 * Last Modified:  8-23-91  (eab)
 *
 * Global Functions:
 *     void init_Q() 
 *     int enqueue(item, tail) 
 *     int dequeue(item)
 *     void insert(tid, head, key)
 *     int getfirst(head)
 *     int getlast(tail)
 *     int newqueue()
 *
 * Global Variables:
 *     Qnode q[QSIZ];
 *
 * Referenced Parameters: MAX_THREADS
 *
 ***************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.		
 ****************************************************************************
 * $Header: /usr/wildcat/dfk/research/parallel/proteus/proteus-V3.01/engine/RCS/q.c,v 1.2 94/01/24 00:39:22 dfk Time64bit Locker: dfk $
 * $Log:	q.c,v $
 * Revision 1.2  94/01/24  00:39:22  dfk
 * Added Sanjay's support for 64-bit timers.
 * Added thread_sleep_until.
 * Fixed a few bugs in thread sleep/wakeup code (my bugs).
 * Fixed printf formats in many places.
 * 
 * Revision 1.1  94/01/23  17:29:05  dfk
 * Initial revision
 * 
 * Revision 1.1  92/02/11  13:56:07  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include "sim.h"
#include "conf.h"
#include "rt_thread_def.h"
#include "cache.h"
#include "shmem.h"
#include "sema.h"
#include "q.h"

GLOBAL Qnode q[QSIZ];
static int nextqueue;

/* min and max key (time) values */
#define MINKEY 0
#ifdef LONG_LONG_TIME
#define MAXKEY (Time)(0xffffffffffffffffLL)
#else
#define MAXKEY (Time)(0xffffffffL)
#endif

/* enqueue -- Insert item at end of a queue */

GLOBAL int enqueue(item, tail) 
     int item;
     int tail;
{
    Qnode *tp, *mp;
    
    tp = &q[tail];
    mp = &q[item];
    mp->qnext = tail;
    mp->qprev = tp->qprev;
    q[tp->qprev].qnext = item;
    tp->qprev = item;
    return(item);
}


/* dequeue -- Remove item from its queue */

GLOBAL int dequeue(item) 
     int item;
{
    Qnode *mp;
    
    mp = &q[item];
    q[mp->qprev].qnext = mp->qnext;
    q[mp->qnext].qprev = mp->qprev;
    return(item);
}


/* insert -- Insert thread tid in priority queue given its head and the
 *           thread's priority as key
 */

GLOBAL void insert(int tid, int head, Time key) 
{
    int next = q[head].qnext;
    int prev;
    
    while( q[next].qkey < key )
      next = q[next].qnext;
    q[tid ].qnext = next;
    q[tid ].qprev = prev = q[next].qprev;
    q[tid ].qkey = key;
    q[prev].qnext = tid;
    q[next].qprev = tid;
}


/* getfirst -- Remove and return first tid in a queue */

GLOBAL int getfirst(head) 
     int head;
{
    int tid;
    
    if ((tid = q[head].qnext) < MAX_THREADS)
      return(dequeue(tid));
    else
      return(EMPTY);
}


/* getlast -- Remove and return last tid in a queue */

GLOBAL int getlast(tail) 
     int tail;
{
    int tid;
    
    if ((tid = q[tail].qprev) < MAX_THREADS)
      return(dequeue(tid));
    else
      return(EMPTY);
}


/* newqueue -- Create and initialize a new queue within q table */

GLOBAL int newqueue() 
{
    int hindex, tindex;
    Qnode *hp;          /* Address of new queue head */
    Qnode *tp;          /* Address of new queue tail */
    
    hp = &q[ hindex = nextqueue++ ];
    tp = &q[ tindex = nextqueue++ ];
    hp->qnext = tindex;
    hp->qprev = EMPTY;
    hp->qkey  = MINKEY;
    tp->qnext = EMPTY;
    tp->qprev = hindex;
    tp->qkey  = MAXKEY;
    return(hindex);
}


/* init_Q -- initialize the q structure */

GLOBAL void init_Q() 
{
    nextqueue = MAX_THREADS;
}

