 
 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator                  *
 *                Eric A. Brewer  and  Chris N. Dellarocas                  *
 *                     Laboratory for Computer Science                      *
 *                  Massachusetts Institute of Technology                   *
 *
 * Module:  Simulator Request Priority Queue                                *
 *
 * Description:  This module uses a hash table to implement a priority      *
 *    queue.  The hash function is simply the bottom k bits, calculated     *
 *    by bitwise AND with the constant "MASK"; this implies the size of     *
 *    the hash table must be a power of 2.  The current time is the sum of  *
 *    "hash_time_", a multiple of the size that represents the time for     *
 *    slot 0 of the table, and "hash_index_", which is the the index of the *
 *    active slot.  The Operations:                                         *
 *        1) Initialize queue                                               *
 *        2) Extract Minimum element (by timestamp)                         *
 *        3) Insert                                                         *
 *        4) Generator routines: initialize and next                        *
 *        5) Memory Management: "new" and "free"                            *
 *        6) Insert "snapshot" request                                      *
 *
 * Last Modified:  6-27-91  (eab)
 *
 * Global Functions:                                                        *
 *        void init_request_queue_()                                        *
 *           Initialize priority queue                                      *
 *        SimRequest *next_request_()                                       *
 *           Extract Min: get request with minimum timestamp                *
 *           Returns NULL if the priority queue is empty                    *
 *        void enqueue_request_(SimRequest *request)                        *
 *           Insert request                                                 *
 *        SimRequest *new_request_()                                        *
 *           Allocate memory for a new request                              *
 *        void free_request_(SimRequest *request)                           *
 *           Deallocate memory of a request (put on a free list)            *
 *        SimRequest *InitGenRequest_()                                     *
 *           Start THE generator; returns NULL if the queue is empty        *
 *        SimRequest *NextGenRequest_()                                     *
 *           Generate next request; returns NULL if no more requests        *
 *        void enqueue_snapshot_(Time time)                                 *
 *           Insert "snapshot" element (without affecting determinism)      *
 *
 * Global Variables:                                                        *
 *        int hash_index_                                                   *
 *        Time hash_time_                                                   *
 *           These are used to compute the current time inside *.c files    *
 *           (GLOBALTIME defined as hash_time_ + hash_index_)               *
 *
 * Referenced Parameters:                                                   *
 *        NET, YESCACHE, RANDOMREQ, MAX_SIMREQS                             *
 *
 ****************************************************************************
 *                                                                          *
 *   Copyright 1991                                                         *
 *   Eric A. Brewer  and  Chris N. Dellarocas                               *
 *   Massachusetts Institute of Technology                                  *
 *                                                                          *
 *   Permission to use, copy, modify, and distribute this program           *
 *   for any purpose and without fee is hereby granted, provided            *
 *   that this copyright and permission notice appear on all copies         *
 *   and supporting documentation, the name of M.I.T. not be used           *
 *   in advertising or publicity pertaining to distribution of the          *
 *   program without specific prior permission, and notice be given         *
 *   in supporting documentation that copying and distribution is           *
 *   by permission of M.I.T.  M.I.T. makes no representations about         *
 *   the suitability of this software for any purpose.  It is pro-          *
 *   vided "as is" without express or implied warranty.		            *
 ****************************************************************************
 * $Header: /usr/wildcat/dfk/research/parallel/proteus/proteus-V3.01/engine/RCS/req_queue.c,v 1.2 94/01/24 00:39:24 dfk Time64bit Locker: dfk $
 * $Log:	req_queue.c,v $
 * Revision 1.2  94/01/24  00:39:24  dfk
 * Added Sanjay's support for 64-bit timers.
 * Added thread_sleep_until.
 * Fixed a few bugs in thread sleep/wakeup code (my bugs).
 * Fixed printf formats in many places.
 * 
 * Revision 1.1  94/01/23  11:35:18  dfk
 * Initial revision
 * 
 * Revision 1.6  92/12/09  15:19:35  brewer
 * updated calls to DisplaySimReq to use FILE *
 * 
 * Revision 1.5  92/05/08  13:35:28  brewer
 * Changed TRACE_ENQUEUE to check for past timestamps.
 * 
 * Revision 1.4  92/05/01  17:24:52  brewer
 * Added tracing code ifdef TRACE_ON
 * 
 * Revision 1.3  92/04/28  11:24:18  brewer
 * changed include file conf.h to conf.param
 * 
 * Revision 1.2  92/02/12  15:08:21  brewer
 * Added code to detect requests with past timestamps
 * 
 * Revision 1.1  92/02/11  13:56:08  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include <stdio.h>
#include "sim.h"
#include "conf.param"
#include "rt_thread_def.h"
#include "thread.h"
#include "simcalls.h"
#include "cache.param"

extern void DisplaySimReq(FILE *fp, SimRequest *);

#define MAX_INT 0x7fffffff

/* if NET and YESCACHE, we must ensure that cache protocol messages with
   the same timestamp are executed in FIFO order.  The PRESERVE_ORDER
   conditional specifies FIFO order; it overrides RANDOMREQ */
#ifdef YESCACHE
#ifdef NET
#define PRESERVE_ORDER
#endif
#endif


/***************************************************************************/
/* tracing for next_request_, normally off */

#define TRACE_ON

#ifdef TRACE_ON
GLOBAL int trace_req = FALSE;    /* GLOBAL => visible to dbx */

#define TRACE(req) \
  if (trace_req) { \
     printf("-----------------------------------------------------------\n");\
     DisplaySimReq(stdout, req); }
#else
#define TRACE(req)
#endif

/***************************************************************************/
/* tracing for enqueue_request_, normally off */

#ifdef TRACE_ON
GLOBAL int trace_enqueue = FALSE;

#define TRACE_ENQUEUE(req) \
  if (req->h.timestamp < GLOBALTIME) {\
      DisplaySimReq(stdout, req); \
      fatal("enqueue_request_: past timestamp!"); } \
  if (trace_enqueue) {\
     printf("enqueue_request_: "); \
     DisplaySimReq(stdout, req); }
#else
#define TRACE_ENQUEUE(req)
#endif


/***************************************************************************/


#ifdef PRESERVE_ORDER

/* use a queue to get FIFO order */
typedef struct hT {
    SimRequest *head;
    SimRequest *tail;
} hashType;
#define FIRST(x) (hash_table[x].head)

#else

/* do not need FIFO order => use simple linked list (LIFO) for speed */
typedef SimRequest *hashType;
#define FIRST(x) (hash_table[x])

#endif

#define SIZE   1024      /* must be a power of 2 */
#define MASK   0x3ff     /* must be SIZE - 1 */

#ifdef LONG_LONG_TIME
#define NOT_MASK 0xfffffffffffffc00LL
#else
#define NOT_MASK (~MASK)
#endif

static hashType hash_table[SIZE];
GLOBAL Time hash_time_ = 0;
GLOBAL int hash_index_ = 0;

/***************************************************************************/

/* Insert a request.  This is a speed-critical routine. */

void enqueue_request_(SimRequest *r)
{
#ifdef PRESERVE_ORDER  /* ignores RANDOMREQ */
    int index;

    TRACE_ENQUEUE(r);
    index = r->h.timestamp & MASK;
    if (hash_table[index].tail == NULL) {
	assert(hash_table[index].head == NULL);
	hash_table[index].head = hash_table[index].tail = r;
    } else {
	assert(hash_table[index].head != NULL);
	hash_table[index].tail->h.nextreq = r;
	hash_table[index].tail = r;
    }
    r->h.nextreq = NULL;

#else
#ifdef RANDOMREQ
    int index;

    TRACE_ENQUEUE(r);
    index = r->h.timestamp & MASK;
    if (hash_table[index] != NULL) {
	/* don't swap with snapshot requests (ruins determinism), see below */
	if (((SimRequest *)hash_table[index])->h.reqtype != SC_SNAPSHOT) {
	    if (random() & 0x01) {
		r->h.nextreq = ((SimRequest *)hash_table[index])->h.nextreq;
		((SimRequest *)hash_table[index])->h.nextreq = r;
		return;
	    }
	}
    }
    r->h.nextreq = hash_table[index];
    hash_table[index] = r;

#else /* normal case (fastest) */

    TRACE_ENQUEUE(r);
    r->h.nextreq = (SimRequest *) hash_table[r->h.timestamp & MASK];
    hash_table[r->h.timestamp & MASK] = r;

#endif
#endif
}

/***************************************************************************/

/* special enqueue for snapshot that does not call random
   (calling random ruins repeatability compared to simulation without the
   snapshot request) */
/* invariant: SC_SNAPSHOT requests are always at the end of the chain
   unless PRESERVE_ORDER is true, in which case random() is never called */
void enqueue_snapshot_(Time time)
{
    SimRequest *req;

    req = new_request_();
    req->h.timestamp = time;
    req->h.reqtype = SC_SNAPSHOT;
    req->h.tid = -1;
    req->h.nextreq = NULL;

#ifdef PRESERVE_ORDER
    enqueue_request_(req);
#else  /* insert at end of chain */
    {
	SimRequest *ptr;

	ptr = (SimRequest *)hash_table[time & MASK];
	if (ptr == NULL) {
	    hash_table[time & MASK] = req;
	} else {
	    while (ptr->h.nextreq != NULL) ptr = ptr->h.nextreq;
	    ptr->h.nextreq = req;
	}
    }
#endif

    printf("Snapshot scheduled at time %s.\n", time_print(time));
}

/***************************************************************************/

/* Extract request with a minimum timestamp. This is a speed-critical
   routine. This routine is written in a way that forces the compiler to
   generate quality assembly language.

   Returns NULL if the priority queue is empty. */

SimRequest *next_request_()
{
    Time time;
    SimRequest *req, **handle;
    hashType *h;
    short wrap_around = FALSE;

    for (h = hash_table + hash_index_;;) {
	/* find next slot */
	while (*((SimRequest **)h) == NULL) {
	    h++;
	    if (h == &hash_table[SIZE]) {
		if (!wrap_around) {
		    wrap_around = TRUE;
		    h = hash_table;
		    hash_time_ += SIZE;
		} else goto Findmin;
                /* Yes, I used a "goto"; it forces the compiler to generate
                   quality code.  Without it, the MIPS compiler pulls loop 
                   invariants out of the FindMin code.  Since the Findmin
		   code is almost never called, computing the invariants is
		   usually a waste of time.  This routine is called over
		   a million times just in 8-queens, so the goto does
		   make a difference. */
	    }
	}

	/* find position within slot */
	time = hash_time_ + (h - hash_table);  /* time for this slot */

	handle = (SimRequest **)h;

	do {
	    if ((*handle)->h.timestamp == time) {
		req = *handle;
		*handle = req->h.nextreq; /* remove request */
#ifdef PRESERVE_ORDER
		if (h->tail == req) {
		    if (h->head == NULL) {
			h->tail = NULL;
		    } else {
			/* h.nextreq at offset 12 bytes, verified by
			   init_request_queue_ */
			h->tail = (SimRequest *)(handle - 3);
		    }
		}
#endif
		hash_index_ = h - hash_table;
		TRACE(req);
		return(req);
	    }
	    handle = &((*handle)->h.nextreq);
	} while (*handle != NULL);

	/* if no valid requests found go to next slot */
	h++;
	if (h == &hash_table[SIZE]) {
	    if (!wrap_around) {
		wrap_around = TRUE;
		h = hash_table;
		hash_time_ += SIZE;
	    } else goto Findmin;  /* see above note about 'goto' */
	}
    }

 /* Findmin is called when the a complete loop through the table was unable
    to find a request.  This usually means that the table is empty (and thus
    the simulation is complete), but can also occur if the next request is
    thousands of cycles ahead of the current time.  This code searches the
    table for the request with the minimum timestamp; it returns NULL if
    the table is empty.  If it finds a request, it updates hash_time_ and
    hash_index_ and then returns the request */

 Findmin:
    time = (Time) -1;
    for (h = hash_table; h < &hash_table[SIZE]; h++) {
	for (req = *((SimRequest **)h); req != NULL; req = req->h.nextreq)
	  if (req->h.timestamp < time) time = req->h.timestamp;
    }
    if (time == -1) return(NULL);

    h = &hash_table[time & MASK];
    hash_time_ = time & NOT_MASK;

    handle = (SimRequest **)h;
    do {
	if ((*handle)->h.timestamp == time) {
	    req = *handle;
	    *handle = req->h.nextreq; /* remove request */
#ifdef PRESERVE_ORDER
	    if (h->head == NULL) {
		h->tail = NULL;
	    }
#endif
	    hash_index_ = h - hash_table;
	    TRACE(req);
	    return(req);
	}
	handle = &((*handle)->h.nextreq);
    } while (*handle != NULL);

    /* should never reach this point, since we know that a request with
       the desired timestamp resides in the slot */
    fatal("Internal inconsistency: end of next_request_() in req_queue.c\n");

    /* unreachable */
    return((SimRequest *) NULL);
}


/***************************************************************************\
* Generator routines: InitGenRequest_, NextGenRequest_                      *
\***************************************************************************/

/* generator for the current list of requests */

/* This code is NOT reentrant.  Only one generator can be active at a time. */

/* The generator is called from snapshot() when the user examines the
   request queue */

static int gen_index;
static Time gen_time;
static Time globaltime;
static SimRequest *gen_req = NULL;
SimRequest *NextGenRequest_();

/* initialize generator, return NULL if queue is empty */
SimRequest *InitGenRequest_()
{
    gen_index = hash_index_;
    gen_time = hash_time_;
    globaltime = GLOBALTIME;

#ifdef PRESERVE_ORDER
    gen_req = hash_table[gen_index].head;
#else
    gen_req = (SimRequest *)hash_table[gen_index];
#endif

    return(NextGenRequest_());
}

/* generate next request; return NULL if no more requests */
SimRequest *NextGenRequest_()
{
    Time time;
    SimRequest *req;
    short wrap_around;
    hashType *h;

    time =  gen_time + gen_index;  /* time for this slot */

    while (gen_req != NULL) {
	if (gen_req->h.timestamp == time) {
	    req = gen_req;
	    gen_req = req->h.nextreq;
	    return(req);
	} else if (gen_req->h.timestamp < globaltime) {
	    fprintf(stderr, "Error: found request with past timestamp:\n*** ");
	    DisplaySimReq(stderr, gen_req);
	}
	gen_req = gen_req->h.nextreq;
    }

    wrap_around = FALSE;

    for (;;) {
	/* find next slot */
	do {
	    gen_index++;
	    if (gen_index == SIZE) {
		if (wrap_around) {
		    time = (Time) -1;
		    for (h = hash_table; h < &hash_table[SIZE]; h++) {
			for (req = *((SimRequest **)h); req != NULL; req = req->h.nextreq)
			  if (req->h.timestamp < time  &&
			      req->h.timestamp > gen_time)
			    time = req->h.timestamp;
		    }
		    if (time == -1) return(NULL);  /* queue empty */

		    gen_index = time & MASK;  /* update to new time */
		    gen_time = time & NOT_MASK;
		    assert(FIRST(gen_index) != NULL);
		    wrap_around = FALSE;
		} else {
		    wrap_around = TRUE;
		    gen_index = 0;
		    gen_time += SIZE;
		}
	    }
	} while (FIRST(gen_index) == NULL);

	/* find position within slot */
	time =  gen_time + gen_index;  /* time for this slot */
	req = FIRST(gen_index);
	do {
	    if (req->h.timestamp == time) {
		gen_req = req->h.nextreq;
		return(req);
	    } else if (req->h.timestamp < globaltime) {
		fprintf(stderr,
			"Error: found request with past timestamp:\n*** ");
		DisplaySimReq(stderr, req);
	    }
	    req = req->h.nextreq;
	} while (req != NULL);
    }
}


/***************************************************************************\
* allocation/deallocation of request blocks                                 *
\***************************************************************************/

static SimRequest request_blocks[MAX_SIMREQS];
static SimRequest *free_list_ = NULL;

void init_request_queue_() 
{
    int i;

    request_blocks[0].h.nextreq = NULL;

    /* implementation assumes "nextreq" field is at offset 12 bytes; this
       assumption is verified here */
    if ((char *)(&request_blocks[0].h.nextreq) - (char *)(&request_blocks[0])
	!= 12)
      fatal("init_request_queue: nextreq field is not at offset 12 bytes\n");

    for (i=1; i<MAX_SIMREQS; i++) {
	request_blocks[i].h.nextreq = &request_blocks[i-1];
    }

    i = MAX_SIMREQS - 1;
    free_list_ = &request_blocks[i];

#ifdef PRESERVE_ORDER
    fprintf(stderr, "(order preserved)\n");
#endif
}

/* retrieve from free list; there is no dynamic memory allocation */
/* exits if there are no more request blocks */

SimRequest *new_request_() 
{
    SimRequest *new;

    if (free_list_ == NULL) {  /* fatal error, exit */
	fatal("out of simulator request blocks");
    }
    new = free_list_;
    free_list_ = free_list_->h.nextreq;
    return(new);
}

/* return request to free list */

void free_request_(request) 
SimRequest *request ;
{
    request->h.nextreq = free_list_;
    free_list_ = request;
}
