
 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: rt_thread.aux.c
 *
 * Description: Auxiliary non-cycle-counted runtime procedures
 *
 * Last Modified:  $Date: 94/01/24 00:39:29 $ (eab)
 *
 * Originally written by C. N. Dellarocas  5-91
 *
 * Global Functions:
 *     Word lock()
 *     void unlock(prev)
 *     void mark_as_killed (Thread *osptr)
 *     void _thread_yieldToScheduler()
 *     void ASSERT_ATOMIC()
 *     void ASSERT_NOT_ATOMIC()
 *     void begin_atomic()
 *     void end_atomic()
 *     void DisplayTask(FILE *fp, int stid)
 *
 * Global Variables: none
 *    
 * Referenced parameters:
 *     OS_QUANTUM, TIMER_PERIOD
 *
 ***************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.		
 ****************************************************************************
 * $Header: /usr/wildcat/dfk/research/parallel/proteus/proteus-V3.01/engine/RCS/rt_thread.aux.c,v 1.3 94/01/24 00:39:29 dfk Time64bit Locker: dfk $
 * $Log:	rt_thread.aux.c,v $
 * Revision 1.3  94/01/24  00:39:29  dfk
 * Added Sanjay's support for 64-bit timers.
 * Added thread_sleep_until.
 * Fixed a few bugs in thread sleep/wakeup code (my bugs).
 * Fixed printf formats in many places.
 * 
 * Revision 1.2  93/02/28  11:19:47  dfk
 * >  * changed the sleeping-thread system to be much more efficient.
 * >  * If a processor is idle, it will skip ahead to the wakeup time
 * >  * for the first sleeping thread.
 * >  * It runs MUCH faster than spinning on the TIMER_PERIOD method.
 * >  * Questions:
 * >  *   how does it interact with preemptive scheduling? It should work.
 * >  *   how does it interact with semaphores, which also use timer requests
 * >  *   does it work with multiple sleeping threads? it should
 * >  *   does it work when a new thread is creating on a sleeping processor? should.
 * 
 * Revision 1.1  93/02/28  10:22:56  dfk
 * Initial revision
 * 
 * Revision 1.6  92/12/09  15:17:27  brewer
 * updated calls to trace_thread_
 * 
 * Revision 1.5  92/12/07  13:55:09  brewer
 * (adj) added Emerald support, added code to prevent quantum expiration
 * across calls to sched_choose
 * 
 * Revision 1.4  92/11/12  12:53:47  brewer
 * Changed T_INTERRUPTED from state to flag; see rt_thread.ca log for details.
 * Also imported DisplayTask from snapshot.c and added ThreadStateName.
 * 
 * Revision 1.3  92/11/08  14:49:31  brewer
 * Added ifdef PREEMPT code (adj)
 * 
 * Revision 1.2  92/02/21  17:59:28  sharon
 * *** empty log message ***
 * 
 * Revision 1.1  92/02/11  13:56:10  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include "user.h"
#include "conf.h"
#include "simreq.h"
#include "processor.h"
#include "simcalls.h"
#include "event.h"
#include "mem.h"  /* for stack routines */

#include "monitorswitch.h"  /* SEP */
#include "perfspec.h"
#ifdef MONITORING_ON  /* SEP */
#include "rt_rqtime.h"
#include "interrupt.param"
#endif

extern threadPrivate *PP;

#ifdef EMERALD
extern void context_switch_hook(int oldtid, int newtid);
extern int emerald;
#endif

/* THESE ROUTINES ARE *NOT* CYCLE-COUNTED */

/* Disable all interrupts. Returns previous interrupt state */
Word lock()
{
    Word previ = currosptr->t_intrflag;
    Word prevp = currosptr->t_preempt;
    currosptr->t_intrflag = FALSE;
    currosptr->t_preempt = FALSE;
    
    return( ((previ << 16) & 0xffff0000) | (prevp & 0x0000ffff) );
}


/* Restore interrupt status to prev */
void unlock(prev)
Word prev;
{
    currosptr->t_preempt = prev & 0x0000ffff;
    currosptr->t_intrflag = (prev >> 16) & 0x0000ffff;
}

void ASSERT_ATOMIC()
{
    if (currosptr->t_intrflag || currosptr->t_preempt)
	fatal("ASSERT_ATOMIC called with interrupts and/or preemption on");
}


void ASSERT_NOT_ATOMIC()
{
    if (!(currosptr->t_intrflag || currosptr->t_preempt))
       fatal("ASSERT_NOT_ATOMIC called with interrupts and/or preemption off");
}


void begin_atomic()
{
    if (!(currosptr->t_intrflag && currosptr->t_preempt))
	fatal("begin_atomic called with interrupt and/or preemption off");
    currosptr->t_intrflag = FALSE;
    currosptr->t_preempt = FALSE;

    /* SEP */
    PSPEC_STARTOP(Interrupts_DisableInterrupts, currosptr->t_processor);
    PSPEC_STARTTIMER(Interrupts_InterruptsDisabled, currosptr->t_tid);
    PSPEC_RECPARAM(Interrupts_BeginTimeStamp, currosptr->t_processor,
		   CURR_TIME);
    PSPEC_RECPARAM(Interrupts_ThreadID, currosptr->t_processor, currtid);
    PSPEC_RECPARAM(Interrupts_Processor, currosptr->t_processor,
		   currosptr->t_processor);
}


void end_atomic()
{
    if (currosptr->t_intrflag || currosptr->t_preempt)
	fatal("end_atomic called with interrupt and/or preemption on");
    currosptr->t_intrflag = TRUE;
    currosptr->t_preempt = TRUE;

    /* SEP */
    PSPEC_ENDTIMER(Interrupts_InterruptsDisabled, currosptr->t_tid);
    PSPEC_ENDOP(Interrupts_DisableInterrupts, currosptr->t_processor);
}


/**************************************************************************/

void mark_as_killed(osptr) 
Thread *osptr;
{
    ThreadInfo *tptr;
    
    /* BEGIN ATOMIC SECTION */
    
    tptr = &thread_table_[osptr->t_stid];

    if( freestk( tptr->t_stkblk ) == ERROR ) {
	  fatal("thread_destroy failed: Cannot freestk for tid %d:%d", 
		CURR_PROCESSOR, osptr->t_tid);
    }
    osptr->t_state = T_FREE;
    release_simulator_tid(osptr->t_stid);	
    numthreads--;	
#ifdef MULT_RL
    proc_table_[osptr->t_processor].p_numthreads--;
#endif
    
    /* END ATOMIC SECTION */
}



/* Not-cycle-counted version of thread_yieldToScheduler (see rt_thread.ca) */

void _thread_yieldToScheduler()
{
    Time t;
    Thread *newt;
    Thread *sched_choose();
    
    SubtractTime(10000);
    newt = sched_choose();
    AddTime(10000);

    if ( newt == NULL ) {
	if (NotIdle(CURR_PROCESSOR)) {
	    MakeIdle(CURR_PROCESSOR);
	    t = CURRENTTIME;
	    EVENT_IDLE(t, CURR_PROCESSOR);
#ifdef EMERALD
	    if (emerald) {
	      SubtractTime(10000);
	      context_switch_hook(currosptr->t_stid, -1);
	      AddTime(10000);
	    }
#endif
#ifdef PREEMPT
	    if (currpptr->p_numthreads > 0)
	      make_timer_request_(CURR_PROCESSOR, CURR_TID, 
						 CURR_TIME + ReadTimer(CURR_PROCESSOR));
#else
	    if (PP->sleep_nonempty) {
		   Time wakeup_time = *(PP->sleep_head); /* first to awake */
		   if (CURR_TIME+10 > wakeup_time) /* avoid race condition */
		     wakeup_time = CURR_TIME+10;
		   make_timer_request_(CURR_PROCESSOR, CURR_TID, wakeup_time);
	    }
#endif
	} else printf("*** Warning: at point 1 in rt_thread.aux.c; proc=%d\n",
		      CURR_PROCESSOR);
	TrapToSimulator();
    }
    else {
	currosptr->t_state = T_READY;
	newt->t_state = T_CURRENT;
	newt->t_processor = CURR_PROCESSOR;
	PP->quantum = OS_QUANTUM;
	WriteTimer(CURR_PROCESSOR, (Time) TIMER_PERIOD);    
	currpptr->p_tid = newt->t_stid;
#ifdef EMERALD
	if (emerald) {
	  SubtractTime(10000);
	  context_switch_hook(currosptr->t_stid, -1);
	  AddTime(10000);
	}
#endif
	
#ifdef MONITORING_ON /* SEP */
        newt->t_rqtime += CURRENTTIME - newt->t_rqstart; 
	newt->t_rqcount += 1; 
	if ((CURRENTTIME - newt->t_rqstart) == 0) newt->t_rqzeros += 1; 
#endif
	MAKE_RESUME_REQUEST(newt->t_stid, CURRENTTIME, OK);
	
	TrapToSimulator();
    }
}


/*************************************************************************\
* DisplayTask -- output thread data
\*************************************************************************/

static const char *ThreadStateName(int state)
{
    if (state & T_ACTIVE) {
	if (state & T_INTERRUPTED)
	  switch (state & 0777) {
	    case T_CURRENT: return("Interrupted (was CURRENT)");
	    case T_READY: return("Interrupted (was READY)");
	    case T_SPINNING: return("Interrupted (was SPINNING)");
	    case T_SUSPENDED: return("Interrupted (was SUSPENDED)");
	    case T_SLEEPING: return("Interrupted (was SLEEPING)");
	  }
	else
	  switch (state & 0777) {
	    case T_CURRENT: return("CURRENT");
	    case T_READY: return("READY");
	    case T_SPINNING: return("SPINNING");
	    case T_SUSPENDED: return("SUSPENDED");
	    case T_SLEEPING: return("SLEEPING");
	  }
    } else switch (state & 03000) {
	case T_FREE: return("INACTIVE");
	case T_FORWARDER: return("FORWARDER");
	case T_RESERVED: return("RESERVED");
    }
    return("Illegal!");
}


GLOBAL void DisplayTask(FILE *fp, int stid) 
{
    ThreadInfo *tptr;
    Thread *osptr;
    int proc, rtid;

    if (Invalidstid(stid)) {
	fprintf(fp, "Invalid thread id: %d.\n", stid);
	return;
    }
    
    tptr = &thread_table_[stid];
    osptr = T_OSBLOCK(stid);

    if (osptr == NULL) {
	fprintf(fp, "NO osptr for thread #%d\n", stid);
	return;
    }

    proc = osptr->t_processor;
    rtid = osptr->t_tid;

    fprintf(fp, "#%d = %d:%d (%s) in state %s on proc %d\n",
	    stid, proc, rtid, tptr->t_name,
	    ThreadStateName(osptr->t_state), proc);

    switch(osptr->t_state & ~T_INTERRUPTED) {
      case T_FREE:
      case T_RESERVED:
	break;
      case T_CURRENT:
	if (proc_table_[osptr->t_processor].p_ihandleractive)
	  fprintf(fp, "\t(Servicing interrupts)\n");
	/* fall through to T_READY */
      case T_READY:
      case T_SUSPENDED:
      case T_SPINNING:
      case T_SLEEPING:
	fprintf(fp, "\tPriority: %lu, PRE-EMPTION %s, INTERRUPTS %s\n", 
		osptr->priority, 
		osptr->t_preempt ? "ON" : "OFF",
		osptr->t_intrflag ? "ON" : "OFF");
	trace_thread_(fp, stid);
	break;
      case T_FORWARDER:
	fprintf(fp, "\t(forwarding to #%d = %d:%d, since time %s)\n",
		((Thread *)osptr->t_forwardaddr)->t_stid,
		osptr->t_forwardproc,
		((Thread *)osptr->t_forwardaddr)->t_tid,
		time_print(tptr->t_time));
	break;
      default:
	break;
    }
}

