 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: rt_thread_def.h
 *
 * Description: Defines all core runtime thread routines
 *
 * Last Modified:  $Date: 94/04/05 14:04:19 $ ($Author)
 *
 * Data Structures:
 *      Thread
 *      threadPrivate
 *
 * Constants:
 *     
 * Macros:
 *     OSBLOCK(runtime_tid)            converts tid to Thread *
 *     T_PROCESSOR(runtime_tid)        gets processor for thread
 *     T_PRIORITY(runtime_tid)         gets priority for thread
 *     MY_TID                          runtime tid for "self"
 *     MY_STID                         simulator tid for "self"
 *     Invalidtid(runtime_tid)
 *     Invalidprio(p)
 *     
 * Referenced parameters: none
 *     MAX_PROC_THREADS, BLOCK_JOIN, SPIN_JOIN, NO_OF_PROCESSORS
 *
 ***************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.		
 ***************************************************************************
 * $Header: /usr/wildcat/dfk/research/parallel/proteus/proteus-V3.01/engine/RCS/rt_thread_def.h,v 1.3 94/04/05 14:04:19 dfk Exp $
 * $Log:	rt_thread_def.h,v $
 * Revision 1.3  94/04/05  14:04:19  dfk
 * added MY_STID macro
 * 
 * Revision 1.2  93/02/28  11:19:50  dfk
 * >  * changed the sleeping-thread system to be much more efficient.
 * >  * If a processor is idle, it will skip ahead to the wakeup time
 * >  * for the first sleeping thread.
 * >  * It runs MUCH faster than spinning on the TIMER_PERIOD method.
 * >  * Questions:
 * >  *   how does it interact with preemptive scheduling? It should work.
 * >  *   how does it interact with semaphores, which also use timer requests
 * >  *   does it work with multiple sleeping threads? it should
 * >  *   does it work when a new thread is creating on a sleeping processor? should.
 * 
 * Revision 1.1  93/02/28  10:22:59  dfk
 * Initial revision
 * 
 * Revision 1.6  92/11/25  17:48:57  brewer
 * Anthony added field for stackpointer.
 * 
 * Revision 1.5  92/11/08  14:57:11  brewer
 * Changed sleep_head field from short to int.
 * 
 * Revision 1.4  92/09/24  12:04:24  brewer
 * Rearranged fields of Thread per adj's request.
 * Added ifndef at the top
 * 
 * Revision 1.3  92/09/23  12:22:29  brewer
 * Added debugging fields to Thread
 * 
 * Revision 1.2  92/02/21  18:01:25  sharon
 * Added rqtime fields to thread structure for performance specification 
 * monitoring
 * 
 * Revision 1.1  92/02/11  14:19:55  brewer
 * Initial revision
 * 
 \**************************************************************************/

#ifndef _RT_THREAD_DEF_
#define _RT_THREAD_DEF_

#include "OS.param"

#include "monitorswitch.h"  /* SEP */

#ifndef BLOCK_JOIN
#define SPIN_JOIN
#endif


/* Runtime system structure kept for each thread */
/* WARNING: Do NOT change this structure without first talking to Eric or
            Anthony. Certain versions of augment and the libraries are
	    dependent upon the offsets of the fields in the Thread data 
	    structure. Only NON-export versions are affected. */

typedef struct rt_thread {
  /* Operating system support */
  int     t_state;
  int     t_processor;
  Word    priority;
  Word    sched_node;
  int     t_tid;             /* runtime tid for this thread (self) */
  int     t_stid;            /* simulator tid for this thread */
  BOOL    t_preempt;
  BOOL    t_intrflag;        /* enable/disable interrupts     */
  int     t_forwardproc;     /* new processor                 */
  Word    t_forwardaddr;     /* thread addr on new processor  */
  Word    t_rip;	     /* Old RIP (used by fixup blocks) */
  Word    t_ripaddr;	     /* Old RIP Address (used by fixup blocks) */
  Word    t_regs[32];	     /* Thread registers (used by fixup blocks) */
  Word    t_fregs[32];	     /* Thread fp registers (used by fixup blocks) */
  BOOL	  t_ispointer;	     /* Proc returns ptr value? (used by fixup) */
  Word	  t_stackmarker;     /* Stack marker (used for Emerald simulations) */
#ifdef MONITORING_ON  /* SEP */
  ulong   t_rqstart;         /* time thread last became ready to run */
  ulong   t_rqtime;          /* accumulated time spent ready to run */
  ulong   t_rqcount;         /* number of times thread became ready to run */
  ulong   t_rqzeros;         /* number of measured zeros for rqtime */
#endif
} Thread;

#define OSBLOCK(t)          (&PP->thread_table[t])

#define T_PROCESSOR(t)      (OSBLOCK(t)->t_processor)
#define T_PRIORITY(t)       (OSBLOCK(t)->t_priority)
#define T_STATE(t)	    		(OSBLOCK(t)->t_state) /* DFK */
#define NO_TID	 	    		(-1)	/* DFK; an invalid tid number */

#define Invalidtid(t)       (( (t)<0 )||( (t)>=MAX_PROC_THREADS ))
#define Invalidprio(p)      (0)

#define MY_TID              (currosptr->t_tid)
#define MY_STID             (currosptr->t_stid)


/* Structure of kernel data that is private to each processor */
/* In the following routines it is accesses by PP-><field>    */
/* where PP is a pointer that points to the base of the       */
/* private structure of the current processor                 */

typedef struct private {
  Thread thread_table[MAX_PROC_THREADS];   /* processor thread table      */
  int nextthread;                          /* possible next available tid */
  BOOL sleep_nonempty;                     /* is the sleep queue nonempty?*/
  Time *sleep_head;                        /* ptr to head of sleep queue  */
  int clockq;                              /* index of clock queue head   */
  int quantum;                             /* timer quantum               */
  /* Add here other fields private to each processor */
} threadPrivate;


/* Pointer to processor's private data structure */
extern threadPrivate *PP;
extern threadPrivate processor_private[NO_OF_PROCESSORS];



/***************************************************************************\
* scheduler procedures
\***************************************************************************/

void sched_initialize(void);
void sched_insert(Thread *t);
void sched_remove(Thread *t);
void sched_change_priority(Thread *t, long delta);
Thread *sched_choose(void);

#endif /* ifndef _RT_THREAD_DEF_ */
