
/****************************************************************************
 *                                                                          *
 *                     Parallel Architecture Simulator                      *
 *                Eric A. Brewer  and  Chris N. Dellarocas                  *
 *                     Laboratory for Computer Science                      *
 *                  Massachusetts Institute of Technology                   *
 *                                                                          *
 * Module: schedule.single.c
 *                                                                          *
 * Description:                                                             *
 *                                                                          *
 * Last Modified:
 *                                                                          *
 * Global Functions:                                                        *
 *                                                                          *
 * Global Variables:                                                        *
 *                                                                          *
 ****************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.
 ****************************************************************************
 * $Header: /psg/proteus/RCS/schedule.single.c,v 1.2 92/11/12 12:58:03 brewer Exp $
 * $Log:	schedule.single.c,v $
 * Revision 1.2  92/11/12  12:58:03  brewer
 * Changed T_INTERRUPTED from state to flag -- see rt_thread.ca log for
 * details.
 * 
 * Revision 1.1  92/02/11  13:56:14  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include "processor.h"
#include "cache.h"
#include "shmem.h"
#include "sema.h"
#include "simcalls.h"
#include "thread.h"
#include "q.h"

#include "event.h"

Time reschedule();

InsertInReadyList ( tid, priority) 
int tid ;
int priority ;
{
  insert(tid, ready_head, priority);
  ready_threads++;

  if ( idle_processors > 0 )
    {
      Time newtime = reschedule(idle_head, NULL);

      if (sim_running_) { 
	EVENT_BUSY(newtime, idle_head);
	RecordProcessorUtilization(newtime, idle_head, ON);
      }

      idle_head = processor[idle_head].p_nextidle;
      idle_processors--;
    }
}


/* Reschedule processor p. time updates current timestamp of processor p.
 * If time = 0 timestamp is not updated and reschedule is considered a
 * zero-latency operation.
 */

Time reschedule ( p, dummy, time) 
int p ;
int dummy ;
Time time ;
{
  ProcBlk *pptr;
  OS_ThreadInfo *optr, *nptr;
  int newtid;
  Time newtime;


  pptr = &processor[p];
  pptr->p_time = max(pptr->p_time, time);

  if ( NotIdle(p) &&
          (optr = OSBLOCK(pptr->p_tid)) != NULL &&
          (optr->t_state == T_CURRENT 
	   || optr->t_state & T_INTERRUPTED
	   || optr->t_state == T_SPINNING
	   || optr->t_state == T_SLEEPING) ) {

    /* If old thread was spinning on a lock, remove it from lock's queue */
    if (optr->t_state == T_SPINNING) {
      dequeue(pptr->p_tid);
      semaphore[optr->t_spinlock].threadswaiting--;
    }

    /* If current thread still has the highest priority, let it run for
     * another quantum
     */
    if ( optr->t_priority > lastkey(ready_tail) )  {
      newtid = pptr->p_tid;
      nptr = optr;
    } else {
      /* Otherwise replace it with the ready thread with highest priority */
      optr->t_state = T_READY;
      insert(pptr->p_tid, ready_head, optr->t_priority);
      nptr = OSBLOCK(newtid = getlast(ready_tail));
    }
  }
  else { /* If p is idle or its current thread has just been killed */
    if (ready_threads > 0) {
      /* If there are some ready threads, pick the one
       * with highest priority 
       */
      nptr = OSBLOCK(newtid = getlast(ready_tail));
      ready_threads--;
    } else {
      /* If there are no ready threads, mark p an idle processor */
      if (NotIdle(p) && intr_pending[p] == 0 && pptr->p_time > 0) {
	EVENT_IDLE(pptr->p_time, p);
	RecordProcessorUtilization(pptr->p_time, p, OFF);
      }
      
      if (NotIdle(p)) 
	if (intr_pending[p] > 0) {
	  pptr->p_tid = pptr->p_ihandler;
	  pptr->p_interrupted = ERROR;
	  OSBLOCK(pptr->p_ihandler)->t_state = T_CURRENT;
	  MAKE_RESUME_REQUEST(pptr->p_ihandler, pptr->p_time, NULL);
	}
	else
	  MakeIdle(p);

      return;
    }
  }

  nptr->t_state = T_CURRENT;
  nptr->t_processor = p;
  WriteTimer(p, OS_QUANTUM);

  pptr->p_tid = newtid;
  newtime =  max(thread_table_[newtid].t_time, pptr->p_time)
    + (time>0 ? RESCHEDULE_LATENCY : 0);
  /* Does not count reschedule latency if processor was idle */

#ifdef THREADDEBUG
  printf("Reschedule %d: Newtid = %d Time = %ld\n", p, newtid, newtime);
#endif

  /* Thread's resume time is processor time */

  MAKE_RESUME_REQUEST(newtid, newtime, OK);

  return( newtime );
}
