
 /****************************************************************************
 *
 *                     Parallel Architecture Simulator         
 *                Eric A. Brewer  and  Chris N. Dellarocas     
 *                     Laboratory for Computer Science         
 *                  Massachusetts Institute of Technology      
 *
 * Module: sema.c
 *
 * Description:  basic semaphore code
 *
 * Last Modified:
 *
 * Global Functions:
 *     void init_semaphores(void)
 *     Sem sem_open(Word value, const char *name, int processor)
 *     int sem_close(Sem addr) 
 *     SemBlk *GetSemaphoreData_(char *shared_address);
 *     void DisplayLock(int lock, FILE *out);
 *     void DisplayActiveLocks(FILE *out);
 *
 * Global Variables:
 *     SemBlk semaphore[MAX_SEMAPHORES];
 *     
 ****************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.
 ****************************************************************************
 * $Header: /usr/wildcat/dfk/research/dmcache/src/proteus/RCS/sema.c,v 1.3 1996/05/13 20:33:20 dfk Exp $
 * $Log: sema.c,v $
 * Revision 1.3  1996/05/13 20:33:20  dfk
 * fixed precedence error
 *
 * Revision 1.2  1994/01/24 00:39:34  dfk
 * Added Sanjay's support for 64-bit timers.
 * Added thread_sleep_until.
 * Fixed a few bugs in thread sleep/wakeup code (my bugs).
 * Fixed printf formats in many places.
 *
 * Revision 1.1  94/01/23  15:57:54  dfk
 * Initial revision
 * 
 * Revision 1.5  92/11/19  13:17:53  brewer
 * Eliminated gcc -Wall -O bogus warnings for uninitialized var sptr.
 * 
 * Revision 1.4  92/11/03  14:28:03  brewer
 * Imported DisplayLock and DisplayActiveLocks from snapshot.c
 * Improved error handling and robustness
 * 
 * Revision 1.3  92/04/01  17:03:23  brewer
 * cleaned up gcc warnings, added ANSI prototypes
 * 
 * Revision 1.2  92/03/31  16:47:24  brewer
 * Added Sem typedef (defined in prototypes.h)
 * Changed shared memory calls to use void * addresses
 * 
 * Revision 1.1  92/02/11  13:56:15  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include "user.h"
#include "simcalls.h"
#include "processor.h" /* for InvalidProc */
#include "q.h"
#include "event.h"

static int nextsema;  /* index of next free semaphore */
GLOBAL SemBlk semaphore[MAX_SEMAPHORES];


GLOBAL void init_semaphores() 
{
    int i;
    SemBlk *sptr;
    
    for(i=1; i<MAX_SEMAPHORES; i++) {
	(sptr = &semaphore[i])->state = S_FREE;
	sptr->sqtail = 1 + (sptr->sqhead = newqueue() );
    }

    nextsema = 1;
}


/* newsid -- allocate an unused semaphore */

static int newsid() 
{
    int sid, i;
    
    for(i=0; i < MAX_SEMAPHORES; i++) {
	sid = nextsema++;
	if (nextsema == MAX_SEMAPHORES)
	  nextsema = 0;
	if (semaphore[sid].state == S_FREE)
	  return(sid);
    }
    
    fatal("newsid() failed: out of semaphores -- increase MAX_SEMAPHORES");
    return(ERROR); /* UNREACHABLE */
}


/* Get a free semaphore and set it to an initial value */

GLOBAL Sem sem_open(Word value, const char *name, int processor) 
{
    Word sid;
    void *saddr;
    SemBlk *sptr;
    int i;
    
    if ( InvalidProc(processor) )
      fatal("sem_open failed: invalid processor %d\n", processor);
    
    sid = newsid();
    
    sptr = &semaphore[sid];
    
    for( i=0; i<S_NAMELEN && (sptr->name[i] = name[i]); i++);
    if (sptr->name[i] != 0) {
	sptr->name[i] = 0;
	fprintf(stderr,
		"Warning semaphore name `%s' shortened to `%s'.\n",
		name, sptr->name);
    }

    sptr->state = S_ACTIVE;
    sptr->processor = processor;
    sptr->threadswaiting = 0;
    sptr->tid = -1;
    saddr = &memory[sid*LINE_SIZE];
    *(char *)saddr = value;
    memtag[sid*LINE_SIZE] = value == 0 ? FULL : EMPTY;
    
#ifdef SEMDEBUG
    printf("#%d Opening semaphore %d (%s) value = %d tag = %d time =%d\n", 
	   currtid, sid, name, value, memtag[sid*LINE_SIZE], CURRENTTIME);
#endif
    
    return((Sem)saddr);
}



/* Free a semaphore that is no LONGER NEEDED -- NO THREAD MUST BE
 * CURRENTLY WAITING ON IT
 */

GLOBAL int sem_close(Sem addr) 
{
    SemBlk *sptr;
    Word sid = AddrToSid(addr);
    
    if ( InvalidSem(sid) || (sptr = &semaphore[sid])->state == S_FREE) {
	fatal("sem_close: invalid semaphore number %d", sid);
	return(0); /* unreachable, but keeps compiler happy */
    }
    
    if (sptr->threadswaiting > 0)
      fatal("sem_close: there are threads waiting on semaphore %d", sid);
    
    sptr->state = S_FREE;
    return(OK);
}


GLOBAL SemBlk *GetSemaphoreData_(char *adr)
{
    Word sid;

    if (adr < memory || adr >= top_of_memory)
      fatal("GetSemaphoreData_(0x%lx): expecting address in shared memory");

    sid = AddrToSid(adr);
    if (InvalidSem(sid)) return(NULL);

    if ((((ulong)adr) & LINE_MASK) != 0) /* should point to start of cache line*/
      fprintf(stderr,
       "Warning: GetSemaphoreData_: arg points into middle of semaphore %s\n",
	      semaphore[sid].name);
    return(&semaphore[sid]);
}


GLOBAL void DisplayLock(int i, FILE *fp) 
{
    SemBlk *sptr;
    int start, end, tid;
    char tmp[40];

    if (InvalidSem(i)) {
	fprintf(fp, "Semaphore %d out of range, must be in [0..%d)\n",
		i, MAX_SEMAPHORES);
	return;
    }

    sptr = &semaphore[i];
    if (sptr->state != S_ACTIVE) {
	fprintf(fp, "Semaphore %d is inactive\n", i);
	return;
    }

    if (sptr->tid >= 0)
      sprintf(tmp, "locked by thread #%d", sptr->tid);
    else
      sprintf(tmp, "unlocked");

    fprintf(fp, 
	    "Semaphore %d (%s) is %s\n", i, sptr->name, tmp);
    fprintf(fp,"\tLocation: %d (0x%lx)\t Value: %d:%s\t Threads Waiting: %d\n",
	    sptr->processor, (ulong)&memory[i*LINE_SIZE], memory[i*LINE_SIZE], 
	    memtag[i*LINE_SIZE] == FULL ? "FULL" : "EMPTY", 
	    sptr->threadswaiting);
    
    start = sptr->sqhead;
    end   = sptr->sqtail;
    
    tid = q[start].qnext;
    
    if(tid != end) {
	fprintf(fp, "Spinlock queue contents: ");
	
	while(tid != end) {
	    if (Invalidstid(tid) ) {
		fprintf(fp, 
			"Spinlock Queue corrupted. Invalid index %d.\n",
			tid);
		break;
	    }
	    fprintf(fp, "#%d ", tid);
	    tid = q[tid].qnext;
	}
	fprintf(fp, "\n");
    }
}

GLOBAL void DisplayActiveLocks(FILE *fp)
{
    int i;
    
    fprintf(fp,"\n STATE OF SPINLOCKS AT TIME %s.\n", time_print(GLOBALTIME));
    fprintf(fp,"\n Active Locks:\n\n");
    for(i=0; i<MAX_SEMAPHORES; i++) {
	if ( semaphore[i].state == S_ACTIVE )
	  DisplayLock(i, fp);
    }
}


#ifdef SEMA_FEBIT
#include "sema.febit.c"
#endif

#ifdef SEMA_TTSET
#include "sema.ttset.c"
#endif
