 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: sema.h -- Header file for semaphores
 *
 * Description: Constants, Structures and Variables for semaphores
 *
 * Last Modified: 8-26-91 (eab)
 *
 * Data Structures:
 *     SemBlk                    Basic semaphore data structure
 *
 * Constants:
 *     S_FREE, S_ACTIVE          Semaphore states
 *     SQ_NULL                   Marks the end of a semaphore queue
 *     SEMA_FRACTION
 *     S_NAMELEN                 Max length for semaphore name
 *
 * Referenced parameters:
 *     MAX_SEMAPHORES
 *
 * Macros:
 *     InvalidSem(s)
 *     AddrToSid(a)
 *     EVENT_P(a)                Generate sem_P event
 *     EVENT_V(a)                Generate sem_V event
 *
 ****************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.
 ***************************************************************************
 * $Header: /usr/wildcat/dfk/research/parallel/proteus/proteus-V3.01/engine/RCS/sema.h,v 1.2 94/01/27 01:20:23 dfk Time64bit $
 * $Log:	sema.h,v $
 * Revision 1.2  94/01/27  01:20:23  dfk
 * Needed to change floats to doubles
 * to workaround a gcc bug where you can't do arithmetic
 * that mixes floats with long longs
 * 
 * Revision 1.1  94/01/27  01:03:15  dfk
 * Initial revision
 * 
 * Revision 1.2  92/11/03  14:28:54  brewer
 * added tid field to SemBlk
 * added prototypes for GetSemaphoreData_, DisplayLock, DisplayActiveLocks
 * 
 * Revision 1.1  92/02/11  14:19:57  brewer
 * Initial revision
 * 
 \**************************************************************************/

#ifndef SEMA_H
#define SEMA_H

#include "shmem.h"  /* for MAX_SEMAPHORES */

#define S_FREE                   0
#define S_ACTIVE                 1

#define SQ_NULL                  -1  /* Marks end of a semaphore queue */

#define SEMA_FRACTION            ((double)0.50)

#define S_NAMELEN                10

typedef struct {
    char  name[S_NAMELEN]; /* symbolic name, for debugging and stats */
    char  state;           /* S_FREE or S_ACTIVE */
    int   processor;       /* memory module where semaphore resides */
    int   threadswaiting;  /* How many threads are waiting */
    int   sqhead;          /* Head of semaphore queue in q table */
    int   sqtail;          /* Tail of semaphore queue in q table */
    int   tid;             /* tid of holder */
} SemBlk;


extern SemBlk semaphore[MAX_SEMAPHORES];
extern SemBlk *GetSemaphoreData_(char *shared_address);
extern void DisplayLock(int lock, FILE *out);
extern void DisplayActiveLocks(FILE *out);

#define InvalidSem(s)           (s<1 || s>MAX_SEMAPHORES-1)

#define AddrToSid(a)            ((int)(((char *)(a)) - memory)/(LINE_SIZE))

#define EVENT_P(a)              EVENT_SEM_P(AddrToSid(a), CURRENTTIME)
#define EVENT_V(a)              EVENT_SEM_V(AddrToSid(a), CURRENTTIME)


#endif /* ifndef SEMA_H */
