
/****************************************************************************
 *                                                                          *
 *                     Parallel Architecture Simulator     
 *                Eric A. Brewer  and  Chris N. Dellarocas 
 *                     Laboratory for Computer Science     
 *                  Massachusetts Institute of Technology  
 *
 * Module: sema.ttset.c
 *
 * Description: test and test-and-set semaphore implementation
 *
 * Last Modified: $Date: 92/11/19 13:18:36 $
 *
 * Global Functions:
 *     int sem_P(Sem addr);
 *     int sem_V(Sem addr);
 *
 * Global Variables: none
 *
 ****************************************************************************
 *   Copyright 1991                                                         *
 *   Eric A. Brewer  and  Chris N. Dellarocas                               *
 *   Massachusetts Institute of Technology                                  *
 *                                                                          *
 *   Permission to use, copy, modify, and distribute this program           *
 *   for any purpose and without fee is hereby granted, provided            *
 *   that this copyright and permission notice appear on all copies         *
 *   and supporting documentation, the name of M.I.T. not be used           *
 *   in advertising or publicity pertaining to distribution of the          *
 *   program without specific prior permission, and notice be given         *
 *   in supporting documentation that copying and distribution is           *
 *   by permission of M.I.T.  M.I.T. makes no representations about         *
 *   the suitability of this software for any purpose.  It is pro-          *
 *   vided "as is" without express or implied warranty.		            *
 ****************************************************************************
 * $Header: /psg/proteus/RCS/sema.ttset.c,v 1.4 92/11/19 13:18:36 brewer Exp $
 * $Log:	sema.ttset.c,v $
 * Revision 1.4  92/11/19  13:18:36  brewer
 *  Eliminated gcc -Wall -O bogus warnings for uninitialized var sptr.
 * 
 * Revision 1.3  92/11/03  14:29:42  brewer
 * Track tid of lock holder
 * improved warnings and added sanity checks
 * 
 * Revision 1.2  92/04/01  17:03:33  brewer
 * cleaned up gcc warnings, added ANSI prototypes
 * 
 * Revision 1.1  92/02/11  13:56:19  brewer
 * Initial revision
 * 
 \**************************************************************************/

/* #define SEMDEBUG */

#ifdef NOBUSY

GLOBAL int sem_P(Sem addr) 
{
    SemBlk *sptr;
    Word sid = AddrToSid(addr);
    
#ifdef SEMDEBUG
    int semval = *(char *)addr;
#endif
    
    EVENT_SEM_P(sid, CURRENTTIME);
    
    if (InvalidSem(sid) || (sptr = &semaphore[sid])->state == S_FREE) {
	EVENT_SEM_V(sid, CURRENTTIME);
	fatal("sem_P failed: Invalid semaphore %d", sid);
    }
    
    for(;;) {
	if ( (int)Shared_Memory_Read((void *)addr, BYTE) > 0)
	  if (Test_and_Set((void *)addr) > 0) break;
	
	/* Place current thread in semaphore queue and trap to simulator */
	enqueue(CURR_TID, sptr->sqtail);
	sptr->threadswaiting++;
#ifdef SEMDEBUG
	printf("#%d P(%d = %s)= %d threadswaiting++ = %d\n",
	       CURR_TID, sid, sptr->name, semval, sptr->threadswaiting);
#endif
	
	currosptr->t_state = T_SPINNING;
	
	if( currosptr->t_preempt )
	  make_timer_request_(processor_, CURR_TID,
			      CURRENTTIME+ReadTimer(processor_) );
	
#ifdef SEMDEBUG      
	printf("Thread #%d SPINS on semaphore. OS_EXPIRE scheduled for %ld\n",
	       CURR_TID, CURRENTTIME+ReadTimer(processor_));
#endif
	
	TrapToSimulator();
    }
    
    sptr->tid = CURR_TID;
    EVENT_SEM_V(sid, CURRENTTIME);
    
    return(OK);
}


GLOBAL int sem_V(Sem addr) 
{
    int resumetid, sw;
    SemBlk *sptr;
    Word sid = AddrToSid(addr);
    ProcBlk *pp;

#ifdef SEMDEBUG
    printf("#%d Entering V(%d)\n", CURR_TID, sid);
#endif
    
    if (InvalidSem(sid) || (sptr = &semaphore[sid])->state == S_FREE)
      fatal("sem_V failed: Invalid semaphore %d", sid);
    
    /* quadratic delay due to spinning */
    sw = sptr->threadswaiting;
    bus_free_again += sw * sw * MEM_ACCESS_LATENCY;

#ifdef SEMDEBUG    
    if (sptr->tid != CURR_TID)
      fprintf(stderr,
	      "Warning: semaphore %d freed by #%d, but locked by #%d\n",
	      sid, CURR_TID, sptr->tid);
#endif

    sptr->tid = -1; /* unowned */

    Shared_Memory_Write((void *)addr, 1, WORD);
    
#ifdef SEMDEBUG
    printf("SEM_V START: #%d V(%d = %s)= %d threadswaiting = %d\n", 
	   CURR_TID, sid, sptr->name, *(char *)addr, sptr->threadswaiting);
#endif

    while (sptr->threadswaiting > 0) {
	Thread *osptr;
	ThreadInfo *tptr;
	
	resumetid = getfirst(sptr->sqhead);
	sptr->threadswaiting--;
	
	osptr = OSBLOCK(resumetid);
#ifdef SEMDEBUG
	printf("resumetid = %d, state = %d (0x%lx), proc = %d\n", resumetid,
	       osptr->t_state, &(osptr->t_state),
	       osptr->t_processor);
#endif
	osptr->t_state = T_CURRENT;
	tptr = &thread_table_[resumetid];
	WriteTimer(osptr->t_processor,
		   ReadTimer(osptr->t_processor)-(CURRENTTIME-tptr->t_time));
	
	pp = &proc_table_[osptr->t_processor];
	if ((pp)->p_timerreq != NULL) { 
	    ((pp)->p_timerreq)->h.reqtype = SC_NOP;
	    (pp)->p_timerreq = NULL;
	}
	/* CancelPendingTimerInterrupt(pp); */
	
#ifdef SEMDEBUG
	printf("LOOP #%d V(%d = %s) threadswaiting-- = %d resumetid = %d\n", 
	       CURR_TID, sid, sptr->name, sptr->threadswaiting, resumetid);
#endif
	
	MAKE_RESUME_REQUEST(resumetid, CURRENTTIME, OK);
	/* Processor time will be updated by resume request */
    }
    return(OK);
}


#else /* ndef NOBUSY */

GLOBAL int sem_P(Sem addr) 
{
    Word sid = AddrToSid(addr);
    SemBlk *sptr;
    int v, fail_count = 0;
    
    if (InvalidSem(sid) || (sptr = &semaphore[sid])->state == S_FREE) {
	fatal("sem_P failed: Invalid semaphore %d", sid);
	return(0); /* unreachable, but eliminates warnings */
    }
    
    EVENT_SEM_P(sid, CURRENTTIME);
    
    sptr->threadswaiting++;
    
#ifdef SEMDEBUG
    printf("P%d. Trying SEM P(%d) %s  addr=0x%lx\n", processor_, sid,
	   currosptr->t_preempt ? "OK" : "NO PREEMPT", addr);
#endif
    
    while(1) {
#ifdef SEMDEBUG
	printf("P%d. Retrying P(%d) %s Timer = %d\n", processor_,
	       sid, currosptr->t_preempt ? "OK" : "NO PREEMPT",
	       ReadTimer(processor_) );
#endif
	
	if ( (v = (int)Shared_Memory_Read((void *)addr, WORD)) > 0) {
#ifdef SEMDEBUG
	    printf("\tP%d: Detected nonzero\n", processor_);
#endif
	    v = Test_and_Set((void *)addr);
	    if (v > 0) break;
#ifdef SEMDEBUG
	    else printf("\tP%d: TAS=%d\n", processor_, v);
#endif
	}

	fail_count++; /* exponential backoff */
       	AddTime(TTSET_DELAY);
	AddTime(fail_count > 160 ? 1024 : 1 << (fail_count >> 4));
	
	if ( ReadTimer(processor_)<=0 && currosptr->t_preempt) {
	    expired_ = TRUE;
	    TrapToSimulator();
	}
    }
    
#ifdef SEMDEBUG
    printf("P%d. #%d SEM P(%d) successful value = %d\n", processor_,
	   CURR_TID, sid, *(char *)addr);
#endif
    sptr->tid = CURR_TID;
    sptr->threadswaiting--;
    
    EVENT_SEM_V(sid, CURRENTTIME);
    return(OK);
}


GLOBAL int sem_V(Sem addr) 
{
    Word sid = AddrToSid(addr);
    SemBlk *sptr;

    if (InvalidSem(sid) || (sptr = &semaphore[sid])->state == S_FREE) {
	fatal("sem_V failed: Invalid semaphore %d", sid);
	return(0); /* unreachable, but eliminates warnings */
    }
    
#ifdef SEMDEBUG
    printf("%lu: P%d. #%d  SEM V(%d)\n",  CURRENTTIME,
	    processor_, CURR_TID, sid);
#endif
    
#ifdef SEMDEBUG    
    if (sptr->tid != CURR_TID)
      fprintf(stderr,
	      "Warning: semaphore %d freed by #%d, but locked by #%d\n",
	      sid, CURR_TID, sptr->tid);
#endif

    Shared_Memory_Write((void *)addr, 1, WORD);
    sptr->tid = -1; /* unowned */
    return(OK);
}

#endif /* else ifdef NOBUSY */
