 
 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: shmem.h
 *
 * Description: Definitions used in the shared memory module
 *
 * Last Modified: 6-28-91
 *
 * Data Structures:
 *     Cmpswp - data structure for compare and swap access
 *
 * Constants:
 *     MODULE_SIZE
 *     MODULE_MASK
 *     MEMSIZE
 *    
 * Referenced parameters:
 *     MODULE_BITS, NO_OF_MODULES, LINE_BITS, MAX_SEMAPHORES,
 *     NO_OF_PROCESSORS
 *
 * Macros: EVENT_SEM_P, EVENT_SEM_V
 *
 ****************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.
 ***************************************************************************
 * $Header: /psg/proteus/RCS/shmem.h,v 1.3 92/11/03 14:34:31 brewer Exp $
 * $Log:	shmem.h,v $
 * Revision 1.3  92/11/03  14:34:31  brewer
 * Added top_of_memory declaration.
 * Added names for processor requests (mr_request_[])
 * Added names for cache responses (cc_response_[])
 * changed EVENT macros to use IndexEvent instead of ATEVENT
 * 
 * Revision 1.2  92/04/16  13:40:47  brewer
 * Added prototypes for stuff in shmem.common.c
 * 
 * Revision 1.1  92/02/11  14:19:58  brewer
 * Initial revision
 * 
 \**************************************************************************/

#ifndef SHMEM_H
#define SHMEM_H

#include "mem.param"
#include "cache.h"  /* for definition of LINE_SIZE */

#define MODULE_SIZE (1 << MODULE_BITS)

#ifdef WATCH_SEM
#define EVENT_SEM_V(sid, time)  IndexEvent(EV_SEM_V, (sid), 0, (time))
#define EVENT_SEM_P(sid, time)  IndexEvent(EV_SEM_P, (sid), 1, (time))
#else
#define EVENT_SEM_V(sid, time)
#define EVENT_SEM_P(sid, time)
#endif

/* These constants are currently not used */
#define MODULE_MASK                 (NO_OF_MODULES-1)

#define MEMSIZE                     (MAX_SEMAPHORES*LINE_SIZE + \
				     MODULE_SIZE * NO_OF_MODULES)

#define BYTE            00000
#define HALFWORD        00001
#define WORD            00002
#define DOUBLEWORD      00003
#define FLOAT           00002
#define DOUBLE          00003

#define ACCESS_MASK     00003

#define TESTTAG         00010
#define SETTAG          00020


#define PREVIOUS_MASK   00100

#define PREVIOUS        00100

/* Atomic operations */

#define ATOMIC_MASK     07000

#define INCREMENT       01000
#define FINCR           INCREMENT
#define DECREMENT       02000
#define FDECR           DECREMENT
#define FADD            03000
#define FAND            04000
#define FAOR            05000
#define CMPSWP          06000

#define SOFT            010000

/* Shared Memory table */

extern char memory[MEMSIZE];
extern char memtag[MEMSIZE];
extern char *top_of_memory;  /* = &memory[MEMSIZE] */

Time mem_free_again[NO_OF_PROCESSORS];
Time cache_free_again[NO_OF_PROCESSORS];
Time bus_free_again;

Time BusAccess();

#define InvalidModule(m)           ((m)<0 || (m)>=NO_OF_MODULES)

#define AddrToIndex(a)             ((int)((char *)(a) - memory))

#define BLOCK_MASK                 ((1 << (sizeof(Word)*8-LINE_BITS))-1)
#define GetBlockFromIndex(i)       ( ((i) >> LINE_BITS) & BLOCK_MASK)


double shmem_result_fl(double, double, Word);
void write_shared_memory_fl(void *, double, Word);
double shared_memory_fl(void *, Word);
Word shmem_result(Word, Word, Word);
void write_shared_memory(void *, Word, Word) ;
Word shared_memory(void *, Word);


/* Processor memory request codes */

#define MR_READ                 0
#define MR_WRITE                1
#define MR_READSOFT             2
#define MR_WRITESOFT            3
#define MR_FLUSHSOFT            4
#define MR_READFL               5
#define MR_WRITEFL              6
#define MR_READTAG              7
#define MR_WRITETAG             8

/* mr_request_ defined in shmem.c */
extern char *mr_request_[] /* = {"MR_READ", "MR_WRITE", "MR_READSOFT",
		                "MR_WRITESOFT", "MR_FLUSHSOFT", "MR_READFL",
		                "MR_WRITEFL", "MR_READTAG", "MR_WRITETAG"} */;

/* Cache controller responses to processor */

#define MEM_READY             0
#define MEM_WAIT              1
#define MEM_SWITCH            2
#define MEM_MISMATCH          3
#define MEM_ERROR             4
#define MEM_RETRY             5

/* cc_response_ defined in shmem.c */
extern char *cc_response_[] /* = {"MEM_READY", "MEM_WAIT", "MEM_SWITCH",
			          "MEM_MISMATCH", "MEM_ERROR", "MEM_RETRY"} */;

/* Special structure for compare & swap */
typedef struct {
    Word old;
    Word new;
} Cmpswp ;

#endif /* ifndef SHMEM_H */
