
 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: shmem.nocache.c
 *
 * Description:
 *         Shared memory operations for machines without caches
 *
 * Last Modified:  $Date: 92/11/19 14:16:36 $
 *
 * Global Functions:
 *
 * Global Variables:
 *
 ****************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.
 ****************************************************************************
 * $Header: /psg/proteus/RCS/shmem.nocache.c,v 1.4 92/11/19 14:16:36 brewer Exp $
 * $Log:	shmem.nocache.c,v $
 * Revision 1.4  92/11/19  14:16:36  brewer
 * Updated calls to build_new_packet_.
 * 
 * Revision 1.3  92/04/16  13:41:16  brewer
 * Fixed bug in memory_controller, case WRITE_FL.  Changed it to call
 * shmem_result_fl, write_shared_memory_fl, and shared_memory_fl
 * instead of the non-fl versions. Also added automatic variable
 * previous_double.
 * 
 * Revision 1.2  92/04/01  11:24:39  brewer
 * cleanup: fixed gcc warnings, ANSI prototypes
 * 
 * Revision 1.1  92/02/11  13:56:27  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include "net.h"
#include "sema.h"
#include "simcalls.h"
#include "simreq.h"
#include "processor.h"
#include "thread.h"
#include "event.h"
#include "packet.h"

#include "shmem.common.c"

#include "shmem.calls.c"

/* Shared Memory call handlers for network-based machines without caches */

#define MPKT_READ         0
#define MPKT_WRITE        1
#define MPKT_READTAG      2
#define MPKT_WRITETAG     3
#define MPKT_REPLY        4
#define MPKT_READFL       5
#define MPKT_WRITEFL      6

#define MSG_READ(t,src,trg,tid,valaddr,addr,mode) \
  send_protocol_message(t,src,trg,tid,MPKT_READ,3,valaddr,addr,mode)
#define MSG_WRITE(t,src,trg,tid,valaddr,addr,mode,value) \
  send_protocol_message(t,src,trg,tid,MPKT_WRITE,4,valaddr,addr,mode,value)
#define MSG_READFL(t,src,trg,tid,valaddr,addr,mode) \
  send_protocol_message(t,src,trg,tid,MPKT_READFL,3,valaddr,addr,mode)
#define MSG_WRITEFL(t,src,trg,tid,valaddr,addr,mode,value) \
  send_protocol_message(t,src,trg,tid,MPKT_WRITEFL,4,valaddr,addr,mode,value)
#define MSG_READTAG(t,src,trg,tid,valaddr,addr,mode) \
  send_protocol_message(t,src,trg,tid,MPKT_READTAG,3,valaddr,addr,mode)
#define MSG_WRITETAG(t,src,trg,tid,valaddr,addr,mode,value) \
  send_protocol_message(t,src,trg,tid,MPKT_WRITETAG,4,valaddr,addr,mode,value)
#define MSG_REPLY(t,src,trg,tid,replyval) \
  send_protocol_message(t,src,trg,tid,MPKT_REPLY,1,replyval)


typedef struct {
    SimHeader  h;
    Word       processor; /* argv[0] */
    ulong      address;   /* argv[1] */
    Word       mode;      /* argv[2] */
    void       *valaddr;  /* argv[3] */
    Word       value;     /* argv[4] */
} MemRequest;

void send_protocol_message();

/************************************************************************/

void processor_request_nocache(Time time, int processor, int tid,
			       int reqtype, void *valaddr, ulong address,
			       Word mode, Word value) 
{
    switch(reqtype) {
      case MR_READ:
	MSG_READ(time, processor, AddressToProcessor(address), tid, 
		 valaddr, address, mode);
	return;
      case MR_WRITE:
	MSG_WRITE(time, processor, AddressToProcessor(address), tid,
		  valaddr, address, mode, value);
	return;
      case MR_READFL:
	MSG_READFL(time, processor, AddressToProcessor(address), tid, 
		   valaddr, address, mode);
	return;
      case MR_WRITEFL:
	MSG_WRITEFL(time, processor, AddressToProcessor(address), tid,
		    valaddr, address, mode, value);
	return;
      case MR_READTAG:
	MSG_READTAG(time, processor, AddressToProcessor(address), tid, 
		    valaddr, address, mode);
	return;
      case MR_WRITETAG:
	MSG_WRITETAG(time, processor, AddressToProcessor(address), tid,
		     valaddr, address, mode, value);
	return;
      default:
	fatal("processor_request_nocache: Illegal reqtype = %d\n", reqtype);
    }
}


/************************************************************************/


void shared_read_handler_(MemRequest *rptr) 
{
    ulong iaddr = AddrToIndex(rptr->address);
    
#ifdef ATDEBUG
    printf("@ %d (%d)will Read(%x) i=%d\n",
	   rptr->h.tid, rptr->processor, rptr->address, iaddr);
#endif
    
    processor_request_nocache(rptr->h.timestamp, rptr->processor, rptr->h.tid, 
			      MR_READ, rptr->valaddr,
			      iaddr, rptr->mode, 0);
}


void shared_read_fl_handler_(MemRequest *rptr) 
{
    ulong iaddr = AddrToIndex(rptr->address);
    
#ifdef ATDEBUG
    printf("@ %d (%d)will ReadFl(%x) i=%d\n",
	   rptr->h.tid, rptr->processor, rptr->address, iaddr);
#endif
    
    processor_request_nocache(rptr->h.timestamp, rptr->processor, rptr->h.tid, 
			      MR_READFL, rptr->valaddr,
			      iaddr, rptr->mode, 0);
}


void shared_readtag_handler_(MemRequest *rptr) 
{
    ulong iaddr = AddrToIndex(rptr->address);
    
#ifdef ATTDEBUG
    printf("@ %d (%d)will ReadTag(%x) i=%d\n",
	   rptr->h.tid, rptr->processor, rptr->address, iaddr);
#endif
    
    processor_request_nocache(rptr->h.timestamp, rptr->processor, rptr->h.tid, 
			      MR_READTAG, rptr->valaddr,
			      iaddr, rptr->mode, 0);
}


void shared_write_handler_(MemRequest *rptr) 
{
    ulong iaddr = AddrToIndex(rptr->address);
    
#ifdef ATDEBUG
    printf("@ %d (%d)will Write(%x, %d) i=%d\n",
	   rptr->h.tid, rptr->processor, rptr->address, rptr->value, iaddr);
#endif
    
    processor_request_nocache(rptr->h.timestamp, rptr->processor, rptr->h.tid, 
			      MR_WRITE, 
			      rptr->valaddr, iaddr, rptr->mode,
			      rptr->value);
}


void shared_write_fl_handler_(MemRequest *rptr) 
{
    ulong iaddr = AddrToIndex(rptr->address);
    
#ifdef ATDEBUG
    printf("@ %d (%d)will Write(%x, %f) i=%d\n",
	   rptr->h.tid, rptr->processor, rptr->address, *(double *)rptr->value, iaddr);
#endif
    
    processor_request_nocache(rptr->h.timestamp, rptr->processor, rptr->h.tid, 
			      MR_WRITEFL, 
			      rptr->valaddr, iaddr, rptr->mode,
			      rptr->value);
}


void shared_writetag_handler_(MemRequest *rptr) 
{
    ulong iaddr = AddrToIndex(rptr->address);
    
#ifdef ATTDEBUG
    printf("@ %d (%d)will WriteTag(%x, %d) i=%d\n",
	   rptr->h.tid, rptr->processor, rptr->address, rptr->value, iaddr);
#endif
    
    processor_request_nocache(rptr->h.timestamp, rptr->processor, rptr->h.tid, 
			      MR_WRITETAG, 
			      rptr->valaddr, iaddr, rptr->mode,
			      rptr->value);
    
}



void memory_controller(Time time, int source, int target, int tid,
		       int reqtype, Word valaddr, Word address,
		       Word mode, Word value) 
{
    Word previous;
    double previous_double;
    void *caddr;
    
    if (reqtype != MPKT_REPLY)
      time = mem_free_again[target] = 
	MAX(mem_free_again[target], time) + MEM_ACCESS_LATENCY;
    
    switch(reqtype) {
      case MPKT_READ:
	
	if (mode & TESTTAG)
	  if (memtag[address] != FULL) {
	      MSG_REPLY(time, target, source, tid, MEM_MISMATCH);
	      return;
	  }
	
	*(Word *)valaddr = shared_memory(&memory[address], mode);
	
	if (mode & SETTAG)
	  memtag[address] = EMPTY;
	
	MSG_REPLY(time, target, source, tid, MEM_READY);
	break;
	
      case MPKT_WRITE:
	
	if (mode & TESTTAG)
	  if (memtag[address] == FULL) {
	      MSG_REPLY(time, target, source, tid, MEM_MISMATCH);
	      return;
	  }
	
	caddr = &memory[address];
	previous = shared_memory(caddr, mode);
	
	/* Special case for compare & swap */
	if ((mode & ATOMIC_MASK) == CMPSWP) {
	    previous = (previous == ((Cmpswp *)value)->old);
	    if (previous)
	      write_shared_memory(caddr, 
				  ((Cmpswp *)value)->new, 
				  mode);
	}
	else
	  write_shared_memory(caddr, value, mode);
	
	*(Word *)valaddr = shmem_result(previous, 
					shared_memory(caddr, mode), 
					mode);
	
	if (mode & SETTAG)
	  memtag[address] = FULL;
	
	MSG_REPLY(time, target, source, tid, MEM_READY);
	break;
	
      case MPKT_READFL:
	
	if (mode & TESTTAG)
	  if (memtag[address] != FULL) {
	      MSG_REPLY(time, target, source, tid, MEM_MISMATCH);
	      return;
	  }
	
	*(double *)valaddr = shared_memory_fl(&memory[address], mode);
	
	if (mode & SETTAG)
	  memtag[address] = EMPTY;
	
	MSG_REPLY(time, target, source, tid, MEM_READY);
	break;
	
      case MPKT_WRITEFL:
	
	if (mode & TESTTAG)
	  if (memtag[address] == FULL) {
	      MSG_REPLY(time, target, source, tid, MEM_MISMATCH);
	      return;
	  }
	
	caddr = &memory[address];
	previous_double = shared_memory_fl(caddr, mode);
	write_shared_memory_fl(caddr, *((double *)value), mode);
	
	*((double *)valaddr) = shmem_result_fl(previous_double,
					       *((double *)value), mode);
	if (mode & SETTAG)
	  memtag[address] = FULL;
	
	MSG_REPLY(time, target, source, tid, MEM_READY);
	break;
	
      case MPKT_READTAG:
	*(Word *)valaddr = (Word)memtag[address];
	MSG_REPLY(time, target, source, tid, MEM_READY);
	break;
	
      case MPKT_WRITETAG:
	previous = (Word)memtag[address];
	memtag[address] = (char)value;
	
	*(Word *)valaddr = shmem_result(previous, value, mode);
	
	MSG_REPLY(time, target, source, tid, MEM_READY);
	break;
	
      case MPKT_REPLY:
	MAKE_RESUME_REQUEST(tid, time, valaddr);
	break;
	
    }
}


void Nocache_Pkt(int sendtid, Time time, int source, int target,
		 NOCPacket *packet) 
{
    memory_controller(time, source, target, sendtid,
		      packet->reqtype,
		      packet->valaddr, packet->address, 
		      packet->mode, packet->value);
}


void send_protocol_message(Time time, int source, int target, int tid,
			   int pkttype, int argc, Word args) 
{
    int i;
    Packet *packet = build_new_packet_(PKT_MEM, argc+1, 2, 
				       NOCACHE_PKT, pkttype);
    
    for(i=0; i<argc; i++)
      packet->data[2+i] = *(&args + i);
    
    MAKE_REQUEST_4(SC_SEND_PACKET, tid, time, 
		   source, target, packet, FALSE);
}


