
/**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: simcalls.c
 *
 * Description: Defines request, interrupt, and packet handler tables
 *
 * Last Modified:  6-6-91  (eab)
 *
 * Global Functions:
 *     init_simcalls_()
 *
 * Global Variables:
 *     FuncPtr request_handler_[MAX_SIMCALLS]
 *         Simulator request handler table
 *     FuncPtr intr_handler_[MAX_INTR_TYPES]
 *         Interrupt handler table
 *
 *     char *request_name_[]
 *         Defines the name of each request type
 *     char *intr_name_[]
 *         Defines the name of each interrupt type
 *
 * Referenced parameters:
 *     NOCACHE, NET, BUS, MAX_INTR_TYPES
 *
 ***************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.		
 ****************************************************************************
 * $Header: /psg/proteus/RCS/simcalls.c,v 1.3 92/07/06 11:25:16 brewer Exp $
 * $Log:	simcalls.c,v $
 * Revision 1.3  92/07/06  11:25:16  brewer
 * Added define_new_simcall
 * Added print_request_ array
 * 
 * Revision 1.2  92/04/28  11:30:34  brewer
 * Removed include of useripi file, which is defunct.
 * 
 * Revision 1.1  92/02/11  13:56:28  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include "sim.h"
#include "conf.h"
#include "simcalls.h"
#include "cache.param"

GLOBAL FuncPtr request_handler_[MAX_SIMCALLS];
GLOBAL FuncPtr intr_handler_[MAX_INTR_TYPES];
GLOBAL void (*print_request_[MAX_SIMCALLS])(FILE *, SimRequest *);

GLOBAL const char *request_name_[MAX_SIMCALLS];
GLOBAL const char *intr_name_[MAX_INTR_TYPES];

void init_simcalls_() 
{
    /* initialize request handler names */
    request_name_[SC_RESUME] = "SC_RESUME";
    request_name_[SC_TIMER] = "SC_TIMER";
    request_name_[SC_SNAPSHOT] = "SC_SNAPSHOT";
    request_name_[SC_SHARED_READ] = "SC_SHARED_READ";
    request_name_[SC_SHARED_WRITE] = "SC_SHARED_WRITE";
    request_name_[SC_SHARED_READTAG] = "SC_SHARED_READTAG";
    request_name_[SC_SHARED_WRITETAG] = "SC_SHARED_WRITETAG";
    request_name_[SC_SHARED_READ_FL] = "SC_SHARED_READ_FL";
    request_name_[SC_SHARED_WRITE_FL] = "SC_SHARED_WRITE_FL";
    request_name_[SC_FLUSH] = "SC_FLUSH";
    request_name_[SC_IPI_BUS] = "SC_IPI_BUS";
    request_name_[SC_SEND_PACKET] = "SC_SEND_PACKET";
    request_name_[SC_RECV_PACKET] = "SC_RECV_PACKET";
    request_name_[SC_ROUTE_PACKET] = "SC_ROUTE_PACKET";
    request_name_[SC_SCHEDULE_INTR] = "SC_SCHEDULE_INTR";

    /* initialize request handler table */
    request_handler_[SC_RESUME]           = (FuncPtr)resume_handler_;
    request_handler_[SC_TIMER]            = (FuncPtr)timer_interrupt_handler_;
    
    request_handler_[SC_SNAPSHOT]         = (FuncPtr)snapshot_handler_;
    
    request_handler_[SC_SHARED_READ]      = (FuncPtr)shared_read_handler_;
    request_handler_[SC_SHARED_WRITE]     = (FuncPtr)shared_write_handler_;
    
    request_handler_[SC_SHARED_READTAG]   = (FuncPtr)shared_readtag_handler_;
    request_handler_[SC_SHARED_WRITETAG]  = (FuncPtr)shared_writetag_handler_;
    
    request_handler_[SC_SHARED_READ_FL]   = (FuncPtr)shared_read_fl_handler_;
    request_handler_[SC_SHARED_WRITE_FL]  = (FuncPtr)shared_write_fl_handler_;

    request_handler_[SC_SCHEDULE_INTR]    = (FuncPtr)schedule_intr_handler_;

#ifndef NOCACHE
    request_handler_[SC_FLUSH]            = (FuncPtr)flush_handler_;
#endif
    
#ifdef BUS
    request_handler_[SC_IPI_BUS]          = (FuncPtr)ipi_bus_handler_;
#endif
    
#ifdef NET
    request_handler_[SC_SEND_PACKET]      = (FuncPtr)send_packet_handler_;
    request_handler_[SC_RECV_PACKET]      = (FuncPtr)receive_packet_handler_;
    request_handler_[SC_ROUTE_PACKET]     = (FuncPtr)route_packet_handler_;
#endif

    /* initialize interrupt handler names */
/*    intr_name_[IPI_KILL_THREAD] = "IPI_KILL_THREAD";
 *    intr_name_[IPI_SUSPEND_THREAD] = "IPI_SUSPEND_THREAD";
 *    intr_name_[IPI_RESUME_THREAD] = "IPI_RESUME_THREAD";
 *    intr_name_[IPI_SPAWN_THREAD] = "IPI_SPAWN_THREAD";
 *    intr_name_[IPI_SET_PRIORITY] = "IPI_SET_PRIORITY";
 *    intr_name_[IPI_SET_PREEMPT] = "IPI_SET_PREEMPT";
 *    intr_name_[IPI_INTERRUPT_CTRL] = "IPI_INTERRUPT_CTRL";
 */

    intr_name_[INTR_SEND_COMPLETE] = "INTR_SEND_COMPLETE";

    /* initialize interrupt handler table */
/*    intr_handler_[IPI_KILL_THREAD]     = (FuncPtr)ipi_OS_kill_handler_;
 *    intr_handler_[IPI_SUSPEND_THREAD]  = (FuncPtr)ipi_OS_suspend_handler_;
 *    intr_handler_[IPI_RESUME_THREAD] = (FuncPtr)ipi_OS_resume_handler_;
 *    intr_handler_[IPI_SPAWN_THREAD]  = (FuncPtr)ipi_OS_spawn_handler_;
 *    intr_handler_[IPI_SET_PRIORITY]  = (FuncPtr)ipi_OS_set_priority_handler_;
 *    intr_handler_[IPI_SET_PREEMPT]   = (FuncPtr)ipi_OS_set_preempt_handler_;
 *    intr_handler_[IPI_INTERRUPT_CTRL]  = (FuncPtr)ipi_OS_intr_ctrl_handler_;
 */

    intr_handler_[INTR_SEND_COMPLETE]  = (FuncPtr)null_handler_;
}



int define_new_simcall(char *name,
		       void (*handler)(SimRequest *),
		       void (*print)(FILE *, SimRequest *))
{
    int number = MIN_USER_SC;

    while (request_handler_[number] != NULL)
      if (++number == MAX_SIMCALLS) {
	  fatal("No free request handlers; unable to define simcall \"%s\".\n",
		name);
      }

    request_handler_[number] = (FuncPtr)handler;
    print_request_[number] = print;
    request_name_[number] = name;
    return(number);
}








