 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: user.h
 *
 * Description: Defines all the basics
 *
 * Last Modified:  8-2-91  (eab)
 *
 * Data Structures: none
 *
 * Constants:
 *     CURR_PROCESSOR
 *     CURR_TID
 *     CURR_TIME
 *     
 * Macros:
 *     AddTime(t)
 *     SubtractTime(t)
 *     CYCLE_COUNTING_ON
 *     CYCLE_COUNTING_OFF
 *     Increment(a)
 *     Decrement(a)
 *     Test_and_Set(a)
 *     Fetch_and_Increment(a)
 *     Fetch_and_Decrement(a)
 *     
 * Referenced parameters: none
 *
 ***************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.		
 ***************************************************************************
 * $Header: /usr/wildcat/dfk.keep/parallel/proteus/current/engine/RCS/user.h,v 1.7 1996/07/20 02:07:37 dfk Exp $
 * $Log: user.h,v $
 * Revision 1.7  1996/07/20 02:07:37  dfk
 * added proteusSeed global variable
 *
 * Revision 1.6  1994/05/15 11:15:04  dfk
 * changed EVENTS flag to WATCH_USER, for consistency with others
 *
 * Revision 1.5  94/05/13  00:51:31  dfk
 * added {} to the null form of the EVENT macros
 * 
 * Revision 1.4  94/05/05  22:20:41  dfk
 * added a flag 
 * EVENTS 
 * that can turn on or off the event recording.
 * It is controlled with the config program.
 * 
 * Revision 1.3  94/01/24  00:39:42  dfk
 * Added Sanjay's support for 64-bit timers.
 * Added thread_sleep_until.
 * Fixed a few bugs in thread sleep/wakeup code (my bugs).
 * Fixed printf formats in many places.
 * 
 * Revision 1.2  93/02/26  01:06:57  dfk
 * wrapped macros to avoid counting cycles
 * 
 * Revision 1.1  93/02/26  01:05:18  dfk
 * Initial revision
 * 
 * Revision 1.2  92/02/11  17:23:16  brewer
 * Moved CYCLE_COUNTING_* macros into ifdef _CA_ region
 * 
 * Revision 1.1  92/02/11  14:20:03  brewer
 * Initial revision
 * 
 \**************************************************************************/

#ifndef _USERH_
#define _USERH_

#include "sim.h"
#include "conf.param"
#include "rt_thread_def.h"
#include "cache.h"
#include "shmem.h"
#include "sema.h"
#include "thread.h"

extern int proteusSeed;	      /* value of -seed argument */

#ifdef _CA_
/* variable for .ca routines (ONLY) that holds current time */

/*
 * We cannot declare current_time_ to have type Time because that
 * confuses augment when Time is a 64-bit quantity.
 */
extern volatile ulong current_time_;

#undef CURRENTTIME		/* redefine CURRENTTIME for .ca files */
#ifdef LONG_LONG_TIME
/*
 * Special mechanism to compute CURRENTTIME.
 * Record time, and then force the basic block to finish
 * by referencing current_time_.
 */
#define CURRENTTIME ({\
	Time _current_time_ = (base_time_ - cycles_);\
	current_time_;\
	_current_time_;\
    })
#else /* LONG_LONG_TIME */
#define CURRENTTIME current_time_
#endif /* ifdef LONG_LONG_TIME */

/* primitive commands for turning cycle counting on and off */
extern volatile int counting_on, counting_off;
/* Macros to control cycle counting */
#define CYCLE_COUNTING_ON  (void)counting_on
#define CYCLE_COUNTING_OFF (void)counting_off

#else /* not a .ca file */
#define CYCLE_COUNTING_ON
#define CYCLE_COUNTING_OFF

#endif /* ifdef _CA_ */

/* macros to update the current simulated time (in cycles) */
#define AddTime(t)  (cycles_-=(t))
#define SubtractTime(t) (cycles_+=(t))

#define Increment(a)           Shared_Memory_Write((a),0,WORD|INCREMENT)
#define Decrement(a)           Shared_Memory_Write((a),0,WORD|DECREMENT)
#define Test_and_Set(a)        Shared_Memory_Write((a),0,WORD|PREVIOUS)
#define Fetch_and_Increment(a) \
  Shared_Memory_Write((a),0,WORD|PREVIOUS|INCREMENT)
#define Fetch_and_Decrement(a) \
  Shared_Memory_Write((a),0,WORD|PREVIOUS|DECREMENT)

#define CURR_PROCESSOR         processor_
#define CURR_TID               currtid
#define CURR_TIME              CURRENTTIME

#define S_PROCESSOR(a)         (InvalidSem(AddrToSid(a)) ? ERROR : \
				semaphore[AddrToSid(a)].processor)
#define S_NAME(a)              (InvalidSem(AddrToSid(a)) ? ERROR : \
				semaphore[AddrToSid(a)].name)
#define S_VALUE(a)             (InvalidSem(AddrToSid(a)) ? ERROR : \
				*(char *)(a))

/* event generation macros */

#ifdef WATCH_USER

#define USER_EVENT(kind, value) { \
  CYCLE_COUNTING_OFF; \
  user_event_(kind, value); \
  CYCLE_COUNTING_ON; \
}
#define USER_TIME_EVENT(kind, value, time) { \
  CYCLE_COUNTING_OFF; \
  user_time_event_(kind, value, time); \
  CYCLE_COUNTING_ON; \
}
#define INDEX_EVENT(kind, index, value) { \
  CYCLE_COUNTING_OFF; \
  user_index_event_(kind, index, value); \
  CYCLE_COUNTING_ON; \
}
#define INDEX_TIME_EVENT(kind, index, value, time) { \
  CYCLE_COUNTING_OFF; \
  user_index_time_event_(kind, index, value, time); \
  CYCLE_COUNTING_ON; \
}

#else // WATCH_USER

#define USER_EVENT(kind, value) {}
#define USER_TIME_EVENT(kind, value, time) {}
#define INDEX_EVENT(kind, index, value) {}
#define INDEX_TIME_EVENT(kind, index, value, time) {}

#endif // WATCH_USER

#endif /* ifndef _USERH_ */
