 
 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: Event and Statistics Module
 *
 * Description: Routines to generate and manage events and statistics
 *
 * Last Modified:  $Date: 92/10/13 13:50:25 $ ($Author)
 *
 * Originally developed by Eric A. Brewer.
 *
 * Global Functions:
 *    ** trace-file procedures **          
 *    void init_statistics_(const char *parameters, const char *fname)
 *    void exit_statistics_(unsigned long endtime)
 *    void flush_event_file_(void)
 *
 *    ** event-generation procedures **
 *    void Event(ulong evt, ulong value, ulong time);
 *    void IndexEvent(ulong evt, ulong index, ulong value, ulong time);
 *
 *    ** metric procedures **
 *    int new_metric(const char *name, double val)
 *    int new_array_metric(const char *name, int size, double val)
 *    double sum_metric_(int metric, double val);
 *    double sum_array_metric_(int metric, int index, double val);
 *    double max_metric_(int metric, double val);
 *    double max_array_metric_(int metric, int index, double val);
 *    double min_metric_(int metric, double val);
 *    double min_array_metric_(int metric, int index, double val);
 *    double avg_metric_(int metric, double val);
 *    double avg_array_metric_(int metric, int index, double val);
 *    double stddev_metric_(int metric, int avg_metric, double val);
 *    double stddev_array_metric_(int metric, int avg_metric,
 *                                int index, double val);
 *
 *    ** stat procedures **
 *    void init_stat(short stat, long value, short rate)
 *    void write_stat(short stat, unsigned long t)
 *    void init_array_stat(short stat, short size, long value, short rate)
 *    void write_array_stat(short stat, short index, unsigned long t)
 * 
 *
 * Global Variables:
 *    Stat stats_[128]
 *    ArrayStat astats_[128]
 *    Metric metrics_[256]
 *    ArrayMetric ametrics_[256]
 *    const char *event_filename_
 *    const char *simulation_title_
 *
 ****************************************************************************
 *   Copyright 1991, 1992                                                   *
 *   Eric A. Brewer  and  Chris N. Dellarocas                               *
 *   Massachusetts Institute of Technology                                  *
 *                                                                          *
 *   Permission to use, copy, modify, and distribute this program           *
 *   for any purpose and without fee is hereby granted, provided            *
 *   that this copyright and permission notice appear on all copies         *
 *   and supporting documentation, the name of M.I.T. not be used           *
 *   in advertising or publicity pertaining to distribution of the          *
 *   program without specific prior permission, and notice be given         *
 *   in supporting documentation that copying and distribution is           *
 *   by permission of M.I.T.  M.I.T. makes no representations about         *
 *   the suitability of this software for any purpose.  It is pro-          *
 *   vided "as is" without express or implied warranty.		            *
 ****************************************************************************
 * $Header: /u/brewer/sim4/libevt/RCS/event.c,v 1.8 92/10/13 13:50:25 brewer Exp $
 * $Log:	event.c,v $
 * Revision 1.8  92/10/13  13:50:25  brewer
 * Changed TraceFileComment format and args
 * 
 * Revision 1.7  92/10/09  16:54:17  brewer
 * Added args checks in TraceFileComment
 * 
 * Revision 1.6  92/10/09  16:00:06  brewer
 * Added TraceFileComment
 * 
 * Revision 1.5  92/10/08  13:45:01  brewer
 * Added AtEvent()
 * Added prototype for tell()
 * 
 * Revision 1.4  92/10/07  16:49:18  brewer
 * added WriteBuffer and support for "file system full" errors
 * added BOOL initialized
 * 
 * Revision 1.3  92/10/05  00:04:34  brewer
 * Updated trace file format, set up for libevt.a
 * 
 * Revision 1.2  92/08/11  13:26:58  brewer
 * Changed init_statistics prototype.
 * Updated to support libevt library.
 * 
 * Revision 1.1  92/02/11  13:55:27  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <malloc.h>

#define TFF_VERSION 1.0     /* trace-file format version number,
			       (exactly one digit right of decimal point) */

/* define buffer size in ulongs */
#define MAX_BUFFER_SIZE 1024

#define GLOBAL
#define LIBRARY extern
#define TRUE 1
#define FALSE 0

typedef int BOOL;
typedef unsigned long ulong;

LIBRARY int fsync(int fd);
LIBRARY long tell(int fd);

#ifdef __STDC__

void fatal(const char *, ...);
LIBRARY void bzero(void *pointer, unsigned length);

#else

void fatal();
LIBRARY void bzero();

/* keyword const not handled by cc (v2.1) */
#define const

#endif  /* ifdef __STDC__ else */

#include "event.h"

#ifndef BUFFERED_EVENT_H
error- out of date event.h file
#endif

extern char params_[];  /* the text of the current set of parameters */
GLOBAL Stat stats_[128];
GLOBAL ArrayStat astats_[128];
GLOBAL Metric metrics_[256];
GLOBAL ArrayMetric ametrics_[256];
GLOBAL const char *event_filename_ = "events.sim";
GLOBAL const char *simulation_title_ = NULL;

static int metric_count = 0;
static int ametric_count = 0;

/****************************************************************************/

static BOOL initialized = FALSE;
static ulong *cur, *buffer_max;
static ulong buffer[MAX_BUFFER_SIZE+4];
static FILE *event_file_;
static int fd;
static BOOL ascii_data = FALSE;  /* indicates if stream data has been output
				    since last WriteBuffer() */

#define AddWord(x) (*cur++ = (x))
#define BufferFull  (cur >= buffer_max)

/****************************************************************************/

static BOOL WriteBuffer(int nested) {
    int status, tries, size;
    long file_offset;
    char tmp[80];

    if (!initialized)
      fatal("WriteBuffer invoked before init_statistics_");

    if (ascii_data) {
	fflush(event_file_);
	ascii_data = FALSE;
    }

    size = sizeof(ulong)*(cur-buffer);
    if (size == 0) return(TRUE);
    if (size < 0) fatal("Internal error, negative size in WriteBuffer!");

    /* save offset in case of error, so that we can retry */
    /* According to the docs I shouldn't have to do this, write() should
       leave the file pointer unchanged if it fails, but the
       Ultrix version (at least) modifies the offset. */

    file_offset = tell(fd);

    status = write(fd, (char *)buffer, size);

    if (status == -1) {
	if (nested) return(FALSE);
	switch(errno) {
	  case EROFS:
	    fatal("Trace-file does not permit writing.\n");
	    break;
	  case EFBIG:
	    fatal("Trace-file exceeds process file size limit (see ulimit)\n");
	    break;
	  case ENOSPC:
	    if (isatty(fileno(stdin))) {
		fprintf(stderr, "\n*** WARNING: file system full!\n");
		for (;;) {
		    fprintf(stderr, "Hit r to retry, a to abort:");
		    fgets(tmp, 80, stdin);
		    if (tmp[0] == 'a')
		      fatal("File system full.");
		    if (tmp[0] == 'r') {
			lseek(fd, file_offset, SEEK_SET);
			if (WriteBuffer(TRUE)) break;
			fprintf(stderr, "\n*** WARNING: file system full!\n");
		    }
		}
	    } else {
		fprintf(stderr,
		 "\n*** WARNING: file system full (retry in 60 seconds)\n\n");
		for (tries=2; tries<=4; tries++) {
		    sleep(60);
		    lseek(fd, file_offset, SEEK_SET);
		    if (WriteBuffer(TRUE)) break;
		    fprintf(stderr,
			    "\n*** WARNING: file system full (%d tries)\n\n",
			    tries);
		}
		if (tries > 4)
		  fatal("File system full, unable to write trace file.");
	    }
	    fprintf(stderr, "*** write completed\n");
	    break;
	  case EDQUOT:
	    fatal("User's disk quota exceeded, unable to write trace-file.");
	    break;
	  case ESTALE:
	    fatal("Stale NFS file handle for \"%s\".", event_filename_);
	    break;
	  case EINTR:
	  case ETIMEDOUT:
	    for (tries=1; tries<=3; tries++) {
		sleep(5);
		lseek(fd, file_offset, SEEK_SET);
		if (WriteBuffer(TRUE)) break;
	    }
	    if (tries <= 3) break;
	    /* fall through */
	  case EIO:
	    fatal("I/O error (errno=%d) while writing trace file.", errno);
	    break;
	  case EBADF:
	  case EPIPE:
	  case EFAULT:
	  case EWOULDBLOCK:
	  case EINVAL:
	  default:
	    fatal("Internal error in WriteBuffer()! (errno=%d)\n", errno);
	}
    }
    cur = buffer;
    return(TRUE);
}

/****************************************************************************/

GLOBAL void init_statistics_(const char *parameters, const char *fname)
{
    int i;

    if (initialized) return;

    if (fname != NULL) event_filename_ = fname;
    
    event_file_ = fopen(event_filename_, "w");
    if (event_file_ == NULL)
      fatal("Unable to open trace file \"%s\"", event_filename_);

    fd = fileno(event_file_);

    cur = buffer;
    buffer_max = &buffer[MAX_BUFFER_SIZE];

    initialized = TRUE;

    /* copy parameters to event file */
    putw(0xffff0000 | (short)(TFF_VERSION*10), event_file_);

    if (parameters == NULL)
      fputs(params_, event_file_);
    else
      fputs(parameters, event_file_);

    if (ferror(event_file_))
      fatal("Unable to write data to \"%s\".", event_filename_);

    /* if defined, write out simulation title */
    if (simulation_title_ != NULL)
      fprintf(event_file_, "TITLE %s\n", simulation_title_);

    /* mark end of parameters */
    fputs("%%\n", event_file_);

    if (ferror(event_file_))
      fatal("Unable to write data to \"%s\".", event_filename_);

    ascii_data = TRUE;

    for (i=0; i<128; i++)
      stats_[i].count = -1;
}

/****************************************************************************/

#define FLOAT_TO_LONG(f)  (*((long *) &(f)))

GLOBAL void exit_statistics_(unsigned long endtime)
{
    int i, m;
    long value;
    extern void OutputFinalEvents(FILE *);

    if (!initialized) return;

    WriteBuffer(FALSE);
    OutputFinalEvents(event_file_);
    ascii_data = TRUE;

    /* write out metrics */
    for (m=0; m<metric_count; m++) {
	value = FLOAT_TO_LONG(GET_METRIC(m));
	AddWord(EV_METRIC | MAKE_SHORT_INDEX(m));
	AddWord(value);
    }

    /* write out array metrics */
    for (m=0; m<ametric_count; m++) {
	for (i=0; i<ametrics_[m].size; i++) {
	    AddWord(EV_ARRAYMETRIC | MAKE_SHORT_INDEX(m) | SHORT_VALUE(i));
	    value = FLOAT_TO_LONG((GET_ARRAY_METRIC(m, i)));
	    AddWord(value);
	}
    }

    AddWord(EV_EXIT);
    AddWord(endtime); /* final timestamp */

    flush_event_file_(); /* to write buffer and flush to disk */
    fclose(event_file_);
}


/****************************************************************************/

/* ensure events on disk */
GLOBAL void flush_event_file_(void)
{
    int status;

    if (!initialized) {
	fprintf(stderr,
	    "Warning: flush_event_file_ invoked before init_statistics_\n");
	return;
    }

    WriteBuffer(FALSE);
    status = fsync(fd);
    if (status == -1)
      switch(errno) {
	case ESTALE:
	  fatal("NFS file handle for trace-file is stale.");
	  break;
	case ETIMEDOUT:
	case EINTR:
	  sleep(10);
	  if (fsync(fd) == 0) return;
	  /* fall through */
	case EIO:
	  fatal("I/O error in flush_event_file_ (errno=%d).", errno);
	  break;
	case EBADF:
	case EINVAL:
	case EROFS:
	default:
	  fatal("Internal error in flush_event_file_()! (errno=%d)", errno);
	  break;
      }
}


/****************************************************************************/

GLOBAL void init_stat(short stat, long value, short rate)
{
    stats_[stat].value = value;
    stats_[stat].count = 1;
    stats_[stat].update = rate;
}


GLOBAL void write_stat(short stat, unsigned long t)
{
    AddWord(MAKE_EVENT_FIELD(stat));
    AddWord(stats_[stat].value);
    AddWord(t);
    if (BufferFull) WriteBuffer(FALSE);

    stats_[stat].count = 0;
}


GLOBAL void init_array_stat(short stat, short size, long value, short rate)
{
    short i;

    astats_[stat].value = (long *) malloc(size * sizeof(long));
    astats_[stat].count = (short *) malloc(size * sizeof(short));
    for (i=0; i<size; i++) {
	astats_[stat].value[i] = value;
	astats_[stat].count[i] = 0;
    }

    astats_[stat].update = rate;
    astats_[stat].size = size;
}


GLOBAL void write_array_stat(short stat, short index, unsigned long t)
{
    AddWord(MAKE_EVENT_FIELD(stat) | LONG_INDEX(index));
    AddWord(astats_[stat].value[index]);
    AddWord(t);
    if (BufferFull) WriteBuffer(FALSE);

    astats_[stat].count[index] = 0;
}


GLOBAL int new_metric(const char *name, double val)
{
    Metric *m;
    int i;

    if (!initialized)
      fatal("new_metric invoked before init_statistics");

    if (metric_count == 256) fatal("too many metrics defined");

    /* make sure name is unique */
    for (i=0; i<metric_count; i++)
      if (strcmp(name, metrics_[i].name)==0)
	fatal("Redefinition of metric named `%s'", name);
    for (i=0; i<ametric_count; i++)
      if (strcmp(name, ametrics_[i].name)==0)
	fatal("Redefinition of metric named `%s'", name);

    m = &metrics_[metric_count];
    bzero(m, sizeof(Metric)); 
    m->name = name;
    m->value = val;

    AddWord(EV_INIT_METRIC | MAKE_SHORT_INDEX(metric_count));
    WriteBuffer(FALSE); /* to ensure order relative to putc below */
    for (i=0; name[i] != 0; i++) putc(name[i], event_file_);
    putc(0, event_file_);
    ascii_data = TRUE;

    return(metric_count++);
}

GLOBAL int new_array_metric(const char *name, int size, double val)
{
    ArrayMetric *m;
    int i;

    if (!initialized)
      fatal("new_array_metric invoked before init_statistics");

    if (ametric_count == 256) fatal("too many array metrics defined");
    if (size > 65535) fatal("Array Metric \"%s\" too large.", name);

    /* make sure name is unique */
    for (i=0; i<metric_count; i++)
      if (strcmp(name, metrics_[i].name)==0)
	fatal("Redefinition of metric named `%s'", name);
    for (i=0; i<ametric_count; i++)
      if (strcmp(name, ametrics_[i].name)==0)
	fatal("Redefinition of metric named `%s'", name);

    m = &ametrics_[ametric_count];
    m->value = (float *) malloc(size * sizeof(float));
    m->tmp1 = (float *) malloc(size * sizeof(float));
    m->tmp2 = (float *) malloc(size * sizeof(float));
    m->tmp3 = (float *) malloc(size * sizeof(float));

    m->name = name;
    m->size = size;
    for (i=0; i<size; i++) m->value[i] = val;
    bzero(m->tmp1, size*sizeof(float));
    bzero(m->tmp2, size*sizeof(float));
    bzero(m->tmp3, size*sizeof(float));

    AddWord(EV_INIT_ARRAYMETRIC | MAKE_SHORT_INDEX(ametric_count)
	   | SHORT_VALUE(size));
    WriteBuffer(FALSE); /* to ensure order relative to putc below */
    for (i=0; name[i] != 0; i++) putc(name[i], event_file_);
    putc(0, event_file_);
    ascii_data = TRUE;

    return(ametric_count++);
}


/****************************************************************************/
/* standard metric routines */

GLOBAL double sum_metric_(metric, val)
int metric;
double val;
{
    return(metrics_[metric].value += val);
}

GLOBAL double sum_array_metric_(metric, index, val)
int metric, index;
double val;
{
    return(ametrics_[metric].value[index] += val);
}

/* metrics for running max, min */
GLOBAL double max_metric_(metric, val)
int metric;
double val;
{
    if (val > metrics_[metric].value)
      metrics_[metric].value = val;
    return(metrics_[metric].value);
}

GLOBAL double min_metric_(metric, val)
int metric;
double val;
{
    if (val < metrics_[metric].value)
      metrics_[metric].value = val;
    return(metrics_[metric].value);
}

GLOBAL double max_array_metric_(metric, index, val)
int metric, index;
double val;
{
    if (val > ametrics_[metric].value[index])
      ametrics_[metric].value[index] = val;
    return(ametrics_[metric].value[index]);
}

GLOBAL double min_array_metric_(metric, index, val)
int metric, index;
double val;
{
    if (val < ametrics_[metric].value[index])
      ametrics_[metric].value[index] = val;
    return(ametrics_[metric].value[index]);
}

/* metrics for running average */
/* tmp1 is used to keep the count of values, tmp2 tracks their sum */

GLOBAL double avg_metric_(metric, val)
int metric;
double val;
{
    metrics_[metric].tmp1++;
    metrics_[metric].tmp2 += val;
    return(metrics_[metric].value = metrics_[metric].tmp2 / 
	   metrics_[metric].tmp1);
}

GLOBAL double avg_array_metric_(metric, index, val)
int metric, index;
double val;
{
    ametrics_[metric].tmp1[index]++;
    ametrics_[metric].tmp2[index] += val;
    return(ametrics_[metric].value[index] = ametrics_[metric].tmp2[index] /
	   ametrics_[metric].tmp1[index]);
}

/* metric for running standard deviation, requires an average metric */
/* tracks the standard deviation for the average metric `avg_metric'.
   which should be updated just before this metric.  tmp1 tracks the
   running sum of the squares.  The average, the count, and the running sum
   are read from the average metric */
GLOBAL double stddev_metric_(metric, avg_metric, val)
int metric, avg_metric;
double val;
{
    metrics_[metric].tmp1 += val*val;
    return(metrics_[metric].value =
      (double)sqrt((double)((metrics_[metric].tmp1 - 
			    metrics_[avg_metric].value
			    * metrics_[avg_metric].tmp2)
			   / metrics_[avg_metric].tmp1)));
}

GLOBAL double stddev_array_metric_(metric, avg_met, i, val)
int metric, avg_met, i;
double val;
{
   ametrics_[metric].tmp1[i] += val*val;
   return(ametrics_[metric].value[i] = 
	  (double)sqrt((double)((ametrics_[metric].tmp1[i] - 
				ametrics_[avg_met].value[i] 
				* ametrics_[avg_met].tmp2[i])
			       / ametrics_[avg_met].tmp1[i]))); 
}



/*****************************************************************************
   General event-generation routines
*****************************************************************************/

GLOBAL void Event(evt, value, time)
ulong evt, value, time;
{
    if (IS_SHORT_FORMAT(evt)) {
	AddWord(evt | SHORT_VALUE(value));
    } else {
	AddWord(evt);
	AddWord(value);
    }
    AddWord(time);
    if (BufferFull) WriteBuffer(FALSE);
}


GLOBAL void IndexEvent(evt, index, value, time)
ulong evt, index, value, time;
{
    if (IS_SHORT_FORMAT(evt)) {
	AddWord(evt | MAKE_SHORT_INDEX(index) | SHORT_VALUE(value));
    } else {
	AddWord(evt | LONG_INDEX(index));
	AddWord(value);
    }
    AddWord(time);
    if (BufferFull) WriteBuffer(FALSE);
}

GLOBAL void AtEvent(evt, index_and_value, time)
ulong evt, index_and_value, time;
{
    if (IS_SHORT_FORMAT(evt)) {
	AddWord(evt | LONG_INDEX(index_and_value));
    } else {
	fatal("AtEvent intended only for short-format events\n");
    }
    AddWord(time);
    if (BufferFull) WriteBuffer(FALSE);
}


/* Add a comment to the trace file (for use by something other
   than stats), set type contians the symbolic name of the comment,
   buf contains the comment, and the length is the number of bytes in buf */

GLOBAL void TraceFileComment(const char *type, int index, unsigned long time,
			     const char *buf, int length)
{
    int len;
    unsigned long ev;

    if (type==NULL)
      fatal("TraceFileComment: <type> is NULL");

    len = strlen(type);

    if (len == 0)
      fatal("TraceFileComment: <type> is empty string");
    if (len > 3)
      fatal("TraceFileComment: <type> too long\n\t<type>=%s", type);
    
    if (length > 0xffff)
      fatal("TraceFileComment: buffer length too long, %d bytes", length);
    if (index > 0xffff)
      fatal("TraceFileComment: index out of range [0:0xffff], =%d", index);

    ev = EV_EXIT | type[0] << 16;
    if (len > 1)
	ev |= type[1] << 8;
    if (len > 2)
	ev |= type[2];
	
    AddWord(ev);
    AddWord(time);
    AddWord(index << 16 | length);
    WriteBuffer(FALSE);
    
    if (length > 0) write(fd, buf, length);
    ascii_data = TRUE;
}
