 /**************************************************************************\
 * hash.c, part of simex
 *
 *  David Kotz, 1996
 *
 *  simex is a stripped-down version of stats, in which I tossed all 
 * of the GUI and graph-making stuff.  It's just enough to read events.sim
 * and print out the metrics. 
 *
 * DERIVED FROM
 *
 *
 *                 Proteus Parallel-Architecture Simulator                
 *                Eric A. Brewer  and  Chris N. Dellarocas                
 *                     Laboratory for Computer Science                    
 *                  Massachusetts Institute of Technology                 
 *
 * Module: hash.c  (stats)
 *
 * Description:                                                           
 *     This is the symbol table module used by 'stats'.  It uses a chained
 *     hash table to store the symbols.
 *
 *   1-14-91: Changed to handle a set of hash tables, one for each event
 *     file.
 *
 * Last Modified: $Date: 1996/07/04 03:30:05 $ ($Author: dfk $)
 * Revision $Revision: 1.4 $
 * 
 * Global Functions:                                                      
 *     void InitTable(int max_sets);
 *         Initialize a set of hash tables.
 * 
 *     char *LocateSymbol(char *name, int *token, void **value);
 *         Find symbol 'name' in the current table.  Return NULL
 *         if it is not there, else return the (unique) string for that
 *         symbol, and fill 'value'.
 *
 *     char *LocateSymInTable(int table, char *name, 
 *                            int *token, void**value);
 *         Same as above, but uses hash table `table'.
 * 
 *     char *InsertSymbol(char *name, int token, void *value);
 *         Insert the symbol 'name' into the current table.  Define
 *         and return a unique string for the symbol.  If the symbol is
 *         already in the table, it is an error.
 *
 *     char *InsertSymInTable(int table, char *name,
 *                            int token, void *value);
 *         Same as above, but uses hash table `table'.
 * 
 *     void SetSymbolValue(char *sym, void *value);
 *
 *     void *GetSymbolValue(char *sym);
 *
 *     void HashStats()
 *         Print hash table statistics
 * 
 * Global Variables:
 *     int current_hash_table;
 *                                                                        
 ****************************************************************************
 *   Copyright 1991                                                       
 *   Eric A. Brewer  and  Chris N. Dellarocas                             
 *   Massachusetts Institute of Technology                                
 *                                                                        
 *   Permission to use, copy, modify, and distribute this program         
 *   for any purpose and without fee is hereby granted, provided          
 *   that this copyright and permission notice appear on all copies       
 *   and supporting documentation, the name of M.I.T. not be used         
 *   in advertising or publicity pertaining to distribution of the        
 *   program without specific prior permission, and notice be given       
 *   in supporting documentation that copying and distribution is         
 *   by permission of M.I.T.  M.I.T. makes no representations about       
 *   the suitability of this software for any purpose.  It is pro-        
 *   vided "as is" without express or implied warranty.		          
 ***************************************************************************
 *
 * $Log: hash.c,v $
 * Revision 1.4  1996/07/04 03:30:05  dfk
 * this is a nice working version.
 *
 * Revision 1.3  1996/07/02 02:58:17  dfk
 * completely new command-line interface
 * detailed control over what gets printed
 * ability to use abbreviated names for metrics
 *
 * Revision 1.2  1996/07/01 22:51:44  dfk
 * first really working version
 *
 * Revision 1.1  1996/06/30 21:33:10  dfk
 * Initial revision
 *
 * Revision 1.1  92/09/14  13:33:27  brewer
 * Initial revision
 * 
 * 
 \**************************************************************************/

#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <assert.h>
#include "simex.h"
#include "hash.h"

#define TABLE_SIZE  231

typedef struct eT {
    char *id;
    short token;
    void *value;
    struct eT *next;
} entryType;

GLOBAL int current_hash_table = 0;
static entryType *hash_table[MAX_TABLES][TABLE_SIZE];
static num_symbols=0;

LIBRARY void bzero(char *, unsigned int);

/** INTERNAL ROUTINES **/

/* Store a symbol.
    Copy string to new location, and return the new location.
*/
static char *StoreSymbol(char *sym)
{
    char *new_loc;

    new_loc = malloc(strlen(sym) + 1);
    strcpy(new_loc, sym);
    return new_loc;
}


/* hash table function */
static short HashFunction(char *sym)
{
    unsigned short i, k;
    unsigned long val;

    assert(sym != NULL);
    
    val = (short)sym[0]+1;
    for(i=1; i<8; i++) {
	if (sym[i]==0) break;
	k = sym[i] & 0x3f;
	val *= k + 7;
    }
    return((short)(val % TABLE_SIZE));
}


static entryType *Locate(int table, char *sym)
{
    entryType *e;
    
    e = hash_table[table][HashFunction(sym)];
    while (e != NULL) {
	if (strcmp(sym, e->id)==0) return e;
	e = e->next;
    }
    return NULL;
}


/**  EXTERNAL ROUTINES **/

GLOBAL void InitTable()
{
    bzero((char *)hash_table, TABLE_SIZE*MAX_TABLES*sizeof(char *));
}


/* Locate a symbol with a given name and entry type.
   Returns NULL if there is no such symbol.
*/  
GLOBAL char *LocateSymbol(char *sym, int *token, void **value)
{
    entryType *e;
    
    e = Locate(current_hash_table, sym);
    if (e==NULL) return NULL;
    *token = e->token;
    *value = e->value;
    return e->id;
}

GLOBAL char *LocateSymInTable(int table, char *sym,
			      int *token, void **value)
{
    entryType *e;
    
    e = Locate(table, sym);
    if (e==NULL) return NULL;
    *token = e->token;
    *value = e->value;
    return e->id;
}


GLOBAL char *InsertSymInTable(int table, char *sym,
			      int token, void *value)
{
    entryType *e;
    short bucket;

    e = Locate(table, sym);
    if (e==NULL) {  /* new symbol */
	bucket = HashFunction(sym);
	e = (entryType *) malloc(sizeof(entryType));
	e->id = StoreSymbol(sym);
	e->token = token;
	e->value = value;
	e->next = hash_table[table][bucket];
	hash_table[table][bucket] = e;
	num_symbols++;
        if (debug) fprintf(stderr,
			   "Adding \"%s\" to table. tok=%d, val=%d\n",
			   sym, token, (int)value);
    } else {  /* symbol already present */
	fatal("Duplicate symbol: %s\n", sym);
    }
    return e->id;
}

GLOBAL char *InsertSymbol(char *name, int token, void *value)
{
    return(InsertSymInTable(current_hash_table, name, token, value));
}

GLOBAL void SetSymbolValue(char *sym, void *value)
{
    entryType *e;

    e = Locate(current_hash_table, sym);
    if (e == NULL) 
      fatal("SetSymbolValue called with unknown symbol \"%s\".\n", sym);

    e->value = value;
}

GLOBAL void *GetSymbolValue(char *sym)
{
    entryType *e;

    e = Locate(current_hash_table, sym);
    if (e == NULL) 
      fatal("GetSymbolValue called with unknown symbol \"%s\".\n", sym);

    return(e->value);
}

GLOBAL void HashStats()
{
    int i, histogram[10], count, used, search;
    entryType *ptr;

    if (debug) bzero((char *)histogram, 10*sizeof(int));

    for (i = used = search = 0; i<TABLE_SIZE; i++) {
	count = 0;

	for (ptr = hash_table[0][i]; ptr != NULL; ptr=ptr->next) {
	    used++;
	    search += (++count);
	}
	    
	if (count >= 9) {
	    histogram[9]++;
	    fprintf(stderr, "Hash Bucket %d has %d elements\n",
		    i, count);
	}
	else histogram[count]++;
    }
    if (debug) {
	fprintf(stderr, "\n\nHash function histogram:\n\n");
	for (i=0; i<10; i++)
	  fprintf(stderr, "\t%-3d\t%d\n", i, histogram[i]);
    }
    fprintf(stderr,
	    "\nAverage Symbol Table Depth (1.0 is perfect): %d/%d = %g\n",
	    search, used, search/(float)used);
}
