 /**************************************************************************\
 * load.c, part of simex
 *
 *  David Kotz, 1996
 *
 *  simex is a stripped-down version of stats, in which I tossed all 
 * of the GUI and graph-making stuff.  It's just enough to read events.sim
 * and print out the metrics. 
 *
 * DERIVED FROM
 *
 *                 Proteus Parallel-Architecture Simulator                
 *                Eric A. Brewer  and  Chris N. Dellarocas                
 *                     Laboratory for Computer Science                    
 *                  Massachusetts Institute of Technology                 
 *
 * Module: load.c (stats)
 *
 * Description:                                                           
 *     Load events, symbols; sort events.
 *
 * Last Modified: $Date: 1996/07/04 03:30:05 $ ($Author: dfk $)
 * Revision $Revision: 1.6 $
 * 
 * Global Functions:                                                      
 *     void LoadEvents(char *fname, BOOL display)
 *     void LoadOnlyMetrics(char *fname)
 *     void SortEvents(void)
 *     void LoadEventNames(void)                Event names -> symbol table
 *     int LoadMetrics(char *name)
 *     void LoadParams(char *fname)
 *     void ReadSpecification(char *fname)
 *
 * Global Variables:
 *     ulong *events                            Event data
 *     ulong num_events                         Actual number of events
 *     long event_list_size                     Allocated space for events
 *
 *     char *files[MAX_TABLES]                  file name for each event file
 *     int file_count                           number of event files
 *     metricType *the_metrics[MAX_TABLES]      &metrics for each event file
 *     ametricType *the_ametrics[MAX_TABLES]    &(array metrics) for each file
 *
 *     char **proc_names                        procedure names
 *     char *simulation_title                   Text for simulation title
 *     BOOL profile_data                        True iff profile data present
 *     BOOL params_loaded                       True iff parameters loaded
 *     metricType *metrics                      current metrics
 *     ametricType *ametrics                    current array metrics
 *     
 ****************************************************************************
 *   Copyright 1991                                                       
 *   Eric A. Brewer  and  Chris N. Dellarocas                             
 *   Massachusetts Institute of Technology                                
 *                                                                        
 *   Permission to use, copy, modify, and distribute this program         
 *   for any purpose and without fee is hereby granted, provided          
 *   that this copyright and permission notice appear on all copies       
 *   and supporting documentation, the name of M.I.T. not be used         
 *   in advertising or publicity pertaining to distribution of the        
 *   program without specific prior permission, and notice be given       
 *   in supporting documentation that copying and distribution is         
 *   by permission of M.I.T.  M.I.T. makes no representations about       
 *   the suitability of this software for any purpose.  It is pro-        
 *   vided "as is" without express or implied warranty.		          
 ***************************************************************************
 *
 * $Log: load.c,v $
 * Revision 1.6  1996/07/04 03:30:05  dfk
 * this is a nice working version.
 *
 * Revision 1.5  1996/07/02 02:58:17  dfk
 * completely new command-line interface
 * detailed control over what gets printed
 * ability to use abbreviated names for metrics
 *
 * Revision 1.4  1996/07/01 22:51:44  dfk
 * first really working version
 *
 * Revision 1.3  1996/06/30 21:33:10  dfk
 * NEW for simex
 *
 * Revision 1.2  1996/06/30 20:22:15  dfk
 * *** empty log message ***
 *
 * Revision 1.2  94/12/14  14:49:52  dfk
 * fixed to improve error handling, particularly when the
 * file is not complete (it used to loop infinitely).
 * 
 * Also fixcxed pinting metrics so that large integers are
 * printed as integers rather than in E format.
 * 
 * Revision 1.3  1993/02/26  22:12:51  brewer
 * Added trace-file format version detection.
 *
 * Revision 1.2  92/09/16  13:58:18  brewer
 * fixed byte-reverse detection code
 * 
 * Revision 1.1  92/09/14  13:47:29  brewer
 * Initial revision
 * 
 * 
 \**************************************************************************/

/* maximum trace-file format version number handled by this version of stats */
#define TFF_VERSION 10

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <malloc.h>
#include <string.h>
#include "simex.h"
#include "event.h"
#include "hash.h"

#ifdef sun
#include <unistd.h>
#endif

/* event storage */
GLOBAL ulong *events = NULL;
GLOBAL ulong num_events = 0;

/* default number of events (initially) */
GLOBAL ulong event_list_size = 100000;

/* file stuff for tracking multiple event files */
GLOBAL const char *files[MAX_TABLES];
GLOBAL int file_count = 0;
GLOBAL metricType *metrics = NULL, *the_metrics[MAX_TABLES];
GLOBAL ametricType *ametrics = NULL, *the_ametrics[MAX_TABLES];

GLOBAL char **proc_names;
GLOBAL char *simulation_title = NULL;
GLOBAL BOOL profile_data = FALSE;
GLOBAL BOOL params_loaded = FALSE;

static FILE *fil;
static ulong *ev;
extern FILE *in_fil;
extern int line_num;

#define ADD_EVENT(w) (*ev++ = (w), num_events++)
#define ADD_WORD(t)  (*ev++ = (t))

static char *EventTypeName[256];

static FILE *zopen(const char *filename, char *mode);
static void zclose(FILE *fp, const char *filename);

static char tmpbuf[MAX_LINE_SIZE];

/**************************************************************************/

#define ByteReverse(a) \
   ( ((a) << 24) | (((a) << 8) & 0x00ff0000) | \
     (((a) >> 8) & 0x0000ff00) | ((unsigned long)(a) >>24) )

GLOBAL BOOL byte_reverse = FALSE;


static void AllocateEvents(void)
{
  if (events == NULL) {
    ev = events = (ulong *) malloc(event_list_size * 3 * sizeof(ulong));
    if (verbose)
      fprintf(stderr, "Allocating space for %ld events.\n", event_list_size);
    if (events == NULL) fatal("out of memory for events");
  }
}

static void AllocateMetrics(void)
{
  if (metrics == NULL) {
    metrics = (metricType *)calloc(MAX_METRICS, sizeof(metricType));
    if (metrics == NULL) fatal("out of memory for metrics.\n");
  }

  if (ametrics == NULL) {
    ametrics = (ametricType *)calloc(MAX_AMETRICS, sizeof(ametricType));
    if (ametrics == NULL) fatal("out of memory for array metrics.\n");
  }
}


static ulong LoadWord(FILE *fil)
{
    register ulong w = getw(fil);

    if (w != EOF && byte_reverse) w = ByteReverse(w);
    return w;
}

static char *LoadString(FILE *fil)
{
    int c;		      /* a character */
    char *ptr;		      /* temporary ptr, then malloc'd location */

    ptr = tmpbuf;	      /* read a null-terminated string into tmpbuf */
    do {
	if ((c = getc(fil)) == EOF)
	  if (feof(fil)) fatal("Expecting more input from event file");
	*ptr++ = c;
    } while (c != 0);
    ptr = malloc(ptr - tmpbuf);
    if (ptr == NULL) fatal("Not enough memory for %d-byte string", ptr-tmpbuf);
    strcpy(ptr, tmpbuf);

    return(ptr);
}

static void LoadParamsFromFile(FILE *fil)
{
    char *token, *id, line[MAX_LINE_SIZE];
    int line_num, i;

    line_num = 0;

    while (fgets(tmpbuf, MAX_LINE_SIZE, fil) != NULL) {
        if (strcmp("%%\n",tmpbuf)==0) break;
	line_num++;
	strcpy(line, tmpbuf);
	token = strtok(tmpbuf, " \t\n");
	if (token == NULL) continue;
	if (strcmp(token, "#define") == 0) {
	    /* find parameter */
	    token = strtok(NULL, " \t\n");
	    if (token != NULL) {
		id = token;
		token = strtok(NULL, " \t\n");
		if (token == NULL) {  /* no value => boolean parameter */
		    if (debug) fprintf(stderr, "bool %s = true\n", id);
		    InsertSymbol(id, PARAM, (void *) 1L);
		    continue; /* done, go on to next parameter */
		}
		/* must be integer parameter */
		if (token[i=0] == '-') i=1;
		for(; token[i] != 0; i++)
		  if (!isdigit(token[i])) {
		      fprintf(stderr, "Warning: parameters file (line %d):\n",
			      line_num);
		      fprintf(stderr,
			  "\tnon-numerical definition of parameter \"%s\"\n",
			      id);
		      break;
		  }
		if (debug) fprintf(stderr, "int %s = %ld\n", id, atol(token));
		InsertSymbol(id, PARAM, (void *)atol(token));
	    } else {
		fprintf(stderr,	"Warning: parameters file (line %d):\n",
			line_num);
		fprintf(stderr, "\tMissing parameter name.\n");
	    }
	} else if (strcmp(token, "#undef") == 0) {
	    /* find paramater */
	    token = strtok(NULL, " \t\n");
	    id = token;
	    if (token != NULL) {
		if (debug) fprintf(stderr, "bool %s = false\n", id);
		InsertSymbol(id, PARAM, (long)0);
	    } else {
		fprintf(stderr,	"Warning: parameters file (line %d):\n",
			line_num);
		fprintf(stderr, "\tMissing parameter name.\n");
	    }
	    token = strtok(NULL, " \t\n");
	    if (token != NULL) {
		fprintf(stderr,	"Warning: parameters file (line %d):\n",
			line_num);
		fprintf(stderr,
			"\tvalue assigned to boolean parameter \"%s\"\n",
			id);
	    }
	} else if (strcmp(token, "TITLE") == 0) {
	    /* read simulation title from line, title starts at line[6] */
	    /* set simulation_title if this is the primary events file */
	    if (current_hash_table == 0) {
		/* remove newline, then copy title */
		line[strlen(line) - 1] = 0;
		simulation_title = malloc(strlen(&line[6]));
		strcpy(simulation_title, &line[6]);
	    }
	} else {
	    fprintf(stderr, "Warning: parameters file (line %d):\n",
		    line_num);
	    fprintf(stderr, "\tunrecognizable line:\n");
	    fprintf(stderr, "%s\n", line);
	}
    }
}


/*************************************************************************/

static int LoadVersionNumber(ulong vn, BOOL *byte_rev, BOOL *read_params)
{
    int version = 0;
    
    *read_params = FALSE;

    if ((vn & 0xffff0000) == (unsigned)0xffff0000) {
	*read_params = TRUE;
	if (vn == (unsigned)0xffffffff)
	  version = 0;
	else
	  version = vn & 0xffff;
    } else if ((vn & 0xffff) == 0xffff) {
	if (verbose)
	  fprintf(stderr, "Detected byte-reversed event file.\n");
	*byte_rev = !(*byte_rev);
	*read_params = TRUE;
	vn = ByteReverse(vn);
	version = vn & 0xffff;
    } else {
	fatal("file is not in proper trace-file format!");
    }
    
    if (version > TFF_VERSION) {
	fprintf(stderr, "WARNING: Trace-file format version (%.1f)!\n",
		version/10.0);
	fprintf(stderr, "\tThis version of simex handles versions <= %.1f\n",
		TFF_VERSION/10.0);
    }
    return(version);
}

/*************************************************************************/

GLOBAL void LoadEvents(char *fname, BOOL display)
{
    ulong w, time, value, index, data, name;
    BOOL read_params = FALSE;
    int version = 0;
    union {
	float f;
	long l;
    } val;
    ulong *realloc_at, delta;

    fil = zopen(fname, "r");
    if (fil == NULL) fatal("unable to open event file \"%s\"", fname);
    
    /* allocate memory for events and metrics */
    AllocateMetrics();
    AllocateEvents(); 

    time = 0;

    realloc_at = events + event_list_size*3 - 3;

    w = LoadWord(fil);
    if (feof(fil)) fatal("No data in trace file.");

    version = LoadVersionNumber(w, &byte_reverse, &read_params);

    if (version != 0 && display) {
	printf("Trace File Format Version %.1f\n", version/10.0);
    }	
	
    if (read_params) {
	if (verbose)
	  fprintf(stderr,
		  "Reading parameters from event file \"%s\".\n", fname);
	LoadParamsFromFile(fil);
	params_loaded = TRUE;
	w = LoadWord(fil);
	if (w == EOF)
	  if (feof(fil)) fatal("Events file incomplete!");
	if (version == 0) { /* check for byte reversal based on first event */
	    if ((w & 0xffff) == ((unsigned)EV_START_THREAD >> 24)
		|| (w & 0xffff) == ((unsigned)EV_BUSY >> 24)) {
		fprintf(stderr,
			"Detected old-format byte-reversed event file.\n");
		byte_reverse = !byte_reverse;
		w = ByteReverse(w);
	    }
	}
    }

    
    for (;; w = LoadWord(fil)) {
	if (w == EOF)
	  if (feof(fil)) break;

	if (ev >= realloc_at) {
	    /* out of space => double table size */
	    delta = ev - events;
	    if (verbose)
	      fprintf(stderr, "Allocating space for %ld more events\n",
		      event_list_size);
	    event_list_size *= 2;
	    events = (ulong *) realloc(events,
				       3*event_list_size*sizeof(ulong));
	    if (events == NULL) fatal("out of memory for events");
	    ev = events + delta;
	    realloc_at = events + event_list_size*3 - 3;
	}

	if (IS_SHORT_FORMAT(w)) {
	    index = SHORT_INDEX(w);
	    value = SHORT_VALUE(w);
	} else {
	    index = LONG_INDEX(w);
	    value = LoadWord(fil);
	    if (value == EOF)
	      if (feof(fil)) fatal("Expecting more input from event file");
	}

	switch (EVENT_FIELD(w)) {
	  case EV_INIT_METRIC:
	    metrics[index].name = LoadString(fil);
	    metrics[index].value = 0.0;
	    InsertSymbol(metrics[index].name, METRIC, &metrics[index]);
	    if (display) printf("Init Metric \"%s\".\n", metrics[index].name);
	    break;
	  case EV_INIT_ARRAYMETRIC:
	    ametrics[index].name = LoadString(fil);
	    ametrics[index].size = value;
	    ametrics[index].value = (float *)calloc(value, sizeof(float));
	    InsertSymbol(ametrics[index].name, AMETRIC, &ametrics[index]);
	    if (display)
	      printf("Init Array Metric \"%s\"[%lu].\n", 
		     ametrics[index].name, value);
	    break;
	  case EV_METRIC:
	    data = LoadWord(fil);
	    if (data == EOF)
	      if (feof(fil)) fatal("Expecting more input from event file");

	    val.l = data;
	    metrics[index].value = val.f;
	    if (display)
	      printf("Metric(%s): %g\n", metrics[index].name, val.f);
	    break;
	  case EV_ARRAYMETRIC:
	    data = LoadWord(fil);
	    if (data == EOF)
	      if (feof(fil)) fatal("Expecting more input from event file");

	    val.l = data;
	    ametrics[index].value[value] = val.f;
	    if (display) printf("Array Metric(%s)[%lu]: %g\n",
				ametrics[index].name, value, val.f);
	    break;
	  case EV_INIT_PROF:
	    ametrics[NUMCALLS_METRIC].name = "Number of Calls";
	    ametrics[NUMCALLS_METRIC].size = value;
	    ametrics[NUMCALLS_METRIC].value =
	      (float *)malloc(value*sizeof(float));
	    ametrics[PROF_METRIC].name = "Total Time";
	    ametrics[PROF_METRIC].size = value;
	    ametrics[PROF_METRIC].value = (float *)malloc(value*sizeof(float));
	    proc_names = (char **)malloc(value * sizeof(char *));
	    profile_data = TRUE;
	    InsertSymbol("profile calls", AMETRIC, &ametrics[NUMCALLS_METRIC]);
	    InsertSymbol("profile time", AMETRIC, &ametrics[PROF_METRIC]);
	    if (display)
	      printf("Profiling Init: %ld procedures\n", value);
	    break;
	  case EV_PROF:
	    proc_names[value] = LoadString(fil);
	    if (strlen(proc_names[value]) > 200)
	      fatal("Invalid trace file format (byte-reversed?)\n");
	    ametrics[NUMCALLS_METRIC].value[value] = (float) LoadWord(fil);
	    ametrics[PROF_METRIC].value[value] = (float) LoadWord(fil);
	    if (display) printf("%s: %g call%s, %g cycles  = %g cycles/call\n",
			      proc_names[value],
			      ametrics[NUMCALLS_METRIC].value[value],
			      (ametrics[NUMCALLS_METRIC].value[value] !=
			        (float)1.0)?"s":"",
			      ametrics[PROF_METRIC].value[value],
			      ametrics[PROF_METRIC].value[value]
			      / ametrics[NUMCALLS_METRIC].value[value]); 
	    break;
	  case EV_EXIT:
	    time = LoadWord(fil);
	    name = LONG_INDEX(w);
	    if (name != 0) {  /* dectected comment */
		int len; ulong tmp;
		
		tmp = LoadWord(fil);
		len = tmp & 0xffff;
		if (display) {
		    char typ[4];
		    typ[3] = 0;
		    typ[2] = name & 0xff; name = name >> 8;
		    typ[1] = name & 0xff; name = name >> 8;
		    typ[0] = name & 0xff;
		    printf("%lu: Detected comment \"%s\" index=%lu ...",
			   time, typ, tmp >> 16);
		}
		fseek(fil, len, SEEK_CUR);
		if (display) printf(" completed\n");
	    } else {
		ADD_EVENT(w);
		ADD_WORD(value);
		ADD_WORD(time);
		if (display) printf("%lu: Simulation done\n", time);
	    }
	    break;
	  default:
	    ADD_EVENT(w);
	    ADD_WORD(value);
	    time = LoadWord(fil);
	    if (time == EOF)
	      if (feof(fil)) fatal("Expecting more input from events file");
	    ADD_WORD(time);
	    if (display) {
		char *ptr = EventTypeName[EVENT_TO_INT(w)];
		if (ptr != NULL) {
		    printf("%lu: %s[%ld] = %lu\n", time, ptr, index, value);
		} else {
		    printf("%lu: Unknown Type 0x%lx [%ld] = %lu\n",
			   time, EVENT_FIELD(w), index, value);
		}
	    }
	}
    }
    ADD_EVENT(EV_EXIT);
    ADD_WORD(0);
    ADD_WORD(time+1);
    zclose(fil, fname);
    if (verbose)
      fprintf(stderr, "Loaded %lu events.\n", num_events);
}


GLOBAL void LoadOnlyMetrics(const char *fname) {
    ulong w, time, value, index, data, name;
    BOOL read_params = FALSE;
    int version;
    union {
	float f;
	long l;
    } val;

    fil = zopen(fname, "r");
    if (fil == NULL) fatal("Unable to open event file \"%s\"", fname);

    w = LoadWord(fil);
    if (feof(fil)) fatal("No data in trace file.");

    version = LoadVersionNumber(w, &byte_reverse, &read_params);

    AllocateMetrics();

    if (read_params) {
	if (verbose)
	  fprintf(stderr,
		  "Reading parameters/metrics from file \"%s\".\n", fname);
	LoadParamsFromFile(fil);
	params_loaded = TRUE;
	w = LoadWord(fil);
    } else {
        fprintf(stderr, "Missing parameters for file \"%s\".\n", fname);
    }

    for (;; w = LoadWord(fil)) {
	if (w == EOF)
	  if (feof(fil)) break;

	if (IS_SHORT_FORMAT(w)) {
	    index = SHORT_INDEX(w);
	    value = SHORT_VALUE(w);
	} else {
	    index = LONG_INDEX(w);
	    value = LoadWord(fil);
	    if (value == EOF)
	      if (feof(fil)) fatal("Expecting more input from events file");
	}

	switch (EVENT_FIELD(w)) {
	  case EV_INIT_METRIC:
	    metrics[index].name = LoadString(fil);
	    metrics[index].value = 0.0;
	    InsertSymbol(metrics[index].name, METRIC, &metrics[index]);
	    if (debug) fprintf(stderr, "Init Metric \"%s\".\n", 
			       metrics[index].name);
	    break;
	  case EV_INIT_ARRAYMETRIC:
	    ametrics[index].name = LoadString(fil);
	    ametrics[index].size = value;
	    ametrics[index].value = (float *)calloc(value, sizeof(float));
	    InsertSymbol(ametrics[index].name, AMETRIC, &ametrics[index]);
	    if (debug) fprintf(stderr, "Init Array Metric \"%s\"[%lu].\n",
			       ametrics[index].name, value);
	    break;
	  case EV_METRIC:
	    data = LoadWord(fil);
	    if (data == EOF)
	      if (feof(fil)) fatal("Expecting more input from event file");

	    val.l = data;
	    metrics[index].value = val.f;
	    if (debug) fprintf(stderr, "Metric(%s): %g\n",
			       metrics[index].name, val.f);
	    break;
	  case EV_ARRAYMETRIC:
	    data = LoadWord(fil);
	    if (data == EOF)
	      if (feof(fil)) fatal("Expecting more input from event file");

	    val.l = data;
	    ametrics[index].value[value] = val.f;
	    if (debug) fprintf(stderr, "Array Metric(%s)[%lu]: %g\n",
			       ametrics[index].name, value, val.f);
	    break;
	  case EV_INIT_PROF:
	    break;
	  case EV_PROF:
	    free(LoadString(fil));
	    LoadWord(fil);
	    LoadWord(fil);
	    break;
	  case EV_EXIT:
	    time = LoadWord(fil);
	    name = LONG_INDEX(w);
	    if (name != 0) {  /* dectected comment */
		int len; ulong tmp;
		
		tmp = LoadWord(fil);
		len = tmp & 0xffff;
		fseek(fil, len, SEEK_CUR);
	    }
	    break;
	  default:
	    time = LoadWord(fil);
	    if (time == EOF)
	      if (feof(fil)) fatal("Expecting more input from events file");
	}
    }
    zclose(fil, fname);
}

typedef struct {
    unsigned long type;
    unsigned long value, time;
} EventType;

static int event_compare(EventType *l, EventType *r)
{
    if (l->time > r->time) {
	return(1);
    } else if (l->time < r->time) {
	return(-1);
    } else return(0);
}


GLOBAL void SortEvents(void)
{
    unsigned int forward_index;
    int backward_index;
    int insert_index;
    EventType *the_events, event_to_be_moved;
    unsigned long time, prevtime;

    if (is_sorted) return;

    the_events = (EventType *) events;
    if (verbose)
      fprintf(stderr, "Sorting %lu events.\n", num_events);

    if (AlternativeSort) {
	qsort(the_events, num_events, sizeof(EventType), event_compare);
    } else {
	prevtime = the_events->time;
	
	for (forward_index = 1; forward_index < num_events; forward_index++)
	  {
	      time = the_events[forward_index].time;
	      
	      if (time >= prevtime) {
		  prevtime = time;
		  continue;
	      }
	      
	      event_to_be_moved = the_events[forward_index];
	      insert_index = 0;
	      
	      for (backward_index = forward_index - 2;
		   backward_index >= 0;
		   backward_index = backward_index - 2)
		if (the_events[forward_index].time >=
		    the_events[backward_index].time)
		  {
		      if (the_events[forward_index].time >=
			  the_events[backward_index+1].time)
			insert_index = backward_index + 2;
		      else
			insert_index = backward_index + 1;
		      break;
		  }
	      
	      for (backward_index = forward_index-1;
		   backward_index >= insert_index;
		   backward_index--)
		the_events[backward_index+1] = the_events[backward_index];
	      the_events[insert_index] = event_to_be_moved;
	  }
    }
    is_sorted = TRUE;
}


/**************************************************************************/

#define ASSERT_NAME(new_name, reserved_name, event, filename) \
  if (strcmp(new_name, reserved_name)!=0) \
    fatal("%s=0x%x redefined to %s in file %s",\
	  reserved_name, event, new_name, filename);

/* load event type definitions from file <fil> */

static void load_event_names(FILE *fil, char *filename)
{
    int token;
    char *ptr;
    ulong event, reserved;
    char *eventcode;

    while (fgets(tmpbuf, MAX_LINE_SIZE, fil) != NULL) {
	ptr = strtok(tmpbuf, " \t");
	if (strcmp("#define", ptr) == 0) {
	    ptr = strtok(NULL, " \t");
	    if (ptr == NULL) {
		fprintf(stderr, "Warning: malformed #define in file \"%s\"\n",
			filename);
		continue;
	    }

	    eventcode = strtok(NULL, "\n \t");
	    if (eventcode == NULL) {
		fprintf(stderr, "Warning: empty #define %s in file \"%s\"\n",
			ptr, filename);
		continue;
	    }

	    event = strtoul(eventcode, NULL, 0);

	    if (debug)
	      fprintf(stderr, "LoadEvent: %s = 0x%lx\n", ptr, event);

	    /* check for redefinition of reserved event TYPES */
	    switch (event) {
	      case EV_EXIT:
		ASSERT_NAME(ptr, "EV_EXIT", event, filename);
		break;
	      case EV_INIT_METRIC:
		ASSERT_NAME(ptr, "EV_INIT_METRIC", event, filename);
		break;
	      case EV_INIT_ARRAYMETRIC:
		ASSERT_NAME(ptr, "EV_INIT_ARRAYMETRIC", event, filename);
		break;
	      case EV_INIT_PROF:
		ASSERT_NAME(ptr, "EV_INIT_PROF", event, filename);
		break;
	      case EV_PROF:
		ASSERT_NAME(ptr, "EV_PROF", event, filename);
		break;
	      case EV_METRIC:
		ASSERT_NAME(ptr, "EV_METRIC", event, filename);
		break;
	      case EV_ARRAYMETRIC:
		ASSERT_NAME(ptr, "EV_ARRAYMETRIC", event, filename);
		break;
	      default: /* check for redefinition of user type */
		if (EventTypeName[EVENT_TO_INT(event)] != NULL) {
		    fprintf(stderr,
			    "Warning: Event type %s=0x%lx000000 redefined ",
			    EventTypeName[EVENT_TO_INT(event)],
			    EVENT_TO_INT(event));
		    fprintf(stderr, "as %s in file \"%s\"\n",
			    ptr, filename);
		}
		break;
	    }

	    if ((event & 0xffffff) != 0)
	      fatal("Event type %s=0x%lx has 1's in bottom 24 bits",
		    ptr, event);

	    event = EVENT_TO_INT(event);

	    /* check for redefinition of reserved event NAMES */
	    if (LocateSymbol(ptr, &token, (void**)&reserved) != NULL) {
		if (event != reserved  && token == EVT)
		  fatal("Reserved type %s=0x%x000000 redefined in file \"%s\"",
			ptr, reserved, filename);
	    } else {
		EventTypeName[event] = InsertSymbol(ptr, EVT, (void *)event);
	    }
	}
    }
}


GLOBAL void LoadEventNames(void)
{
    FILE *fil;
    ulong event;

    fil = fopen(types_filename, "r");
    if (fil == NULL)
      fatal("Unable to file \"%s\".\n", types_filename);

    /* insert reserved symbols */
    event = EVENT_TO_INT(EV_EXIT);
    EventTypeName[event] = InsertSymbol("EV_EXIT", EVT, (void *)event);

    event = EVENT_TO_INT(EV_INIT_METRIC);
    EventTypeName[event] = InsertSymbol("EV_INIT_METRIC", EVT, (void *)event);
    
    event = EVENT_TO_INT(EV_INIT_ARRAYMETRIC);
    EventTypeName[event] = InsertSymbol("EV_INIT_ARRAYMETRIC", EVT,
					(void *)event);

    event = EVENT_TO_INT(EV_INIT_PROF);
    EventTypeName[event] = InsertSymbol("EV_INIT_PROF", EVT, (void *)event);

    event = EVENT_TO_INT(EV_PROF);
    EventTypeName[event] = InsertSymbol("EV_PROF", EVT, (void *)event);

    event = EVENT_TO_INT(EV_METRIC);
    EventTypeName[event] = InsertSymbol("EV_METRIC", EVT, (void *)event);

    event = EVENT_TO_INT(EV_ARRAYMETRIC);
    EventTypeName[event] = InsertSymbol("EV_ARRAYMETRIC", EVT, (void *)event);
    
    load_event_names(fil, types_filename);
    
    fclose(fil);

    fil = fopen(usertypes_filename, "r");
    if (fil == NULL) {
	if (strcmp(usertypes_filename, "user-events.h") != 0)
	  fprintf(stderr,
		  "\n*** WARNING: unable to find user types file \"%s\".\n\n",
		  usertypes_filename);
	return;
    }

    if (verbose)
      fprintf(stderr, "Loading user types file \"%s\".\n", usertypes_filename);
    load_event_names(fil, usertypes_filename);
    fclose(fil);
}


/**************************************************************************/

GLOBAL int LoadMetrics(const char *name)
{
    int i;

    AllocateMetrics();

    if (file_count == 0) {
	files[0] = event_filename;
	the_metrics[0] = metrics;
	the_ametrics[0] = ametrics;
	file_count++;
    }

    for(i=0; i<file_count; i++)
      if (strcmp(files[i], name)==0) return(i);

    if (file_count == MAX_TABLES)
      fatal("Exceeded maximum number of event files (%d)", MAX_TABLES);

    files[file_count] = name;

    metrics = NULL;
    ametrics = NULL;
    AllocateMetrics();
    the_metrics[file_count] = metrics;
    the_ametrics[file_count] = ametrics;

    SETFILE(file_count);
    LoadOnlyMetrics(name);
    SETFILE(0);

    file_count++;

    return(file_count - 1);
}


GLOBAL void LoadParams(const char *fname)
{
    FILE *fil;

    fil = fopen(fname, "r");
    if (fil == NULL)
      fatal("Unable to open parameter file \"%s\".\n", fname);
    if (verbose)
      fprintf(stderr, "Reading parameter file \"%s\".\n", fname);

    LoadParamsFromFile(fil);

    fclose(fil);
}



/* DFK */
static FILE *
zopen(const char *filename, char *mode)
{
    const char *endfilename;
    int compressed;
    char *commandline = NULL;
    FILE *fp;

    if (filename == NULL || mode == NULL)
      return(NULL);

    endfilename = filename + strlen(filename);
    compressed = 
      (strcmp(endfilename-3, ".gz") == 0) || 
      (strcmp(endfilename-2, ".Z") == 0) || 
      (strcmp(endfilename-2, ".z") == 0);

    if (compressed) {
	commandline = malloc(strlen(filename) + 20);
	if (commandline == NULL)
	  return(NULL);
    }

    if (strcmp(mode, "r") == 0) {
	if (compressed) {
	    sprintf(commandline, "gunzip -c %s", filename);
	    fp = popen(commandline, "r");
	} else
	  fp = fopen(filename, "r");
    } else if (strcmp(mode, "w") == 0) {
	if (compressed) {
	    sprintf(commandline, "gzip -c > %s", filename);
	    fp = popen(commandline, "w");
	} else
	  fp = fopen(filename, "w");
    } else {
	fatal("Unable to zopen file '%s' with mode '%s'\n", filename, mode);
    }

    if (commandline != NULL)
      free(commandline);

    return(fp);
}

/* DFK */
static void
zclose(FILE *fp, const char *filename)
{
    const char *endfilename;
    int compressed;

    if (filename == NULL)     /* huh? */
      fclose(fp);

    endfilename = filename + strlen(filename);
    compressed = 
      (strcmp(endfilename-3, ".gz") == 0) || 
      (strcmp(endfilename-2, ".Z") == 0) || 
      (strcmp(endfilename-2, ".z") == 0);

    if (compressed)
      pclose(fp);
    else
      fclose(fp);
}
