/* @TITLE "buflist.h: a linked-list of MBUFFERs"*/
/* 
 * buflist: a data structure that represents a double-linked list of
 * MBUFFERs. Useful in the IOP caches.  The entire structure is exposed,
 * for perusal, but do not manipulate the links directly; use the
 * functions provided.
 *
 * Part of 
 *           The STARFISH Parallel file-system simulator
 *      (Simulation Tool for Advanced Research in File Systems)
 *
 *                              David Kotz
 *                          Dartmouth College
 *                             Version 3.0
 *                             October 1996
 *                         dfk@cs.dartmouth.edu
 */

/* $Id: buflist.h,v 3.0 1996/10/18 06:05:51 dfk RELEASE3 dfk $ */

#ifndef BUFLIST_H
#define BUFLIST_H

#include "condition.h"
#include "protocol.h"	      /* for MBUFFER */

typedef struct BufListEntry_s BUFLISTENTRY;
struct BufListEntry_s {
    MBUFFER *mbuf;	      /* a buffer is associated with this entry */
    BUFLISTENTRY *next, *prev; /* within the buffer list */

    ulong nowblock;	      /* which file block is in mbuf? */
    ulong futureblock;	      /* which file block do some WANT in mbuf? */

    int claim;		      /* # threads who have a claim on this entry */
    int firstTID;	      /* TID of the first waiter */
    CONDITION *qp;	      /* used for other waiting threads */
};

/* A data structure representing a whole buffer list */
typedef struct BufList_s BUFLIST;
struct BufList_s {
    char *debugname;	      /* name of this list */
    BUFLISTENTRY *first, *last;	/* begin and end of list */
};

extern void BufListInit(BUFLIST *bl, char *debugname); /* initialize */
extern void BufListFree(BUFLIST *bl); /* free contents */
extern boolean BufListEmpty(BUFLIST *bl); /* is this BufList empty? */

extern BUFLISTENTRY *BufListWrap(MBUFFER *mbuf); /* make ble from mbuf */

extern BUFLISTENTRY *
BufListAlloc(
	     MBUFFER *mbuf,   /* a buffer is associated with this entry */
	     BUFLISTENTRY *next, /* within the buffer list */
	     BUFLISTENTRY *prev, /* within the buffer list */

	     ulong nowblock,  /* which file block is in mbuf? */
	     ulong futureblock, /* which file block do some WANT in mbuf? */
	     
	     int claim,	      /* # threads who have a claim on this entry */
	     int firstTID,    /* TID of the first waiter */
	     CONDITION *qp);  /* used for other waiting threads */
extern void BufListFreeEntry(BUFLISTENTRY *ble);

extern BUFLISTENTRY *BufListLookup(BUFLIST *bl, ulong block); /* lookup block*/

extern void BufListAddFront(BUFLIST *bl, BUFLISTENTRY *ble);
extern void BufListAddLast(BUFLIST *bl, BUFLISTENTRY *ble);
extern void BufListMoveToFront(BUFLIST *bl, BUFLISTENTRY *ble);

extern void BufListRemove(BUFLIST *bl, BUFLISTENTRY *ble);
extern BUFLISTENTRY *BufListRemoveLast(BUFLIST *bl); /* remove tail block */
extern BUFLISTENTRY *BufListOnlyFuture(BUFLIST *bl); /* remove first future-only */

extern void BufListPrint(BUFLIST *bl); /* print the list */
extern void BufListPrintDetail(BUFLIST *bl); /* print the list in detail */

#endif BUFLIST_H
