/*  diskdevices.h
 *
 *  This file contains all the disk parameters used in the simulation.
 *  The purpose of keeping this file around is to keep diskdevice.c a
 *  little cleaner.
 *
 * Part of 
 *           The STARFISH Parallel file-system simulator
 *      (Simulation Tool for Advanced Research in File Systems)
 *
 *                              David Kotz
 *                          Dartmouth College
 *                             Version 3.0
 *                             October 1996
 *                         dfk@cs.dartmouth.edu
 */

#ifndef DISKDEVICES_H
#define DISKDEVICES_H

struct DiskRegion
{
  ulong logicalstart;          /* starting and finishing sector of a */
  ulong logicalend;	       /* region in logical space */	
  ulong physicalstart;	       /* starting and finishing sector */
  ulong physicalend;	       /* of a region in physical space */	
};


/* @SUBTITLE "HP 97560" */
/* ****************************************************** */
/* ****************** HP 97560 *********************** */
#ifdef HP97560

/* These disk parameters extracted from (ruemmler:model2, p17) */
/* The disk parameters in this file is extracted from the data sheet
   for HP97560 */

#define IMMEDIATE_REPORT      /* supports "immediate-reported writes" */

/* These disk parameters extracted from (ruemmler:model2, p17) */

#define SECTOR_SIZE 	      512              /* in bytes */
#define NSECTORS              72	      /* sectors per track */
#define NTRACKS_PER_CYL       19              /* tracks per cylinder */
/* see below for total tracks and sectors */

/* number of logical, thus physical regions on the disk */
#define NUM_REGIONS         3

/* Physical sector number of start and end of each region, INCLUSIVE */
#define Sregion0 ((   1 * NTRACKS_PER_CYL +  4) * NSECTORS    )
#define Eregion0 (( 646 * NTRACKS_PER_CYL +  4) * NSECTORS - 1)
#define Sregion1 (( 654 * NTRACKS_PER_CYL +  0) * NSECTORS    )
#define Eregion1 ((1298 * NTRACKS_PER_CYL + 19) * NSECTORS - 1)
#define Sregion2 ((1308 * NTRACKS_PER_CYL +  0) * NSECTORS    )
#define Eregion2 ((1952 * NTRACKS_PER_CYL + 19) * NSECTORS - 1)

/* length of each region, in sectors */
#define Lregion0	(Eregion0 - Sregion0 + 1)
#define Lregion1	(Eregion1 - Sregion1 + 1)
#define Lregion2	(Eregion2 - Sregion2 + 1)

/* data structure that keeps translation information between physical
   and logical disk sector layout */
struct DiskRegion Regions[NUM_REGIONS] =
{
  { 0, Lregion0 - 1, Sregion0, Eregion0 },
  
  { Lregion0, (Lregion0 + Lregion1 - 1), Sregion1, Eregion1 },

  { (Lregion0 + Lregion1), (Lregion0 + Lregion1 + Lregion2 - 1),
      Sregion2, Eregion2 }
};
/* This boils down to the following (in sectors)
 *              Lstart   Lend    Pstart   Pend
 * Regions[0]:       0   882359    1656   884015
 * Regions[1]:  882360  1764719  894672  1777031
 * Regions[2]: 1764720  2647079 1789344  2671703
 */

/* Total number of sectors and tracks usable FOR DATA */
#define TOTAL_NSECTORS (Lregion0 + Lregion1 + Lregion2)
#define NTRACKS (TOTAL_NSECTORS / NSECTORS)



/* macro to convert from sector number to cylinder number */
/* physical sector numbers only! */
#define SECTOR2CYLINDER(sect)  (ulong) ((sect / NSECTORS) / NTRACKS_PER_CYL)

/* macro to convert from sector number to track number */
/* physical sector numbers only! */
#define SECTOR2TRACK(sect)     (ulong) (sect / NSECTORS)

/* macro to convert from track number to cylinder number */
/* physical track numbers only! */
#define TRACK2CYLINDER(track)     (ulong) (track / NTRACKS_PER_CYL)

/* cylinder skew of the disk in sectors */
#define CYLINDERSKEW       18
/* track skew of the disk in sectors */
#define TRACKSKEW           8

/* =============== Constants with Trailing _t are in TICS ================ */

/* number of TICS needed to send a command to bus */
/* from HP manual, assume we always use the extended 10-byte command */
#define COMMANDTIME_t      (10. / BUSSPEED  * TICSperMSEC)

/* number of TICS needed to send a done ("status") command to host */
#define DONETIME_t         (1. / BUSSPEED  * TICSperMSEC)

/* number of bytes per millisecond (SCSI-II max speed = 10 MB/sec) */
#define BUSSPEED           (10.0 * 1024.0 * 1024.0 / 1000.0) 

/* read and write fence size : */
#define READFENCE_kb        64
#define READFENCE          (READFENCE_kb * 1024 / SECTOR_SIZE) /* #slots */
#define WRITEFENCE_kb       64	
#define WRITEFENCE         (WRITEFENCE_kb * 1024 / SECTOR_SIZE) /* #slots */

/* time needed to acquire the bus (50 microseconds) */
#define BUSGRABTIME_t      (50.0 / 1000.0 * TICSperMSEC)

/* controller overhead time for READS */
/* manual says <1msec, but Wilkes says really 2.2msec */
#define CTRLOVHDREAD_t     (2.2 * TICSperMSEC)

/* controller overhead time for READS */
/* manual says <1msec, but Wilkes says really 2.2msec */
#define CTRLOVHDWRITE_t    (2.2 * TICSperMSEC)

/* disk rotations per minute */
#define DISKRPM             4002

/* number of bytes per millisecond (HP 97560 - 1.3 GB) 
   = (rpm * min/sec * sec/msec * sect/track * bytes/sector) */
/* this is 2.345 MB/s (in true MB) */
#define DISKSPEED          (DISKRPM / 60.0 / 1000.0 * NSECTORS * SECTOR_SIZE)

/* rotation time: Amount of time needed for the disk to rotate 1 full cycle 
   = 1/rpm * 60 sec/min * 1000 msec/sec * TICSperMSEC */
#define ROT_TIME_t         (1.0 /DISKRPM * 60 * 1000 * TICSperMSEC) 

/* track switch time (during transfer) */
#define TRACK_SWITCH_t     (ROT_TIME_t * CYLINDERSKEW / NSECTORS) 

/* head switch time (during transfer) */
#define HEAD_SWITCH_t      (ROT_TIME_t * TRACKSKEW / NSECTORS)

/* number of zones on the disk: assuming each zone has the same number
   of cylinders */
#define NUM_ZONES           1

/* size of disk controller's cache, in sectors */
#define CACHE_kb            128
#define CACHE_SLOTS        (CACHE_kb * 1024 / SECTOR_SIZE)

/* returns the finish time for a single sector transfer starting at t */
#define DSECTORXFER_t(t)   ((double) SECTOR_SIZE / DISKSPEED * TICSperMSEC + \
			    (t))

/* returns the finish time for a single sector transfer across the bus */
#define BSECTORXFER_t(t)   ((double) SECTOR_SIZE / BUSSPEED * TICSperMSEC + \
			    (t))

#endif HP97560

/***********************************************************************/
/* @SUBTITLE "Trivial disk model" */
#ifdef TRIVIAL_DISK
# define TOTAL_NSECTORS	DISK_NSECTORS
# define SECTOR_SIZE 	DISK_SECTOR_SIZE
# define NTRACKS    	DISK_NSECTORS
# define NSECTORS    	1     /* per track */
#endif

#endif /* DISKDEVICES_H */
