/* @TITLE "diskdriver.h: Definitions for disk driver"*/
/* 
 * Interface for the disk-driver module.
 * macros:
 *   OFFSET2BLOCK
 *   OFFSET_IN_BLOCK
 *   FILEOFFSET
 *   BLOCK2DISK
 *   BLOCK_IN_DISK
 *   FILEBLOCK
 *   DISK2PROC
 *   DISK_IN_PROC
 *   DISKS_PER_PROC
 *   GLOBALDISK
 * functions:
 *  DiskDriverInit
 *  DiskDriverRead
 *  DiskDriverWrite
 *  DiskDriverRewrite
 *  DiskDriverIsRead
 *  DiskDriverWait
 *  DiskDriverSync
 *  DiskDriverPreread
 *  DiskDriverReadNext
 *  DiskDriverPrewrite
 *  DiskDriverWhatNext
 *  DiskDriverWriteNext
 *
 * Part of 
 *           The STARFISH Parallel file-system simulator
 *      (Simulation Tool for Advanced Research in File Systems)
 *
 *                              David Kotz
 *                          Dartmouth College
 *                             Version 3.0
 *                             October 1996
 *                         dfk@cs.dartmouth.edu
 */

/* $Id: diskdriver.h,v 3.0 1996/10/18 06:05:51 dfk RELEASE3 dfk $ */
/* Based on disk.h,v 1.33 93/12/24 12:40:44 dfk GENERALWORKS */

#ifndef DISKDRIVER_H
#define DISKDRIVER_H

#include "userdata.h"
#include "disk.h"      /* parameters of interest outside the disk */

/* translate an offset into a block number, and byte in block */
#define OFFSET2BLOCK(offset) ((ulong)(offset) >> LOG_BLOCK_SIZE)
#define OFFSET_IN_BLOCK(offset) ((ulong)(offset) & BLOCK_SIZE_MASK)

/* translate a block number and offset into a file offset number */
/* this reverses OFFSET2BLOCK and OFFSET_IN_BLOCK */
#define FILEOFFSET(block, offset) (((block) << LOG_BLOCK_SIZE) | (offset))

/* translate a block number into a disk number, and block on disk */
#define BLOCK2DISK(block) ((int)((block) % NO_OF_DISKS))
#define BLOCK_IN_DISK(block) ((ulong)((block) / NO_OF_DISKS))

/* translate a disk/diskblock number into a file block number */
/* this reverses BLOCK2DISK and BLOCK_IN_DISK */
#define FILEBLOCK(disk, diskblock) ((ulong)(disk) + (diskblock) * NO_OF_DISKS)

/* translate a disk number into an I/O proc number */
#define DISK2PROC(disk) (IOP2PROC((disk) % Nio))
#define DISK_IN_PROC(disk) ((int)((disk) / Nio))
#define DISKS_PER_PROC ceildiv(NO_OF_DISKS, Nio) /* at most */
#define DISKS_THIS_PROC(IOPnum) \
    (NO_OF_DISKS / Nio + (IOPnum < NO_OF_DISKS % Nio))

/* translate an IOP number and local disk number into a disk number */
/* this reverses DISK2PROC and DISK_IN_PROC */
#define GLOBALDISK(IOPnum, localdisk) ((int)(Nio * (localdisk) + (IOPnum)))

/* TYPES */
typedef struct disk_request DiskRequest;

/* FUNCTIONS */
extern void DiskDriverInit(int IOPnum);
extern void DiskDriverDone(int IOPnum);

/* This is one interface, for block-by-block requests */
extern DiskRequest *DiskDriverRead(int disk, ulong diskblock, UserData buffer);
extern DiskRequest *DiskDriverWrite(int disk, ulong diskblock, UserData buffer);
extern void         DiskDriverRewrite(DiskRequest *dreq);
extern boolean      DiskDriverIsRead(DiskRequest *dreq);
extern void         DiskDriverWait(DiskRequest *dreq);
extern void         DiskDriverSync(int disk);

/* the two components of DiskDriverWait, for special cases */
extern void DiskDriverWaitNoFree(DiskRequest *req);
extern void DiskDriverFree(DiskRequest *req);
/* this changes the TID waiting for req, to some other TID */
extern void DiskDriverWaitTID(DiskRequest *req, int tid);


/* This is another interface, for a predetermined set of reads */
extern void DiskDriverPreread(int nbufs, ulong firstblock, ulong lastblock);
extern boolean DiskDriverReadNext(int *disk, ulong *diskblock, UserData buffer);

/* This is another interface, for a predetermined set of writes */
extern void DiskDriverPrewrite(int nbufs, ulong firstblock, ulong lastblock);
extern DiskRequest *DiskDriverWhatNext(int *disk, ulong *diskblock);
extern void DiskDriverWriteNext(DiskRequest *dreq, UserData buffer);

#endif /* DISKDRIVER_H */
