/* lutest.c - DFK 10/20/88, 10/94
 * 
 * LU factorization without pivoting
 *   reads COLUMN-MAJOR binary matrices; the input matrix is in a file,
 * whose name is the argument,  and the supposedly correct answer is 
 * from stdin.  We compute the real answer and compare.
 * 
 * usage: lutest input_matrix < supposed_answer
 *
 * Part of 
 *           The STARFISH Parallel file-system simulator
 *      (Simulation Tool for Advanced Research in File Systems)
 *
 *                              David Kotz
 *                          Dartmouth College
 *                             Version 3.0
 *                             October 1996
 *                         dfk@cs.dartmouth.edu
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "file.h"
#include "lu.param"

/* define this if you want the matrix printed before and after  */
/* #define PRINT */

/*
#define DEBUG1(x) { printf x;}
#define DEBUG2(x) { printf x;}
*/
#define DEBUG1(x) {}
#define DEBUG2(x) {}


/* use an NxN matrix */
#define N	    LU_MATRIX_SIZE

float *A_data;	      /* the input matrix, and our answer matrix, data */
#define A(i,j)    A_data[ (j) * N + (i) ] /* zero-based subscripts */

float *B_data;	      /* the supposed answer matrix's data */
#define B(i,j)    B_data[ (j) * N + (i) ] /* zero-based subscripts */

/* either one is zero and the other is close, or they are within 1% */
#define SAME(x, y) ( \
    ((x) == 0 && fabs(y) < 0.01) \
 || ((y) == 0 && fabs(x) < 0.01) \
 || (fabs((x) - (y)) / (x) <= 0.01) )

void print_matrix(float *data);

int
main(int argc, char **argv)
{
    int i,j,k;		      /* loop vars */
    float m;		      /* multiplier */
    FILE *fp;

    if (argc != 2) {
	fprintf(stderr, "usage: lutest input_matrix < supposed_answer\n");
	exit(1);
    }

    if ((fp = fopen(argv[1], "r")) == NULL) {
	fprintf(stderr, "lutest: cannot open '%s' for input\n", argv[1]);
	exit(1);
    }

    /* allocate our in-core, local array */
    A_data = (float *)malloc(N * N * sizeof(float));
    if (A_data == NULL) {
	fprintf(stderr, "lutest: can't malloc %dx%d matrix\n", N, N);
	exit(1);
    }

    if (fread(A_data, sizeof(float), N*N, fp) != N*N) {
	fprintf(stderr, "lutest: error reading input matrix\n");
	exit(1);
    }

#ifdef PRINT
    printf("Initial situation:\n");
    print_matrix(A_data);
#endif

    /* LU factorization */
    for (i = 0; i < N-1; i++) {
	DEBUG1(("LU: col %d: pivot is %g\n", i+1, A(i,i)));

	for (j = i+1; j < N; j++) {
	    if (A(i,i) == 0) {
		printf("Can't do this matrix without pivoting\n");
		exit(1);
	    }				
	    m = A(j,i) / A(i,i);
	    A(j,i) = m;
	    for (k = i+1; k < N; k++)
	      A(j,k) -= m * A(i,k);

	    DEBUG2(("LU: updated col %d, diag now %g\n", j+1, A(j,j)));
	}
    }

#ifdef PRINT
    printf("After LU factorization:\n");
    print_matrix(A_data);
#endif

    /* allocate our in-core, local array */
    B_data = (float *)malloc(N * N * sizeof(float));
    if (B_data == NULL) {
	fprintf(stderr, "lutest: can't malloc %dx%d matrix\n", N, N);
	exit(1);
    }

    if (fread(B_data, sizeof(float), N*N, stdin) != N*N) {
	fprintf(stderr, "lutest: error reading answer matrix\n");
	exit(1);
    }

#ifdef PRINT
    printf("Supposed answer:\n");
    print_matrix(B_data);
#endif

    for (i = 0; i < N; i++)
      for (j = 0; j < N; j++)
	if (!SAME(A(i,j), B(i,j)))
	  printf("(%d,%d) differ: is %g, should be %g\n", 
		 i+1, j+1, B(i,j), A(i,j));
    
    return(0);
}

/* print a matrix */
#define M(i,j)    M_data[ (j) * N + (i) ] /* zero-based subscripts */

void
print_matrix(float *M_data)
{
    int i,j;

    for (i = 0; i < N; i++) {
	for (j = 0; j < N; j++)
	  printf(" %8.2f", M(i,j));
	printf("\n");
    }
}
