/* @TITLE "message.h: dmcache message facilities: definitions"*/
/* 
 * This provides a model of message passing that works well for me.
 * The actual message types that I use are in protocol.h
 *
 * types:
 *   REPLY_ACTION
 *   MESG_HEAD
 * functions:
 *   InitMessage
 *   ThreadRequest
 *   QueueRequest
 *   InstantRequest
 *   Reply
 *   WaitReply
 *   SpinWaitReply
 *   Memget, Memget_set_base
 *   Memput, Memput_set_base, Memput_done, Memput_done_wait
 * macros:
 *   INIT_ACTION
 *   INIT_ACTION_SKIP
 *   INIT_ACTION_NO_TID
 *   INIT_ACTION_NO_TID_SKIP
 *
 * Part of 
 *           The STARFISH Parallel file-system simulator
 *      (Simulation Tool for Advanced Research in File Systems)
 *
 *                              David Kotz
 *                          Dartmouth College
 *                             Version 3.0
 *                             October 1996
 *                         dfk@cs.dartmouth.edu
 */

#ifndef MESSAGE_H
#define MESSAGE_H

#include "dmcache.h"
#include "userdata.h"
#include "message.param"
#include "net.param"	      /* for MAX_PACKET_WORDS */
#include "dualq.h"	      /* sigh; needed for QueueRequestInit */
#include "pool.h"	      /* sigh; needed for QueueRequestInit */

/* A pointer to this structure specifies a point for replies.
 * It contains the buffer for the incoming message, a flag indicating
 * message arrival, a tid to wakeup on arrival, and a number of initial bytes
 * of the message to be thrown out (used for message-redirection).
 */
typedef struct {
    void *bufp;		      /* buffer for message */
    boolean hasarrived;	      /* the message has arrived */
    int tid;		      /* tid to wakeup on arrival, or NO_TID */
    ulong skip;		      /* number of leading message bytes to toss */
} REPLY_ACTION;

/* This structure should be at the top of all messages given 
 * to ThreadRequest, QueueRequest, InstantRequest, or Reply. 
 * The costbytes field is used to compute the cost of message transmission,
 * memory copies, etc, and may be more or less than the actual size of the
 * message being copied. (For example, when #undef REAL_DATA, the costbytes
 * may say the message is very big, when in fact it is just a header).
 */
typedef union {
    struct {
	FuncPtr function;     /* for requests */
	int costwords;	      /* effective message size in words */
    } request;
    struct {		      /* for replies */
	REPLY_ACTION *actionp;
	int costwords;	      /* effective message size in words */
    } reply;
} MESG_HEAD;

/* A request for Memget. See below. */
struct Memget_req {
    MESG_HEAD h;	      /* h.request.function = Memget */
    int reply_to;	      /* processor making the request */
    REPLY_ACTION *reply_at;   /* for the reply action pointer */
    int offset;		      /* offset into remote buffer (in words) */
    int words;		      /* number of words desired */
};

/* A request for Memput. See below. */
struct Memput_req {
    MESG_HEAD h;	      /* h.request.function = Memput */
    int reply_to;	      /* processor making the request */
    REPLY_ACTION *reply_at;   /* for the reply action pointer */
    int offset;		      /* offset into remote buffer (in words) */
    int copywords;	      /* number of words that actually follow */
    /* data follows this */
};
/* for n data bytes, */
#define MEMPUT_COSTWORDS(n) (((n) + sizeof(struct Memput_req)) / sizeof(Word))
#ifdef REAL_DATA
# define MEMPUT_COPYBYTES(n) ((n) + sizeof(struct Memput_req))
#else
# define MEMPUT_COPYBYTES(n) (sizeof(struct Memput_req))
#endif

extern void InitMessages(void);
/* initialize the message package */

/* We can choose what kind of requests the CP will make to the IOP
 * simply by setting this flag. 
 */
#ifdef USE_QUEUE_REQUESTS
# define IOPRequest QueueRequest
#else
# define IOPRequest ThreadRequest
#endif

extern void ThreadRequest(int proc, int copybytes, void *data);
/* Start a new thread on the given proc.
 * The first part of the data must be MESG_HEAD.
 * You should fill in the function that you want to request, and
 * the appropriate costwords value; the 'copybytes' parameter tells the
 * actual data to be copied.
 * A reply is not required, or waited for. If you expect a reply, 
 * you'd better provide a reply_to proc and a reply_at action somewhere 
 * in the message. The action should be pre-initialized.
 * Warning: data copied is rounded up to multiple of Words.
 */

extern void QueueRequestInit(int bufsize, DUALQ *fullq, POOL *freePool);
/* See QueueRequest below.  This function must be called on any processor
 * that hopes to RECEIVE QueueRequests.  Nothing special is needed to send 
 * them.  bufsize is the maximum buffer size, in Words, allowed to be 
 * received.  Exceeding this size causes a crash.  freePool is where
 * QueueRequest will hope to find an empty buffer: make sure that 
 * it always can!   fullQ is where filled buffers (new requests) are placed
 * on arrival.  Both should be preallocated and ready for use
 * before the first request arrives.
 */

extern void QueueRequest(int proc, int copybytes, void *data);
/* This is like ThreadRequest, except that we stuff the data into a buffer,
 * and enqueue the buffer onto a request queue.  Presumably some thread
 * will pull it off the queue, and later recycle the buffers back through 
 * the free-buffer queue.  
 * You should fill in the function that you want to request, and
 * the appropriate costwords value; the 'copybytes' parameter tells the
 * actual data to be copied.
 * A reply is not required, or waited for. If you expect a reply, 
 * you'd better provide a reply_to proc and a reply_at action somewhere 
 * in the message. The action should be pre-initialized.
 * Warning: data copied is rounded up to multiple of Words.
 */

extern void InstantRequest(int proc, int copybytes, void *data);
/* This is like ThreadRequest, except that it does not start a new thread.
 * The request is served directly by the interrupt handler (which could
 * of course start a new thread if desired). 
 *
 * You should fill in the function that you want to request, and
 * the appropriate costwords value; the 'copybytes' parameter tells the
 * actual data to be copied.
 * A reply is not required, or waited for. If you expect a reply, 
 * you'd better provide a reply_to proc and a reply_at action somewhere 
 * in the message. The action should be pre-initialized.
 * Warning: data copied is rounded up to multiple of Words.
 */

extern void Reply(int proc, int copybytes, void *data);
/*   The first part of data must be a MESG_HEAD. This provides 
 * an action pointer for the recipient to use, as well as the costwords
 * used in calculating the cost of sending, copying, etc.  The recipient
 * copies the REST of the message into the buffer specified in the action, 
 * and wakes up the specified thread id.
 *   The destination proc is provided as an argument.
 * Warning: data copied is rounded up to multiple of Words.
 * The cost of sending the data through the network is controlled by  
 * header.reply.costwords.
 */

extern void ReplyACK(int proc, REPLY_ACTION *actionp);
/* The caller provides an action pointer for the recipient to use. 
 * The destination proc is provided as an argument.
 * This form is used when no data is to be sent, just the message header.
 * Typically this serves as an ACK only, waking up a thread waiting on the
 * action pointer.
 *   The destination proc is provided as an argument.
 */

extern void WaitReply(REPLY_ACTION *actionp);
/* suspend until the message arrives, for a given action */
/* if the message has already arrived, we return immediately */
/* you should have already set  actionp->tid = MY_TID */

extern void SpinWaitReply(REPLY_ACTION *actionp);
/* spinwait until the message arrives, for a given action */
/* this gives the fastest response, but hurts other threads */
/* you should have already set  actionp->tid = MY_TID */

extern void RedirectReply(REPLY_ACTION *actionp, 
			  void *newbufp, ulong skip, ulong bytes);
/* Change the destination buffer pointer for a given action. 
 * If the message has not yet arrived, just change the pointer;
 * if the message has arrived, copy the data to the new location.
 * The first 'skip' bytes of the message will be skipped, and then
 * 'bytes' will be copied. (The latter is needed for an already-arrived
 * message.)
 * This DOES NOT WAIT for the message to arrive, it just redirects it.
 */

extern void UD_RedirectReply(REPLY_ACTION *actionp, UserData newbufp, 
			     ulong skip, ulong bytes);
/* This is like RedirectReply, except it takes a UserData value instead
 * of a pointer.  Thus, copybytes == 0, costbytes == bytes.
 */

extern void Memput(struct Memput_req *request, 
		   FuncPtr freereq, void *freearg);
extern void Memput_set_base(UserData base, boolean append); /* by local */
/* This is does a copy of a local buffer into a remote processor's memory.
 * The request is served directly by the remote interrupt handler.
 *
 * The request contains an offset,
 * from which we compute the destination address by adding the MEMPUT_BASE
 * (see above).  The header contains costwords, used to calculate cost of 
 * data copying; the actual amount data to copy is implicit in the length
 * of the message.  If the append flag is set by the local proc, the data is
 * stored at the base address, which is then incremented by the amount of
 * data stored.   Otherwise the data is stored at the given offset from the 
 * base, which remains constant.  The request also tells us where to reply.
 *
 * DO NOT CALL MEMPUT DIRECTLY.
 * Memput is different from the others.  Calling sequence:
 *   InstantRequest(proc, sizeof(struct Memput_req), struct Memput_req *req);
 * where req->h.request.function = Memput
 *       req->actionp = & your reply action
 *       req->offset = offset in remote buffer (in words)
 *       req->words = length of data following this header (in words)
 *       INIT_ACTION(req->actionp, & your destination buffer);
 * and put your data right after the request. 
 * Wait for the reply to arrive in any of the usual ways.
 *
 * Warning: data copied is rounded up to multiple of Words.
 */

extern void Memput_done(int first, int last); /* by remote */
extern void Memput_done_wait(int nprocs); /* by local */
/* The remote processor, that has been calling Memput(proc...) for a 
 * while, should call Memput_done when it has sent its last Memput.
 * The local side should call Memput_done_wait(nprocs) before calling
 * Memput_set_base(NULL), at the end of an operation, to make sure  
 * all nprocs procs have checked in as finished. This makes sure that 
 * all outstanding Memput messages have cleared. (It's only really 
 * necessary when the Memput_set_base is caused by a third-party
 * message, eg, release of a barrier).  Since I typically use Memput to 
 * a set of processors, the arguments allow specification of a range of
 * processor numbers [first...last] inclusive.
 */

extern void Memputq(int proc, ulong offset, UserData buf, ulong nbytes);
extern void Memputq_flush(void);
/* This is a queued version of Memput, which you DO call like a normal 
 * function.  It copies the data from your buffer into a holding buffer.
 * When the holding buffer is full, it sends it out to 'proc'.  There is 
 * one holding buffer for each other 'proc' that you talk to.  If the 
 * data cannot fit in the buffer now, it sends the current bufferfull,
 * possibly waiting for the previous send to be acknowledged, then copies
 * in your new data.  Thus, this routine may block.  
 *  NOTE offset is in bytes, nbytes is in bytes.
 *  You still need to call Memput_set_base on the reeceiving processors.
 *  Be sure to call Memputq_flush(), AND Memput_done(), when you're through.
 *  Do not mix calls to Memputq with regular Memput, or Memgetq! 
 *  NOT multithread safe.
 */

extern void Memget(struct Memget_req *request, 
		   FuncPtr freereq, void *freearg);
extern void Memget_set_base(UserData base);
/* DO NOT CALL Memget DIRECTLY.
 * This does a copy from a remote processor's memory to a local buffer.
 * It is asynchronous, in that you specify a reply action.
 *
 * Memget is different from the others. Calling sequence:
 *   InstantRequest(proc, sizeof(struct Memget_req), struct Memget_req *req);
 * where req->h.requestf = Memget
 *       req->reply_to = your proc number
 *       req->reply_at = & your reply action
 *       req->offset = offset in remote buffer (in words)
 *       req->words = length of remote buffer (in words)
 *       INIT_ACTION(req->actionp, & your destination buffer);
 * then wait for the reply to arrive in any of the usual ways.
 * There is no Memget_done.
 */

extern void Memgetq(int proc, ulong offset, UserData buf, ulong nbytes);
extern void Memgetq_flush(boolean *which); /* which[NO_OF_PROCESSORS] */
/* This is a queued version of Memget, which you DO call like a normal 
 * function.  It places your request into a holding buffer.
 * When the holding buffer is full (more specifically, when the reply to the 
 * request would fill a message), it sends the request to 'proc'.  There is 
 * one holding buffer for each other 'proc' that you talk to.  If the 
 * current request cannot fit in the buffer now, it sends the holding buffer,
 * possibly waiting for the previous request to be acknowledged, then copies
 * in your new request.  Thus, this routine may block.  
 * The data, when returned, will go into the specified buf.
 *  NOTE offset is in bytes, nbytes is in bytes.
 *  You still need to call Memget_set_base on the receiving processors.
 *  Do not mix calls to Memputq with regular Memget, or Memputq! 
 *
 * Be sure to call Memgetq_flush(NULL) when you're through.
 *  Must be called when you've finished with all your Memgetq calls.
 *     (Do not run concurrently with more Memgetq calls!)
 * Optionally, you can call Memgetq_flush with which != NULL, 
 *  at any time while you're busy making Memgetq calls; that will 
 *  only send unsent requests from processors where which[proc] == TRUE,
 *  and wait for only those replies.  But you still need the (which==NULL)
 *  call at the very end, so Memgetq_flush knows that you are all done.
 */


/* @SUBTITLE "Initializing reply actions" */
/* In all of these, the buffer pointer could be a UserData, which 
 * may not be an actual pointer.
 */

/* Initialize an action pointer, with the given buffer pointer */
/* default is to skip the message header */
#define INIT_ACTION(actionp, bufferp) { \
  (actionp)->bufp = (void *)(bufferp); \
  (actionp)->hasarrived = FALSE; \
  (actionp)->tid = MY_TID; \
  (actionp)->skip = sizeof(MESG_HEAD); \
}

/* Initialize an action pointer, with the given buffer pointer */
/* specify different skip */
#define INIT_ACTION_SKIP(actionp, bufferp, skipbytes) { \
  (actionp)->bufp = (void *)(bufferp); \
  (actionp)->hasarrived = FALSE; \
  (actionp)->tid = MY_TID; \
  (actionp)->skip = (skipbytes); \
}

/* Initialize an action pointer, with the given buffer pointer */
/* specify no wakeup call */
#define INIT_ACTION_NO_TID(actionp, bufferp) { \
  (actionp)->bufp = (void *)(bufferp); \
  (actionp)->hasarrived = FALSE; \
  (actionp)->tid = NO_TID; \
  (actionp)->skip = sizeof(MESG_HEAD); \
}

/* Initialize an action pointer, with the given buffer pointer */
/* specify no wakeup call */
/* specify different skip */
#define INIT_ACTION_NO_TID_SKIP(actionp, bufferp, skipbytes) { \
  (actionp)->bufp = (void *)(bufferp); \
  (actionp)->hasarrived = FALSE; \
  (actionp)->tid = NO_TID; \
  (actionp)->skip = (skipbytes); \
}

#endif /* MESSAGE_H */
