/* @TITLE "protocol.h: message protocols"*/
/* 
 * This file defines the different message types.
 * There are some basic structures that are defined, and for each
 * we define a message form, that includes the standard message header.
 *
 * macros:
 *   MBUFFER_INIT_REPLY(mbuf)
 *   MBUFFER_INIT_REQUEST(mbuf)
 *   INIT_MBUFFER_ACTION(actionp, mbufp)
 *   INIT_MBUFFER_ACTION_NO_TID(actionp, mbufp)
 * types:
 *   typedef struct message_buffer MBUFFER;
 *   struct MIOPread
 *   struct MIOPwrite 
 *   struct MIOPwrite_memget 
 *   struct MIOPblockread
 *   MIOPblockwrite = MBUFFER
 *   struct MIOPdirect
 *   struct MIOPgeneral
 *   struct MIOPdisktest
 *
 * Part of 
 *           The STARFISH Parallel file-system simulator
 *      (Simulation Tool for Advanced Research in File Systems)
 *
 *                              David Kotz
 *                          Dartmouth College
 *                             Version 3.0
 *                             October 1996
 *                         dfk@cs.dartmouth.edu
 */

/* $Id: protocol.h,v 3.0 1996/10/18 06:05:51 dfk RELEASE3 dfk $ */

#ifndef PROTOCOL_H
#define PROTOCOL_H

#include "dmcache.h"
#include "message.h"
#include "pattern.h"	      /* for pattern_e */
#include "diskdriver.h"	      /* for DiskRequest */

/* @SUBTITLE "Message types" */

/* A generic buffer.
 * Also usable as a request buffer or a reply buffer.
 */
typedef struct message_buffer MBUFFER;
struct message_buffer {
    MESG_HEAD h;

    /* reply address, for replies only */
    int reply_to;	      /* proc */
    REPLY_ACTION *reply_at;   /* action */
    
    ulong block;	      /* which block is this in the file? */
    boolean valid:1;	      /* holds valid data */
    boolean dirty:1;	      /* is it different from disk copy? */
    boolean prefetched:1;     /* was this block prefetched? */
    DiskRequest *diskio;      /* handle for outstanding disk request */
    ulong bytes;	      /* bytes written to this block */

    /* the data, contents[BLOCK_SIZE]  -- don't forget to UD_ArrayInit */
    UD_ArrayDecl(contents, BLOCK_SIZE);
};
    
#define MBUFFER_INIT_REPLY(mbuf) { \
    UD_ArrayInit(((mbuf).contents), BLOCK_SIZE); \
    (mbuf).h.reply.costwords = MBUFFER_COSTWORDS; \
}

#define MBUFFER_INIT_REQUEST(mbuf) { \
    UD_ArrayInit(((mbuf).contents), BLOCK_SIZE); \
    (mbuf).h.request.costwords = MBUFFER_COSTWORDS; \
}

/* define the size, and effective size, of an MBUFFER */
#ifdef REAL_DATA
# define MBUFFER_COPYBYTES sizeof(MBUFFER)
# define MBUFFER_COSTBYTES sizeof(MBUFFER)
# define MBUFFER_COSTWORDS (MBUFFER_COSTBYTES / sizeof(Word))
#else
# define MBUFFER_COPYBYTES sizeof(MBUFFER)
# define MBUFFER_COSTBYTES (sizeof(MBUFFER) - sizeof(UserData) + BLOCK_SIZE)
# define MBUFFER_COSTWORDS (MBUFFER_COSTBYTES / sizeof(Word))
#endif REAL_DATA

/* initialize an action pointer to receive into an MBUFFER */
#define INIT_MBUFFER_ACTION(actionp, mbufp) \
  INIT_ACTION_SKIP((actionp), (mbufp), 0)
#define INIT_MBUFFER_ACTION_NO_TID(actionp, mbufp) \
  INIT_ACTION_NO_TID_SKIP((actionp), (mbufp), 0)

/* "Arbitrary" read, within a block */
/* buffer and offset must be word-aligned, bytes integral # of words */
struct MIOPread {
    MESG_HEAD h;	      /* h.request.function = IOPread */
    int reply_to;	      /* proc */
    REPLY_ACTION *reply_at;   /* action: direct into buffer */
    ulong offset;	      /* where in the file (word-aligned) */
    ulong bytes;	      /* how much of the file */
};

/* "Arbitrary" write, within a block */
/* buffer and offset must be word-aligned, bytes integral # of words */
/* reply_at=NULL means you want no reply; the reply is just a MESG_HEAD */
/* reply_to must be set for proper event generation, in any case */
struct MIOPwrite {
    MESG_HEAD h;	      /* h.request.function = IOPwrite */
    int reply_to;	      /* proc */
    REPLY_ACTION *reply_at;   /* action: direct from buffer */
    ulong offset;	      /* where in the file (word-aligned) */
    ulong bytes;	      /* how much of the file */
    /* the data contents[BLOCK_SIZE]  -- don't forget to UD_ArrayInit */
    UD_ArrayDecl(contents, BLOCK_SIZE);
};

/* "Arbitrary" write, within a block, USING MEMGET for data transfer  */
/* This is like IOPwrite, but we don't include any data. Instead, we 
 * provide a "pointer" for the data, and the IOP issues a Memget to
 * pull the data from us when it is ready. */
/* buffer and offset must be word-aligned, bytes integral # of words */
/* reply_at=NULL means you want no reply; the reply is just a MESG_HEAD */
/* reply_to must be set for proper event generation, in any case */
struct MIOPwrite_memget {
    MESG_HEAD h;	      /* h.request.function = IOPwrite_memget */
    int reply_to;	      /* proc */
    REPLY_ACTION *reply_at;   /* action: direct from buffer */
    ulong file_offset;	      /* where in the file (word-aligned) */
    ulong bytes;	      /* how much to transfer */
    ulong memget_offset;      /* offset to use for memget */
};

/* Block read, using an MBUFFER */
/* buffer and offset must be word-aligned, bytes integral # of words */
struct MIOPblockread {
    MESG_HEAD h;	      /* h.request.function = IOPblockread */
    int reply_to;	      /* proc */
    REPLY_ACTION *reply_at;   /* action: into an MBUFFER */
    ulong block;	      /* which block of the file */
};

/* disk-directed read/write */
/* reply_at=NULL means you want no reply; the reply is just a MESG_HEAD */
/* reply_to must be set for proper event generation, in any case */
struct MIOPdirect {
    MESG_HEAD h;	      /* h.request.function = some disk-directed f */
    int reply_to;	      /* proc */
    REPLY_ACTION *reply_at;   /* action: no buffer pointer, just for wakeup */
};


/* "general" disk-directed read/write (IOPread_gen and IOPwrite_gen)*/
/* reply_at=NULL means you want no reply; the reply is just a MESG_HEAD */
struct MIOPgeneral {
    MESG_HEAD h;	      /* h.request.function = some disk-directed f */
    int reply_to;	      /* proc */
    REPLY_ACTION *reply_at;   /* action: no buffer pointer, just for wakeup */
    PATTERN pattern;	      /* access pattern to use */
    ulong nRecords;	      /* total number of records in the file */
    ulong nRcols;	      /* number of columns in the record matrix */
    ulong nRrows;	      /* number of rows in the record matrix */
    int nPcols;		      /* number of columns in the processor topology */
    int nProws;		      /* number of rows in the processor topology */
    ulong firstbyte;	      /* first byte file desired (usually 0) */
    ulong lastbyte;	      /* last byte file desired (usually at EOF) */
};

/* for disk device testing (do_diskTest in tests.ca) */
struct MIOPdisktest {
    MESG_HEAD h;	      /* h.request.function = some disk-directed f */
    int reply_to;	      /* proc */
    REPLY_ACTION *reply_at;   /* action: no buffer pointer, just for wakeup */
    int disk;		      /* which to test */
};

#endif /* PROTOCOL_H */
