# a nawk script
# we expect the following columns:
# patname iop sort rsize layout fsize y
#   where patname is a string
#   iop sort rsize layout fsize  are levels (-1, 1)
#   y is a floating-point number


# Part of
#              The STARFISH Parallel file-system simulator
#        (Simulation Tool for Advanced Research in File Systems)
# 
#                               David Kotz
#                           Dartmouth College
#                              Version 3.0
#                              January 1996
#                          dfk@cs.dartmouth.edu
#



function init()
{
    for (i = 0; i < cols; i++)
      dotprod[i] = 0;
    rows = 0;
}

# print a header
function header()
{
    printf "%5s", "pat";

# mean effect
    printf "%10s", "I";
    
# direct effects
    for (i = 1; i <= k; i++)
      printf "%10d", i;

# and the 2-factor interactions
    for (i = 1; i <= k; i++)
      for (j = i+1; j <= k; j++)
	printf "%9d%d", i,j;
    printf "\n";
}

# compute the effects for the previous pattern
function find_effects()
{
    for (i = 0; i < cols; i++)
      effect[i] = dotprod[i] / rows;
}

# assign the fractional variations for the previous pattern
function find_variations()
{
    SST = 0;			# actually, it will be SST/2^k
    for (i = 1; i < cols; i++)
      SST += effect[i] * effect[i];

    sum = 0;
    for (i = 1; i < cols; i++) {
	variation[i] = effect[i] * effect[i] / SST * 100;
	sum += variation[i];
    }
    variation[0] = sum;
}

# print the two-line summary for the previous pattern
function doprint(pat)
{
    printf "%5s", pat;
    for (i = 0; i < cols; i++)
      printf "%10.1f", effect[i];
    printf "\n";
    printf "%5s", "";
    for (i = 0; i < cols; i++)
      printf "%10.1f", variation[i];
    printf "\n";
}

################################################
BEGIN {
    k = 5;			# number of direct factors
    yfield = k+2;		# $1 is pattern, $2..$(k+1) are factors
    cols = 1 + k + k*(k-1)/2;	# mean, 1-factor, and 2-factor effects
    first = 1;
    header();
    init();
}

################################################
# for every input line
NF == yfield {
    if ($1 != lastpat && !first) {
	find_effects();		# compute effects for previous pattern
	find_variations();	# assign variations for previous pattern
	doprint(lastpat);	# print summary for previous pattern
	init();			# reset for this new pattern
    }
    first = 0;
    rows++;
    lastpat = $1;

# mean effect
    col = 0;
    dotprod[col++] += $yfield;

# direct effects
    for (i = 1; i <= k; i++) {
	field = i+1;
	dotprod[col++] += $field * $yfield;
    }

# and the 2-factor interactions
    for (i = 1; i <= k; i++) {
	ifield = i+1;
	for (j = i+1; j <= k; j++) {
	    jfield = j+1;
	    dotprod[col++] += $ifield * $jfield * $yfield;
	}
    }
}

NF != yfield {
    printf "Bad input line '%s' (%d fields)\n", $0, NF;
}

END {
    find_effects();		# compute effects for previous pattern
    find_variations();		# assign variations for previous pattern
    doprint(lastpat);		# summary for last pattern
}

