/* @TITLE "userdata.c: 'pretend' data movement operations"*/
/* 
 * Some equivalents to functions in aux.c and util.c, but for
 * the 'pretend' (fake) version of UserData, as defined in userdata.h.
 * By defining REAL_DATA, we actually move data around, which is important
 * only if content matters;  if we do not define it, we use these 'pretend'
 * functions, who charge the same Proteus-time-cost for the operations,
 * but don't actually move data.  This is 1) faster, and 2) needs MUCH 
 * less memory if we're pretending to work with lots of data.
 *
 * We define user data to be a negative integer
 * whose absolute value is the word-length of the buffer.
 * Or, you can look at it as the difference between a pointer 
 * just following the buffer, and this 'pointer'.  This makes
 * it possible to do UserData "pointer" arithmetic with no changes.
 * 
 * functions:
 *  UDx_WordCopy
 *  UDx_WordZero
 *  noncyc_UDx_WordCopy
 *
 * Some of these functions could be implemented as macros, but then they
 * would take less time because the function-call overhead would not 
 * be there.
 *
 *   I can't make alignment checks because src or dest 
 * could be a UserData or could be a real pointer;
 * a real pointer refers to bytes, but a UserData refers to words...
 *
 *  See util.ca, aux.c for descriptions of the functions.
 *
 * Part of 
 *           The STARFISH Parallel file-system simulator
 *      (Simulation Tool for Advanced Research in File Systems)
 *
 *                              David Kotz
 *                          Dartmouth College
 *                             Version 3.0
 *                             October 1996
 *                         dfk@cs.dartmouth.edu
 */

/* $Id: userdata.c,v 3.0 1996/10/18 06:05:51 dfk RELEASE3 dfk $ */

#include "dmcache.h"		/* the overall include file */
#include "userdata.h"

#ifndef REAL_DATA	      /* throw out the whole file if REAL_DATA */

/* @SUBTITLE "UD_AllocWords: non-cycle-counted malloc for UserData" */
/* also, takes length in words, and returns UserData* */
UserData
UD_AllocWords(int wordcount)
{
    return(-wordcount);
}

/* @SUBTITLE "UDx_WordCopy: copy word-aligned UserData memory"  */
/* The macro UD_WordCopy calls this function.
 * Compare with WordCopy in util.ca.
 * I looked at aug-util.s to get the constant AddTimes().
 */
void
UDx_WordCopy(UserData src, UserData dest, int length)
{
    register int groups = length >> 3; /* groups of 8 words */
    register int leftover = length & 0x7; /* leftovers from the groups */

    UD_TRACE(0, length * sizeof(Word), "WordCopy");

    AddTime(17);	      /* for the above setup */
    QuantumCheck();	      /* manual check for quantum expiration */

    AddTime(4);		      /* for the loop setup */
    while (groups-- > 0) {
	AddTime(33+4);	      /* for each loop */
	QuantumCheck();
    }

    AddTime(4);		      /* for the loop setup */
    while (leftover-- > 0) {
	AddTime(5+4);	      /* for each loop */
	QuantumCheck();
    }

    AddTime(9);		      /* to get out */
    QuantumCheck();
}

/* @SUBTITLE "UDx_WordZero: zero word-aligned UserData memory"  */
/* The macro UD_WordZero calls this function.
 * Compare with WordZero in util.ca.
 * I looked at aug-util.s to get the constant AddTimes().
 */
void
UDx_WordZero(UserData dest, int length)
{
    register int groups = length >> 3; /* groups of 8 words */
    register int leftover = length & 0x7; /* leftovers from the groups */

    UD_TRACE(0, length * sizeof(Word), "WordZero");

    AddTime(14);	      /* for above set up */
    QuantumCheck();	      /* manual check for quantum expiration */

    AddTime(4);		      /* for loop set up */
    while (groups-- > 0) {
	AddTime(17+4);	      /* for each loop */
	QuantumCheck();
    }

    AddTime(4);		      /* for loop set up */
    while (leftover-- > 0) {
	AddTime(3+4);	      /* for each loop */
	QuantumCheck();
    }

    AddTime(8);		      /* to leave */
    QuantumCheck();
}

/* @SUBTITLE "noncyc_UDx_WordCopy: UDx_WordCopy, but not cycle-counted" */
/* The macro noncyc_UD_WordCopy calls this function. */
void
noncyc_UDx_WordCopy(UserData src, UserData dest, int words)
{
    UD_TRACE(0, 0, "noncyc_WordCopy");
}

#endif REAL_DATA    	/* whole file */
