/* @TITLE "userdata.h: definitions for userdata.c"*/
/* 
 * Functions from userdata.c are defined here.  See that file for more 
 * explanation.
 * macros:
 *   UD_NCFREE
 *   UD_AllocWords
 *   UD_WordCopy
 *   UD_WordZero
 *   noncyc_UD_WordCopy
 *   UD_Check
 *
 * Unfortunately, UserData is a pointer in one case, and an integer in 
 * the other case, the compiler treats them differently, 
 * and different code is produced. 
 * This fact leads to slight differences in the cycle count in some places;
 * it seems to come out in the wash, but the execution sequences could be
 * drastically affected by even a 1-cycle difference.  Thus, using !REAL_DATA
 * to debug a problem with REAL_DATA may not reproduce the same problem.
 * This effect shows up, for example, when foo is a UserData member of
 * a structure bar, and you manipulate bar->foo, eg, to pass to UD_WordCopy.
 *
 * Part of 
 *           The STARFISH Parallel file-system simulator
 *      (Simulation Tool for Advanced Research in File Systems)
 *
 *                              David Kotz
 *                          Dartmouth College
 *                             Version 3.0
 *                             October 1996
 *                         dfk@cs.dartmouth.edu
 */

/* $Id: userdata.h,v 3.0 1996/10/18 06:05:51 dfk RELEASE3 dfk $ */

#ifndef USERDATA_H
#define USERDATA_H

#include "dmcache.h"	      /* get the basic types */
#include "userdata.param"
#include "aux.h"
#include "util.h"

/* *********************************************************** */
#ifdef REAL_DATA

/* Basically, just treat UserData as a Word*,
 * so we can use all the normal functions.
 */

typedef Word *UserData;

/* To declare an array for BLOCK_SIZE bytes of user data, 
 *  	UD_ArrayDecl(userbuf, BLOCK_SIZE);
 * Then initialize it with 
 *  	UD_ArrayInit(userbuf, BLOCK_SIZE);
 */
#define UD_ArrayDecl(array, bytes)     Word array[(bytes) / sizeof(Word)]
#define UD_ArrayInit(array, bytes)

#define UD_NCFREE(p)	      		  NCFREE(p)
#define UD_AllocWords(wordcount)	((UserData) AllocWords(wordcount))
#define UD_WordCopy(src, dest, words)	    WordCopy((src), (dest), (words))
#define UD_WordZero(dest, words)	    WordZero((dest), (words))
#define noncyc_UD_WordCopy(src, dest, words) \
    noncyc_WordCopy((src), (dest), (words))

#define UD_Check(place, src) \
    INVARIANT4((ulong)(src) % sizeof(Word) == 0, \
	       "%s: Buffer 0x%x is not word-aligned\n", (place), (src))

/* *********************************************************** */
#else  /* FAKE DATA */

/* Here, we define user data to be a negative integer
 * whose absolute value is the word-length of the buffer.
 * Or, you can look at it as the difference between a pointer 
 * just following the buffer, and this 'pointer'.  This makes
 * it possible to do UserData "pointer" arithmetic with no changes.
 * All the functions ignore their src/dest arguments, but to count
 * time properly, we pass them anyway.  Since they could be either a
 * UserData or a Word*, we have to cast first to a neutral type,
 * like ulong.  Note all this assumes that 
 *    sizeof(ulong) == sizeof(long) == sizeof(pointer)
 *
 */
typedef long UserData;

/* To declare an array for BLOCK_SIZE bytes of user data, 
 *  	UD_ArrayDecl(userbuf, BLOCK_SIZE);
 * Then initialize it with 
 *  	UD_ArrayInit(userbuf, BLOCK_SIZE);
 */
#define UD_ArrayDecl(array, bytes)     UserData array
#define UD_ArrayInit(array, bytes) { \
    CYCLE_COUNTING_OFF; \
    (array) = ((long) -(bytes) / (long) sizeof(Word)); \
    CYCLE_COUNTING_ON; \
}

#define UD_NCFREE(p)
extern  UserData UD_AllocWords(int wordcount);
#define UD_WordCopy(src, dest, words) \
	UDx_WordCopy((UserData)(src), (UserData)(dest), (words))
#define UD_WordZero(dest, words) \
    	UDx_WordZero((UserData)(dest), (words))
#define noncyc_UD_WordCopy(src, dest, words) \
    	noncyc_UDx_WordCopy((UserData)(src), (UserData)(dest), (words))

/* in userdata.c */
extern void UDx_WordCopy(UserData src, UserData dest, int words);
extern void UDx_WordZero(UserData dest, int words); 
extern void noncyc_UDx_WordCopy(UserData src, UserData dest, int words);

#define UD_Check(place, src) \
    INVARIANT4((src) < 0, \
	       "%s: Buffer pointer %ld is past the end of its buffer!\n", \
	       (place), (src))

#endif REAL_DATA


/*
#define UD_TRACE(copybytes, costbytes, where) { \
    CYCLE_COUNTING_OFF; \
    printf("udtrace %2d %12s %4d %4d %s\n", \
	   CURR_PROCESSOR, time_print(GetTime()), \
	   (copybytes), (costbytes), (where)); \
    CYCLE_COUNTING_ON; \
}
*/

#define UD_TRACE(copybytes, costbytes, where) {}



#endif /* USERDATA_H */

