#if !defined(SOCKET_H)
#define SOCKET_H

#if !defined(TO_LIST_H)
#include "to_list.h"
#endif

#if !defined(TNO_LIST_H)
#include "tno_list.h"
#endif

#if !defined(MESSAGE_H)
#include "message.h"
#endif


class ToopsSocket;
class ToopsProcess;
class ToopsChannel;

//-----------------------------------------------------------------------------
// ToopsSocketLink
// ===========
// Set up a link to a ToopsSocket. See also to_list.h, tno_list.h.
class ToopsSocketLink : public ToopsNamedObjectLink
{
public:
    ToopsSocketLink ( ToopsSocket *p) : 
         ToopsNamedObjectLink(( ToopsNamedObject*)p) {}
    virtual void write(int depth = 0, int mode = 0) const;
    ToopsSocket* object(void) { return (ToopsSocket*) _object();}
    ToopsSocketList* linkedTo(void) 
         { return (ToopsSocketList*)ToopsObjectLink::linkedTo();}
    int isLinked(void) { return ToopsObjectLink::isLinked(); }
    inline void unLink(void);
    
    DECLARE_CLASSINFO(ToopsSocketLink); // pt 9.94

private:
    // prevent compiler from using copy / default constructor and assignment
    ToopsSocketLink();
    //ToopsSocketLink(const ToopsSocketLink&); //mjk 280895
    void operator=(const ToopsSocketLink&);
};


//-----------------------------------------------------------------------------
// ToopsSocket (source code -> socket.cpp/.C)
// ======================================
//
//class ToopsSocket : private ToopsNamedObject / pt 9.94
class ToopsSocket : public ToopsNamedObject
{
friend class ToopsProcess;   // accesses only startSync(), cancelSync()
public:
    ToopsSocket(const char *name, ToopsProcess *owner, ToopsChannel *channel);
    ToopsSocket(const char *name, ToopsProcess *owner, ToopsChannel &channel);
    ToopsSocket(const char *name, ToopsProcess &owner, ToopsChannel *channel);
    ToopsSocket(const char *name, ToopsProcess &owner, ToopsChannel &channel);
    ~ToopsSocket();

    // states of a ToopsSocket
    // ready: ready for sending / receiving / synchronizing
    // inSync: synchronize or receive was called.
    // create, gotMessage, sending: ToopsSocket switches for a short time to one
    //  of these states at construction, when getting a message and when sending
    //  a message to inform atStateChange() (note: atStateChange() never sees
    //  create, see toops.h). Immediatly after this, the state changes to ready
    //  again.
    // destroy: set in destructor.
    enum socketState { ready, inSync, gotMessage,
                       sending, create, destroy };
    socketState status(void) const { return s_state; }

    // is there a message in the queue ?
    int hasMessage(void) const { return receivedMessages.len() != 0; }
    // number of received messages
    instance_counter messages(void) const { return receivedMessages.len(); }

    // receive() returns the first ToopsMessage in the queue of received ToopsMessages,
    // if there is one.
    // If no ToopsMessage is in the queue, the ToopsProcess (owner) will wait until:
    // - a ToopsMessage arrives at this ToopsSocket (will be returned) or
    // - a ToopsTimer wakes the ToopsProcess up (0 will be returned)
    // The ToopsMessage will be removed from the queue (like get() does).
    ToopsMessage* receive(void);

    // Like receive, but nothing will be returned and all ToopsMessages are left
    // in the queue.
    void synchronize(void);

    // Send a ToopsMessage to all other connected ToopsSockets (connected via ToopsChannel).
    // Optionally specify a delay time, that is added to the ToopsChannels delay
    // (if there is one). The ToopsMessage is duplicated, so the sent ToopsMesssage can
    // be deleted or modified.
    void send(ToopsMessage*, simtime delay=0);

    // Get the <num>th ToopsMessage* in the queue. The ToopsMessage will be removed
    // from the queue. <num> starts with 0 for the first ToopsMessage.
    ToopsMessage *getMsg (instance_counter num=0)
        { checkCaller("getMsg"); return receivedMessages.get(num); }
    // Touch the <num>th ToopsMessage* in the queue. The ToopsMessage will remain
    // from the queue. <num> starts with 0 for the first ToopsMessage.
    ToopsMessage *touchMsg (instance_counter num=0)
        { checkCaller("touchMsg"); return receivedMessages.touch(num); }
    // Delete the <num>th ToopsMessage* in the queue.
    // <num> starts with 0 for the first ToopsMessage.
    void delMsg (instance_counter num=0)
        { checkCaller("delMsg"); receivedMessages.del(num); }

    // Get the ToopsProcess the ToopsSocket belongs to.
    ToopsProcess* owner(void) const { return s_owner; }
    // Get the ToopsChannel the ToopsSocket is connected to.
    ToopsChannel* channel(void) const { return s_channel; }

    // Get a ToopsSocket by it's name.
    inline static ToopsSocket* search(const char *n);
    // Get an array of all ToopsSockets in the simulation. The array _must_ be
    // deleted by the user. Determine the size of the array with numSockets().
    inline static ToopsSocket** systemSockets(void);

    // used internally
    simtime nextTime(void) const { return s_nextTime; }

/*    // re-export some information lost by the private inheritance:
    ToopsObject::isInheritedFrom;
    ToopsObject::DBGout;
    ToopsObject::isSameClass;
    ToopsNamedObject::name;
    ToopsSimulation::simstate;
    ToopsSimulation::state;
    ToopsSimulation::responsibility;
    ToopsSimulation::responsible;
    ToopsSimulation::stopReason;
    ToopsSimulation::time;
    ToopsSimulation::objects;
    ToopsSimulation::order;
    ToopsSimulation::stopSimulation;
    ToopsSimulation::stopWithError;
    ToopsSimulation::getTimeOut;
    ToopsSimulation::numProcessors;
    ToopsSimulation::numChannels;
    ToopsSimulation::numMessages;
    ToopsSimulation::numProcesses;
    ToopsSimulation::numTimers;
    ToopsSimulation::numSockets; */    // pt 9.94

    enum debugInfo { cstr=1, dstr=2, onSend=4, onReceive=8,
                     onTakeMessage=16, onStartSync=32,
                     onCancelSync=64, onActivate=128,
                     onSynchronize=256};
	virtual void write(int depth=0, int mode=0) const;
	  //was "void virtual"  (compiler warning) mjk 310895
    
    DECLARE_CLASSINFO(ToopsSocket); // pt 9.94
    
protected:
    // explained in toops.h
    virtual void atStart(void) {}
    virtual void atEnd(void) {}
    virtual void atStateChange(void) {}

    instance_counter arrivingMessages(void)
    { return messagesToReceive.len(); }

    virtual void activate(void);   // don't ever use this

private:
    ToopsProcess* s_owner;        // the ToopsProcess this ToopsSocket belongs to
    ToopsChannel* s_channel;      // the Tchannel this ToopsSocket is connected to
    socketState s_state;      // this ToopsSocket's state

    ToopsSocketLink sysLink;      // Links to the system wide list,
    ToopsSocketLink eventLink;    // the event list,
    ToopsSocketLink processLink;  // the owner and
    ToopsSocketLink channelLink;  // to the ToopsChannel

    ToopsMessageList receivedMessages;   // queue for arrived ToopsMessages
    ToopsMessageList messagesToReceive;  // holds the delayed ToopsMessages until
                                     // they will arrive.

    simtime s_nextTime;  // the next time a delayed ToopsMessage will arrive

    static ToopsError fSimNotStarted;
    static ToopsError fNoAccess;

    void initSocket(ToopsProcess* owner, ToopsChannel *channel);
    inline void stateChange(void);
    void takeMessage(ToopsMessage*);
    void cancelSync(void);
    void startSync(void);
    void checkCaller(const char* n);

    // prevent compiler from using copy / default constructor and assignment
    ToopsSocket();
    //ToopsSocket( ToopsSocket&); //mjk 280895
    void operator=(const ToopsSocket&);
};

//-----------------------------------------------------------------------------
// ToopsSocketList
// ===========
// See timer.h  or explanation. See also to_list.h, tno_list.h.
class ToopsSocketList : public ToopsNamedObjectList
{
public:
    ToopsSocketList() : ToopsNamedObjectList() {}
    ~ToopsSocketList() {}
    void insert(ToopsSocketLink &r) { insert(&r); }
    void insert(ToopsSocketLink *p) { _insert((ToopsObjectLink*)p); }
    void append(ToopsSocketLink &r) { append(&r); }
    void append(ToopsSocketLink *p) { _append((ToopsObjectLink*)p); }
    void inserthere(ToopsSocketLink &r) { inserthere(&r); }
    void inserthere(ToopsSocketLink *p) { _inserthere((ToopsObjectLink*)p); }
    void appendhere(ToopsSocketLink &r) { appendhere(&r); }
    void appendhere(ToopsSocketLink *p) { _appendhere((ToopsObjectLink*)p); }
    ToopsSocket *current(void) { return (ToopsSocket*) _current(); }
    ToopsSocket *first(void) { return (ToopsSocket*) _first(); }
    ToopsSocket *last(void) { return (ToopsSocket*) _last(); }
    ToopsSocket *currentLink(void) { return (ToopsSocket*) _currentLink(); }
    ToopsSocket *firstLink(void) { return (ToopsSocket*) _firstLink(); }
    ToopsSocket *lastLink(void) { return (ToopsSocket*) _lastLink(); }

    int next(ToopsSocket*& rp)
    {
        if (_setnext())
            { rp = (ToopsSocket*) _current(); return 1; }
        else
            { rp = 0; return 0; }
    }
    int prev(ToopsSocket*& rp)
    {
        if (_setprev())
            { rp = (ToopsSocket*) _current(); return 1; }
        else
            { rp = 0; return 0; }
    }
    int getnext(ToopsSocket*& rp)
    {
        ToopsSocket * pptmp;
        if ((pptmp = (ToopsSocket*) _getnext()) != 0)
            { rp = pptmp; return 1; }
        else
            { rp = 0; return 0; }
    }
    int getprev(ToopsSocket*& rp)
    {
        ToopsSocket* pptmp;
        if ((pptmp = (ToopsSocket*) _getprev()) != 0)
            { rp = pptmp; return 1; }
        else
            { rp = 0; return 0; }
    }
    int next(ToopsSocketLink*& rpl)
    {
        if (_setnext())
            { rpl = (ToopsSocketLink*) _currentLink(); return 1; }
        else
            { rpl = 0; return 0; }
    }
    int prev(ToopsSocketLink*& rpl)
    {
        if (_setprev())
            { rpl = (ToopsSocketLink*) _currentLink(); return 1; }
        else
            { rpl = 0; return 0; }
    }
    int getnext(ToopsSocketLink*& rpl)
    {
        ToopsSocketLink * pptmp;
        if ((pptmp = (ToopsSocketLink*) _getnextlink()) != 0)
            { rpl = pptmp; return 1; }
        else
            { rpl = 0; return 0; }
    }
    int getprev(ToopsSocketLink*& rpl)
    {
        ToopsSocketLink* pptmp;
        if ((pptmp = (ToopsSocketLink*) _getprevlink()) != 0)
            { rpl = pptmp; return 1; }
        else
            { rpl = 0; return 0; }
    }
    
    void remove(ToopsSocketLink* t) { _remove((ToopsObjectLink*)t); }
    ToopsSocket* search(const char* n) { return (ToopsSocket*) _search(n); }
    ToopsSocket** touchAll(void) { return (ToopsSocket**) _touchAll(); }
    virtual void write(int depth = 0, int mode = 0) const;
    
    DECLARE_CLASSINFO(ToopsSocketList); // pt 9.94
    
private:
    // prevent compiler from using copy / default constructor and assignment
    //ToopsSocketList(const ToopsSocketList&);  //mjk 280895
    void operator=(const ToopsSocketList&);
};

inline ToopsSocket* ToopsSocket::search(const char *n)
    { return allSockets()->search(n); }

inline ToopsSocket** ToopsSocket::systemSockets(void)
    { return allSockets()->touchAll(); }

inline void ToopsSocketLink::unLink(void)
    {((ToopsSocketList*)linkedTo())->remove(this); }

#endif
