#if !defined(TEST_H)
#define TEST_H

// ============================================================================
// test functions
// ============================================================================

// note: if the function has parameters, all are defaulted.
//       if you call the function, for undefined parameters and for parameters
//       replaced with a '-', the default values are used.
//       e.g. test 3 0 - - - 40 calls test 3, the 1st parameter is set to 0,
//            the 5th is set to 40, for all others are their default values
//            used.
//

// ----------------------------------------------------------------------------
// test1()
//
// 3 ToopsProcesses, each with only one testProcess_1
// Call: test 1 [<endTime> [<contextChangeTime>]]
// Parameters:  endTime (default 0, i.e until all processes have terminated)
//               the simulation stops at endTime (i.e. the last events are
//               processed at endTime-1)
//              contextChangeTime (default 0) specifies the time the processor
//               needs for a context change (this value should in case of this
//               test have no effect, because no processors should do a context
//               change here !!)
// Outputs full debug info, unless contextChangeTime is specified, in this
// case only ToopsProcessor::activate() will show debug info;
int test1(int argc, char** argv);

// ----------------------------------------------------------------------------
// test2()
//
// Call: test 2
// Parameters: none
// Two ToopsProcessors, each with one testProcess_2.
// tp2_1_1::behavior() returns , tp_2_2_2::behavior() just ends.
// Tests correct process calling mechanism.
// Debug info turned off, except ToopsProcess::onActivate ( for testing activate()
// _and_ start())
int test2(void);

// ----------------------------------------------------------------------------
// test3()
//
// Call: test 3 [...]
// Tests consumeTime() and setWaiting().
// One ToopsProcessor with one testProcess_1 (tp_1) and one testProcess_3 (tp_3).
// Parameters: test 3 [<p1>|-] [<p2>|-] [...] [<p10>|-]
//      def.
//  p1    0     endtime of simulation
//  p2    0     ToopsProcessor's time for context change
//  p3    1     tp_1's priority
//  p4    0     tp_3's priority
//  p5  100     tp_1's number of loops
//  p6   10     tp_1's time to consume
//  p7   10     tp_3's number of times the outer loop is repeated
//  p8    5     tp_3's number of times the inner loop is repeated each time
//  p9   15     tp_3's time to wait in each call to setWaiting()
// p10    2     tp_3's time to consume in each call to consumeTime()
int test3(int argc ,char** argv);


// ----------------------------------------------------------------------------
// test4()
//
// Call: test 4
// Parameters: none
// Two ToopsProcessors, each with one testProcess_4.
// Tests correct process stack save / restore mechanism.
// Debug info turned off.
int test4(void);


// ----------------------------------------------------------------------------
// test5()
//
// Call: test 5 [<contextChangeTime>]]
// Parameters: contextChangeTime  default 0
// Shows the effect of process' instantiation order on their start and restart
// order and tests also a little bit the multiple start of simulations in one
// program run.
int test5(int argc ,char** argv);

// ----------------------------------------------------------------------------
// test6()
//
// Call: test 6 [...]
// One ToopsProcessor p1 with two testProcess_3 (tp3_1_1, tp3_1_2)
// Tests ToopsProcessor for behaviour, when all processes are waiting.
// Parameters: test 6 [<p1>|-] [<p2>|-] [...] [<p4>|-]
//      def.
//  p1    0     endtime of simulation
//  p2    2     ToopsProcessor's time for context change
//  p3    1     tp3_1_1's priority
//  p4    0     tp3_1_2's priority
int test6(void);

// ----------------------------------------------------------------------------
// test7()
//
// Call: test 7 [<endTime>] [<debugInfo>]
// Parameters: endTime  time for sim stop (def. 50000)
//             debugInfo 0 -> off
//                       1 -> off, except ToopsSimulation::start
//                       2 -> as 1, but also ToopsSimulation::getNextEvent
//                       3 -> as 2, but also ToopsProcessor::activate
//                       >3 -> all
//                       default: 0
//                       has only effect with NDEBUG defined !
// Some kind of benchmark: 25 testProcess_5 on 5 ToopsProcessors.
// Does some additional output via cerr to show progress. Hint: redirect the
// standard output to a file for better performance.
//
int test7(int argc ,char** argv);

// ----------------------------------------------------------------------------
// test8()
//
// Call: test 8 [...]
// One ToopsProcessor with one testProcess_6 (tp6_1_1) and one
// testProcess_1 (tp1_1_1).
// Tests terminate()'ing another process
// Parameters: test 8 [<p1>|-] [<p2>|-] [...] [<p8>|-]
//      def.
//  p1    5     after <p1> loops tp6_1_1 terminates tp1_1_1
//  p2    0     tp6_1_1's priority
//  p3   10     tp6_1_1's total number of loops
//  p4   15     tp6_1_1's time for setWaiting()
//  p5    5     tp6_1_1's time for consumeTime()
//  p6    1     tp1_1_1's priority
//  p7   20     tp1_1_1's wanted total nuber of loops until self termination
//  p8    5     tp1_1_1's time to consume in each call to consumeTime()
int test8(int argc, char** argv);

// ----------------------------------------------------------------------------
// test9()
//
// Call: test 9 [...]
// One ToopsProcessor with one testProcess_7 (tp7_1_1) and one
// testProcess_8 (tp8_1_1).
// Tests resume() and suspend() (a ToopsProcess suspending itself).
// Parameters: test 9 [<p1>|-] [<p2>|-] [...] [<p8>|-]
//      def.
//  p1    2     after _each_ <p1> loops tp8_1_1 tries to resume() tp7_1_1
//  p2    0     tp8_1_1's priority
//  p3   30     tp8_1_1's total number of loops
//  p4   15     tp8_1_1's time for setWaiting()
//  p5    5     tp8_1_1's time for consumeTime()
//  p6    3     after _each_ <p6> loops tp7_1_1  suspends itself
//  p7    1     tp7_1_1's priority
//  p8   20     tp7_1_1's wanted total nuber of loops until self termination
//  p9    5     tp7_1_1's time for setWaiting()
// p10    5     tp7_1_1's time for consumeTime()
int test9(int argc ,char** argv);

// ----------------------------------------------------------------------------
// test10()
//
// Call: test 10 [...]
// One ToopsProcessor with one testProcess_9 (tp9_1_1), one
// testProcess_8 (tp8_1_1) and one testProcess_1(tp1_1_1).
// Tests resume() and suspend() (a ToopsProcess suspending another).
// Parameters: test 10 [<p1>|-] [<p2>|-] [...] [<p11>|-]
//      def.
//  p1    0     tp9_1_1's priority
//  p2    2     tp8_1_1's priority
//  p3    4     tp1_1_1's priority
//  p4    2     after _each_ <p4> loops tp9_1_1 tries to suspend() tp1_1_1
//  p5    5     after _each_ <p5> loops tp8_1_1 tries to resume() tp1_1_1
//  p6    5     tp9_1_1's time to consume in each loop
//  p7   25     tp9_1_1's time for setWaiting() in each loop
//  p8    8     tp8_1_1's time to consume in each loop
//  p9   16     tp8_1_1's time for setWaiting() in each loop
// p10    4     tp4_1_1's time to consume in each loop
// p11   30     tp9_1_1's total number of loops
// p12   50     tp8_1_1's total number of loops
// p13  200     tp1_1_1's wanted total number of loops (depending on the
//              test parameters it might be left suspended, when tp8_1_1
//              terminates before tp9_1_1)
int test10(int argc ,char** argv);

// ----------------------------------------------------------------------------
// test11()
//
// Call: test 11 [...]
// One ToopsProcessor 'p1' with one testProcess_10 (tp10_1_1). (see also output)
// Parameters: test 11 [<p1>|-] [<p2>|-] [...] [<p7>|-]
//      def.
//  p1 10000    end of simulation
//  p2    0     tp10_1_1's priority
//  p3   10     after each <p3> loops tp10_1_1 creates a new testProcess_5
//  p4   50     tp10_1_1's total number of loops
//  p5    1     priority of each testProcess_5 (0 -> random value [0..99])
//  p6   10     tp10_1_1's time to consume in each loop
//  p7   30     tp10_1_1's time for setWaiting() in each loop
int test11(int argc ,char** argv);

// ----------------------------------------------------------------------------
// test12()
//
// Call: test 12 [...]
int test12(int argc ,char** argv);
// One ToopsProcessor 'p1' with two testProcess_11 (tp11_1_1, tp11_1_2).
// (see also output and class testProcess_11)
// Parameters: test 12 [<p1>|-] [<p2>|-] [...] [<p8>|-]
//      def.
//  p1    0     tp11_1_1's priority
//  p2    1     tp11_1_2's priority
//  p3   20     tp11_1_1's time for setWaiting() in each loop
//  p4   30     tp11_1_2's time for setWaiting() in each loop
//  p5   40     tp11_1_1's time to consume in each loop
//  p6   10     tp11_1_2's time to consume in each loop
//  p7   50     tp11_1_1's total number of loops
//  p8   50     tp11_1_2's total number of loops

// ----------------------------------------------------------------------------
// test13()
//
// Call: test 13 [...]
int test13(int argc ,char** argv);

// ----------------------------------------------------------------------------
// test14()
//
// Call: test 14 [...]
int test14(int argc ,char** argv);

// ----------------------------------------------------------------------------
// test15()
//
// Call: test 15 [...]
int test15(int argc ,char** argv);



// ----------------------------------------------------------------------------
// sizeInfo()
//
void sizeInfo(void);

// ----------------------------------------------------------------------------
// errHandler1()
//
void errHandler1(int num, ToopsError::severity sev, const char* errText);



// ============================================================================
// test classes
// ============================================================================

// ----------------------------------------------------------------------------
// testProcess_1
//
// testProcess_1 is a simple process, performing <h> loops, each doing some
// output and calling consumeTime(<cT>) then. The process' priority can be
// specified via <pr>, the name is <n>, the owning ToopsProcessor is <p>.
class testProcess_1 : public ToopsProcess
{
public:
    testProcess_1(ToopsProcessor &p,const char*n, simtime ct, uint h, uint pr=0);
    void behavior(void);
    void write(int d,int m) const { ToopsProcess::write(d,m);}
private:
    simtime cons;  // the time to consume for each loop
    uint times;    // the number of loops
};


// ----------------------------------------------------------------------------
// testProcess_2
//
// serves to test the effect of ending a ToopsProcess without terminating it.
// testProcess_2(p,n,ct,ret) sets up a testProcess_2 on ToopsProcessor <p> with
// name <n>, consuming time <ct> once and then ending with a return (if <ret>
// is != 0) or just without anything.
class testProcess_2 : public ToopsProcess
{
public:
    testProcess_2(ToopsProcessor &p,const char*n, simtime ct, int ret);
    void behavior(void);
    void write(int d,int m) const { ToopsProcess::write(d,m);}
private:
    simtime cons;
    int ret;
};


// ----------------------------------------------------------------------------
// testProcess_3
//
// Runs on ToopsProcessor <p>, has the name <n> and the priority <prio>.
// behavior() performs an outer loops <loops> times, each time
// calling <cTLoops> times consumeTime(<cT>).
// At the end of each time the outer loop performs setWaiting(<sW>) is called.
// Serves to test setWaiting.
// If initcT> 0, at the process' start a consumeTime(initcT) is called.
class testProcess_3 : public ToopsProcess
{
public:
    testProcess_3(ToopsProcessor &p, const char*n, uint prio,
                  uint loops, uint cTLoops,
                  simtime cT, simtime sW, simtime initcT=0) ;
    void behavior(void);
    void write(int d,int m) const { ToopsProcess::write(d,m);}
private:
    simtime tp_cT;      // time for each consumeTime
    simtime tp_sW;      // time for each setWaiting
    simtime tp_initcT;  // initial consumeTime()
    uint tp_loops;      // number of outer loops
    uint tp_cTLoops;    // number of inner loops
};

// ----------------------------------------------------------------------------
// testProcess_4
//
// testProcess_4 is mostly the same as tProcess_1, but it calls once
// consumeTime() from TheProcess and once from a function (no inline),
// to test the stack save/restore mechanism on correct working, when
// calling it not from behavior(), but from a sub function. In addition,
// terminate() is also called from a own function.
// parameters are the same as for testProcess_1.
class testProcess_4 : public ToopsProcess
{
public:
    testProcess_4(ToopsProcessor &p,const char*n, simtime ct, uint h, uint pr=0);
    void behavior(void);
    void wasteMyTime(void);
    void thatsItFolksIveGotEnough(void);
    void write(int d,int m) const { ToopsProcess::write(d,m);}
private:
    simtime cons;  // the time to consume for each loop
    uint times;    // the number of loops
};

// ----------------------------------------------------------------------------
// testProcess_5
//
// Runs on ToopsProcessor <p>, has the name <n> and the priority <prio>.
// Random controlled. (behaviour -> output)
// If <useCerr> != 0, the process states terminating itself via cerr.
class testProcess_5 : public ToopsProcess
{
public:
    testProcess_5(ToopsProcessor &p, const char*n, uint prio,
                  ToopsSimulation::responsibility r=user, int useCerr=1);
    void behavior(void);
    void write(int d,int m) const { ToopsProcess::write(d,m);}
    ulong counter(void) const { return t_counter; }
private:
    ulong t_counter;    // number of context switches
    int tp_cerr;        // state termination via cerr
};


// ----------------------------------------------------------------------------
// testProcess_6
//
// Runs on ToopsProcessor <p>, has the name <n> and the priority <prio>.
// Performs a loop <loop> times, calling consumeTime(<ct>) and setWaiting(sW),
// Terminates the ToopsProcess* <term> after <doit> loops.
class testProcess_6 : public ToopsProcess
{
public:
    testProcess_6(ToopsProcessor &p, const char*n, uint prio,
                  uint loops, simtime cT, simtime sW,
                  ToopsProcess* term, uint doit);
    void behavior(void);
    void write(int d,int m) const { ToopsProcess::write(d,m);}
private:
    simtime tp_cT;      // time for each consumeTime
    simtime tp_sW;      // time for each setWaiting
    uint tp_loops;      // number of loops
    uint tp_doit;       // number of loops for call to tp_term->terminate()
    ToopsProcess* tp_term;  // process to terminate
};


// ----------------------------------------------------------------------------
// testProcess_7
//
// Runs on ToopsProcessor <p>, has the name <n> and the priority <prio>.
// Performs a loop <loop> times, calling consumeTime(<ct>) and setWaiting(sW),
// Suspends itself after _each_ <doit> loops.
// If <useCerr> != 0, the process states suspending itself via cerr.
class testProcess_7 : public ToopsProcess
{
public:
    testProcess_7(ToopsProcessor &p, const char*n, uint prio,
                  uint loops, simtime cT, simtime sW,
                  uint doit, int useCerr=1);
    void behavior(void);
    void write(int d,int m) const { ToopsProcess::write(d,m);}
private:
    simtime tp_cT;      // time for each consumeTime
    simtime tp_sW;      // time for each setWaiting
    uint tp_loops;      // number of loops
    uint tp_doit;       // number of loops for call to suspend()
    uint tp_cerr;       // !=0 -> state suspending itself via cerr;
};

// ----------------------------------------------------------------------------
// testProcess_8
//
// Runs on ToopsProcessor <p>, has the name <n> and the priority <prio>.
// Performs a loop <loop> times, calling consumeTime(<ct>) and setWaiting(sW),
// tries to resume() the ToopsProcess* <res> after _each_ <doit> loops.
// If <useCerr> != 0, the process states resuming <res> via cerr.
class testProcess_8 : public ToopsProcess
{
public:
    testProcess_8(ToopsProcessor &p, const char*n, uint prio,
                  uint loops, simtime cT, simtime sW,
                  ToopsProcess *res, uint doit, int useCerr=1);
    void behavior(void);
    void write(int d,int m) const { ToopsProcess::write(d,m);}
private:
    simtime tp_cT;      // time for each consumeTime
    simtime tp_sW;      // time for each setWaiting
    uint tp_loops;      // number of loops
    uint tp_doit;       // number of loops for call to tp_term->resume()
    ToopsProcess* tp_res;   // process to resume
    uint tp_cerr;    // !=0 -> state resuming tp_res via cerr;
};


// ----------------------------------------------------------------------------
// testProcess_9
//
// Runs on ToopsProcessor <p>, has the name <n> and the priority <prio>.
// Performs a loop <loop> times, calling consumeTime(<ct>) and setWaiting(sW),
// tries to suspend() the ToopsProcess* <toManage> after _each_ <sus> loops
// If <useCerr> != 0, the process states suspending <tp_toManage> via cerr.
class testProcess_9 : public ToopsProcess
{
public:
    testProcess_9(ToopsProcessor &p, const char*n, uint prio,
                  uint loops, simtime cT, simtime sW,
                  ToopsProcess *toManage, uint sus,
                  int useCerr=1);
    void behavior(void);
    void write(int d,int m) const { ToopsProcess::write(d,m);}
private:
    simtime tp_cT;      // time for each consumeTime
    simtime tp_sW;      // time for each setWaiting
    uint tp_loops;      // number of loops
    uint tp_sus;        // number of loops for call to tp_toManage->suspend()
    ToopsProcess* tp_toManage;   // process to suspend
    uint tp_cerr;    // !=0 -> state suspending tp_toManage via cerr;
};

// ----------------------------------------------------------------------------
// testProcess_10
//
// Runs on ToopsProcessor <p>, has the name <n> and the priority <prio>.
// Performs a loop <loop> times, calling consumeTime(<ct>) and setWaiting(sW),
// tries to create a new ToopsProcess after _each <create> loops.
// The new ToopsProcess has the priority <newPrio>, if <newPrio> is 0, then
// a random priority is taken (value: 0..99). This new ToopsProcess is
// a testProcess_5 (random controlled !!) and has the name
// "nP_<n>_x", where x is a counter for the newly created processes.
// <resp> is the resposibility for the new process.
// If <useCerr> != 0, the process states creation of new process via cerr.
class testProcess_10 : public ToopsProcess
{
public:
    testProcess_10(ToopsProcessor &p, const char*n, uint prio,
                  uint loops, simtime cT, simtime sW,
                  uint create, uint newPrio,
                  ToopsSimulation::responsibility resp=user,
                  int useCerr=1);
    void behavior(void);
    void write(int d,int m) const { ToopsProcess::write(d,m);}
private:
    simtime tp_cT;      // time for each consumeTime
    simtime tp_sW;      // time for each setWaiting
    uint tp_loops;      // number of loops
    uint tp_create;     // number of loops to wait until new process is created
    uint tp_newPrio;    // new process' priority
    uint tp_cerr;       // !=0 -> state  via cerr;
    instance_counter np;     // counter for new processes
    ToopsSimulation::responsibility tp_resp;  // responsibility for new processes
};

// ----------------------------------------------------------------------------
// testTimer_1
//
class testTimer_1 : public ToopsTimer
{
public:
    testTimer_1(const char* n, ToopsProcess* w);
protected:
    virtual void atStart(void);
    virtual void atEnd(void);
    virtual void atStateChange(void);
private:
    simtime lastTime;
    ToopsTimer::timerState lastState;
};

// ----------------------------------------------------------------------------
// testProcess_11
//
// Runs on ToopsProcessor <p>, has the name <n> and the priority <prio>.
// Performs a loop <loop> times, calling consumeTime(<ct>) and setWaiting(sW),
// Sets the testTimer_1 <ti> before each setWaiting(sw) to sW + 2 * owner's
// time for a context change + 1. That means, that the timer will alert only
// in the case another process with a higher priority claims the processor
// for himself longer than sW. Note: a setWaiting is in no case interrupted
// by a ToopsTimer, so this testProcess is only designed to test the basic
// timer alert mechanism (via the atStateChange() function of testTimer_1)
// and the effect of alerting a setWaiting ToopsProcess (the effect should be
// no effect, except that ToopsProcess::isTimerAlert() returns true)
// Shows also a convenient way to handle a ToopsTimer as member of a ToopsProcess.
class testProcess_11 : public ToopsProcess
{
public:
    testProcess_11(ToopsProcessor &p, const char*n, uint prio,
                    uint loops, simtime cT, simtime sW);
    ~testProcess_11() { delete tp_ti; delete timer_name; }
    void behavior(void);
    void write(int d,int m) const { ToopsProcess::write(d,m);}
private:
    simtime tp_cT;      // time for each consumeTime
    simtime tp_sW;      // time for each setWaiting
    uint tp_loops;      // number of loops
    testTimer_1* tp_ti; // our timer
    simtime tp_period;
    char *timer_name;
};

// ----------------------------------------------------------------------------
// testMessage_1, testChannel_1 and testSocket_1 are a simple example for
// using derivation from ToopsMessage, ToopsChannel and ToopsSocket to realize a type
// safe communication interface in a TOOPS based simulation.
//

// ----------------------------------------------------------------------------
// testMessage_1
//
// testMessage_1 just transports a number and a pointer to the sender.
class testProcess_12;
class testMessage_1 : public ToopsMessage
{
public:
    testMessage_1(uint msgCounter, testProcess_12* sender) : ToopsMessage()
        { m_counter = msgCounter; m_sender = sender;}
    uint counter(void) const { return m_counter; }
    testProcess_12* sender(void) const { return m_sender; }
    virtual ToopsMessage* duplicate(void)
        { return new testMessage_1(m_counter, m_sender); }
    virtual void write(int,int) const;
private:
    uint m_counter;
    testProcess_12* m_sender;
};

// ----------------------------------------------------------------------------
// testChannel_1
//
// Adds no capability to ToopsChannel. Serves just to show a method of
// implementing type safety in TOOPS communication.
class testChannel_1 : public ToopsChannel
{
public:
    testChannel_1(const char* name) : ToopsChannel(name) {}
};

// ----------------------------------------------------------------------------
// testSocket_1
//
// Adds no capability to ToopsSocket. Serves just to show a method of
// implementing type safety in TOOPS communication.
class testSocket_1 : public ToopsSocket
{
public:
    testSocket_1(const char* name, ToopsProcess* owner, ToopsChannel *channel) :
         ToopsSocket(name, owner, channel) {}
    void send(testMessage_1* m, simtime d = 0)
        { ToopsSocket::send((testMessage_1*)m,d); }
    testMessage_1* getMsg (instance_counter num=0)
        { return (testMessage_1*) ToopsSocket::getMsg(num); }
    testMessage_1* touchMsg (instance_counter num=0)
        { return (testMessage_1*) ToopsSocket::touchMsg(num); }
};

// ----------------------------------------------------------------------------
// testProcess_12
// testProcess_12 runs on ToopsProcessor <p>, has the name <n> and a fixed priority
// of 1. It will run a loop <loops> times, consuming <cT> each loop.
// A testSocket_1 will be set up (named socket_<n>). It will be used to send
// each loop a testMessage_1 to <c>. The messages contains a counter, that is
// incremented each time the message is sent.
// testProcess_12 'polls' the socket each loop using messages(), getMsg(),
// touchMsg() and delMsg() (details see source code).
// If listDebugging is > 0, the ToopsSocket::write() function is output before
// the messages are removed from testSocket_1.
// Tests the basic ToopsChannel / ToopsSocket / ToopsMessage functionality.

class testProcess_12 : public ToopsProcess
{
public:
    testProcess_12(ToopsProcessor &p, const char*n, uint loops, simtime cT,
                   testChannel_1* c, int listDebugging=0 );
    ~testProcess_12() { delete tp_ts; delete socket_name; }
    void behavior(void);
    void write(int d,int m) const { ToopsProcess::write(d,m);}
private:
    simtime tp_cT;      // time for each consumeTime
    uint tp_loops;      // number of loops
    testSocket_1* tp_ts; // our socket
    char *socket_name;
    uint counter;       // counter for sent messages
    int tp_list;        // list debugging enabled, if != 0
};

// ----------------------------------------------------------------------------
// testMessage_2 and testSocket_2 have been made less type safe than .._1.
// Reason: they transport no class specific information. 
//

// ----------------------------------------------------------------------------
// testMessage_2
//
// testMessage_1 just transports a number and a pointer to the sender' name.
class testProcess_13;
class testMessage_2 : public ToopsMessage
{
public:
    testMessage_2(uint msgCounter, const char * sender) : ToopsMessage()
        { m_counter = msgCounter; m_sender = sender;}
    uint counter(void) const { return m_counter; }
    const char * sender(void) const { return m_sender; }
    virtual ToopsMessage* duplicate(void)
        { return new testMessage_2(m_counter, m_sender); }
    virtual void write(int,int) const;
private:
    uint m_counter;
    const char * m_sender;
};

// ----------------------------------------------------------------------------
// testSocket_2
//
class testSocket_2 : public ToopsSocket
{
public:
    testSocket_2(const char* name, ToopsProcess* owner, ToopsChannel *channel) :
         ToopsSocket(name, owner, channel) {}
    void send(testMessage_2* m, simtime d = 0)
        { ToopsSocket::send((testMessage_2*)m,d); }
    testMessage_2* receive(void)
        { return (testMessage_2*)ToopsSocket::receive(); }

    testMessage_2* getMsg (instance_counter num=0)
        { return (testMessage_2*) ToopsSocket::getMsg(num); }
    testMessage_2* touchMsg (instance_counter num=0)
        { return (testMessage_2*) ToopsSocket::touchMsg(num); }
};



// ----------------------------------------------------------------------------
// testProcess_13
// testProcess_13 runs on ToopsProcessor <p>, has the name <n> and a fixed priority
// of 1. It will run a loop <loops> times, consuming <cT> each loop.
// A testSocket_2 will be set up (named socket_<n>). It will be used to send
// each loop a testMessage_2 to <c>. The messages contains a counter, that is
// incremented each time the message is sent.
// testProcess_13 only sends messages.
// Tests the receive mechanism together with testProcess_14.

class testProcess_13 : public ToopsProcess
{
public:
    testProcess_13(ToopsProcessor &p, const char*n, uint loops, simtime cT,
                   ToopsChannel* c);
    ~testProcess_13() { delete tp_ts; delete socket_name; }
    void behavior(void);
    void write(int d,int m) const { ToopsProcess::write(d,m);}
private:
    simtime tp_cT;      // time for each consumeTime
    uint tp_loops;      // number of loops
    testSocket_2* tp_ts; // our socket
    char *socket_name;
    uint counter;       // counter for sent messages
};

// ----------------------------------------------------------------------------
// testProcess_14
// testProcess_14 runs on ToopsProcessor <p>, has the name <n> and a fixed priority
// of 1. It will run a loop <loops> times, consuming <cT> each loop.
// A testSocket_2 will be set up (named socket_<n>). It will receive()
// each loop.
// testProcess_14 only receives messages.
// <rec> decides, which of the receive function shall be used:
//   0  ToopsSocket::receive()
//   1  ToopsProcess::receive(void)
//   2  ToopsProcess::receive( ToopsSocket* t1,...) (tests also
//        ToopsProcess::receive( ToopsSocket**), since the function maps to the
//        second one)
// If <listDebugging> is > 0, the ToopsSocket::write() function is output before
// the messages are removed from testSocket_1.
// Tests the receive mechanism together with testProcess_13.

class testProcess_14 : public ToopsProcess
{
public:
    testProcess_14(ToopsProcessor &p, const char*n, uint loops, simtime cT,
                   ToopsChannel* c, int rec, int listDebugging=0 );
    ~testProcess_14() { delete tp_ts; delete socket_name; }
    void behavior(void);
    void write(int d,int m) const { ToopsProcess::write(d,m);}
private:
    simtime tp_cT;      // time for each consumeTime
    uint tp_loops;      // number of loops
    testSocket_2* tp_ts; // our socket
    char *socket_name;
    int tp_list;        // list debugging enabled, if != 0
    int tp_rec;
};

// ----------------------------------------------------------------------------
// testProcess_15
// testProcess_15 runs on ToopsProcessor <p>, has the name <n> and a fixed priority
// of 1. It will run a loop <loops> times, consuming <cT> each loop.
// A testSocket_2 will be set up (named socket_<n>). It will receive()
// each loop.
// testProcess_15 only receives messages via testSocket_2::receive().
// A ToopsTimer named timer_<n> is started before each receive() with time
// <tim>.
// Tests the interruption of a receive() by a ToopsTimer.

class testProcess_15 : public ToopsProcess
{
public:
    testProcess_15(ToopsProcessor &p, const char*n, uint loops, simtime cT,
                   simtime tim, ToopsChannel* c);
    ~testProcess_15()
    {
        delete tp_ts; delete socket_name;
        delete tp_ti; delete timer_name;
    }
    void behavior(void);
    void write(int d,int m) const { ToopsProcess::write(d,m);}
private:
    simtime tp_cT;      // time for each consumeTime
    uint tp_loops;      // number of loops
    simtime tp_timer;
    ToopsTimer* tp_ti;
    char *timer_name;
    testSocket_2* tp_ts; // our socket
    char *socket_name;
};


#endif
