#if !defined(TIMESTEP_H)
#define TIMESTEP_H

#include "toops.h"
#include "to_list.h"
#include "procunit.h"

//-----------------------------------------------------------------------------
// ToopsTimeStep (source code --> timestep.cpp/.C)
// ===========================================
//
//
//
class ToopsTimeStep : public ToopsObjectLink
{
public:
    // The types of ToopsTimeStep:
    // normal: the simulation shall continue, when that ToopsTimeStep is reached
    // timeOutIndicator: stop the simulation, when that ToopsTimeStep is reached
    enum _type { normal=0, timeOutIndicator=1 };

    // Sets up a ToopsTimeStep with _type t
    ToopsTimeStep (simtime time, _type t ) : 
         ToopsObjectLink(( ToopsObject*)this)
        { t_time = time; t_type = t; }

    // Set up a ToopsTimeStep holding a ToopsTimer, ToopsSocket or ToopsProcess
    ToopsTimeStep (simtime time, ToopsTimerLink* tl) : 
         ToopsObjectLink(( ToopsObject*)this)
        { t_time = time; t_type = normal; t_timers.insert(tl); }
    ToopsTimeStep (simtime time, ToopsProcessLink* tp) : 
         ToopsObjectLink(( ToopsObject*)this)
        { t_time = time; t_type = normal; t_processes.insert(tp); }
	 ToopsTimeStep (simtime time, ToopsSocketLink* ts) :
        ToopsObjectLink(( ToopsObject*)this)
        { t_time = time; t_type = normal; t_sockets.insert(ts); }
    ~ToopsTimeStep() {};

    // Add a ToopsTimer,... to a ToopsTimeStep
    void add(ToopsTimerLink *toAdd)
    {
        // no sort order necessary, because timer events are managed before
        // processes.
        t_timers.append(toAdd);
    }

    void add(ToopsSocketLink *toAdd)
    {
        // no sort order necessary, because socket events are managed before
        // processes.
        t_sockets.append(toAdd);
    }

    void add(ToopsProcessLink *tp);  // sorting necessary!

	 // Return the next event held in a ToopsTimeStep in that special order:
    // first all ToopsSockets, then all ToopsTimers, at least all ToopsProcesses, to
    // guarantee a explicit order in managing the events and to make sure
    // that timer alerts and message arrivals are managed before ToopsProcesses
    ToopsSimulation* nextEvent(void)
	 {
        if (t_sockets.len())
        {
            ToopsSocket* next = 0;
            t_sockets.reset();
            t_sockets.getnext(next);
            return (ToopsSimulation*) next;
        }
        else if (t_timers.len())
        {
            ToopsTimer* next = 0;
            t_timers.reset();
            t_timers.getnext(next);
            return (ToopsSimulation*) next;
        }
        else if (t_processes.len())
        {
            ToopsProcess* next = 0;
            t_processes.reset();
            t_processes.getnext(next);
            return (ToopsSimulation*) next;
        }
		  else
            return 0;
    }

    int isEmpty(void) const
		  { return !(t_timers.len() || t_processes.len() || t_sockets.len()); }

    _type type(void) const { return t_type; }

    simtime time(void) const { return t_time; }

    // virtual void write(int depth = 0, int mode = 0) const; // pt 9.94
    virtual void write(int depth = 0, int mode = 0);     
    
    DECLARE_CLASSINFO(ToopsTimeStep); 
private:
    simtime t_time;
    _type t_type;
    ToopsTimerList t_timers;
    ToopsSocketList t_sockets;
    ToopsProcessList t_processes;
    // prevent compiler from using copy / default constructor and assignment
    ToopsTimeStep();
    //ToopsTimeStep(const ToopsTimeStep&); //mjk 280895
    void operator=(const ToopsTimeStep&);
};


//-----------------------------------------------------------------------------
// ToopsTimeStepList (source code timestep.cpp/.C)
// ===========================================

// Class ToopsTimeStepList forms a doubly linked list with a built in iterator.
// See ToopsTimerList(in timer.h) for explanations (ToopsTimerList works exactly the
// same way).
class ToopsTimeStepList : public ToopsObjectList
{
public:
    ToopsTimeStepList() : ToopsObjectList() {}
    ~ToopsTimeStepList() {}
    void insert(ToopsTimeStep &r) { insert(&r); }
    void insert(ToopsTimeStep *p) { _insert((ToopsObjectLink*)p); }
    void append(ToopsTimeStep &r) { append(&r); }
    void append(ToopsTimeStep *p) { _append((ToopsObjectLink*)p); }
    void inserthere(ToopsTimeStep &r) { inserthere(&r); }
    void inserthere(ToopsTimeStep *p) { _inserthere((ToopsObjectLink*)p); }
    void appendhere(ToopsTimeStep &r) { appendhere(&r); }
    void appendhere(ToopsTimeStep *p) { _appendhere((ToopsObjectLink*)p); }
    ToopsTimeStep *current(void) { return (ToopsTimeStep*) _current(); }
    ToopsTimeStep *first(void) { return (ToopsTimeStep*) _first(); }

    int next(ToopsTimeStep*& rp)
    {
        if (_setnext())
            { rp = (ToopsTimeStep*) _current(); return 1; }
        else
            { rp = 0; return 0; }
    }
    int prev(ToopsTimeStep*& rp)
    {
        if (_setprev())
            { rp = (ToopsTimeStep*) _current(); return 1; }
        else
            { rp = 0; return 0;}
    }
    int getnext(ToopsTimeStep*& rp)
    {
        ToopsTimeStep * pptmp;
        if ((pptmp = (ToopsTimeStep*) _getnext()) != 0)
            { rp = pptmp; return 1; }
        else
            { rp = 0; return 0; }
    }
        
    void remove(ToopsTimeStep* t) { _remove((ToopsObjectLink*)t); }

    // Sort in a given link to an event.
    void sortIn(ToopsProcessLink* toSort);
    void sortIn(ToopsTimerLink* toSort);
    void sortIn(ToopsSocketLink* toSort);

    virtual void write(int depth = 0, int mode = 0) const;
    
    DECLARE_CLASSINFO(ToopsTimeStepList);
    
private:
    // prevent compiler from using copy constructor and assignment
    //ToopsTimeStepList(const ToopsTimeStepList&); //mjk 280895
    void operator=(const ToopsTimeStepList&);
};

#endif
