#if !defined(TO_LIST_H)
#define TO_LIST_H

#include "toops.h"

// to_list.h declares the basic list type used in TOOPS. All these lists are
// specialized for their intended purposes. They are not to be considered
// for general use as container classes.
//
// The design of the doubly linked lists is heavily inspired by Tony L.
// Hansens class dlink (The C++ Answer Book, pp. 361, Addison-Wesley),
// but the links are designed as parts of the linked instances to keep
// the heap clean. An element counter has been introduced and more
// functionality had been added. The addition of a pointer to the list
// the object is held in made the lists safer and better usable.

// The lists work only with classes derived from ToopsObject.
typedef ToopsObject* ENTRY;

class ToopsObjectList;

//-----------------------------------------------------------------------------
// ToopsObjectLink (source code --> to_list.cc)
// ============================================

// To be declared as member of the class to be linked. Class ToopsTimeStep and
// class ToopsMessage inherit directly to make their handling easier.
class ToopsObjectLink : public ToopsObject
{
friend class ToopsObjectList;
public:
    ToopsObjectLink(ENTRY w) : ToopsObject()
        { who = w; next = prev = 0; list = 0; }
    virtual void write(int depth = 0, int mode = 0) const;
    virtual ~ToopsObjectLink() {}

    // check, if and to what the object is linked
    int isLinked(void) { return (next !=0); }
    ToopsObjectList* linkedTo(void) { return list; }
    
    DECLARE_CLASSINFO(ToopsObjectLink); // pt 9.94    

protected:
    ENTRY _object(void) const { return who; }
    void unlink(void);
    ToopsObjectLink* _next(void) const { return next; }
    ToopsObjectLink* _prev(void) const { return prev; }

private:
    ToopsObjectLink *next;   // next element in list
    ToopsObjectLink *prev;   // previous element in list
    ENTRY who;           // points to the linked object
    ToopsObjectList* list;   // the list, the object is linked in
    void insert(ToopsObjectLink* n, ToopsObjectList *l);  // insert a link before this
    void append(ToopsObjectLink* n, ToopsObjectList *l);  // append a link after this

    // prevent compiler from using copy / default constructor and assignment
    ToopsObjectLink() {}
    ToopsObjectLink(const ToopsObjectLink&) {}
    void operator=(const ToopsObjectLink&) {}
};


//-----------------------------------------------------------------------------
// ToopsObjectList (source code --> to_list.cpp/.C)
// ============================================
//
// A ToopsObjectList forms a doubly linked list with an element counter and
// a built in iterator.
//
// ToopsObjectList is not used directly. Instead of this, the list classes used
// in TOOPS are derived from ToopsObjectList (or ToopsNamedObjectList) to set up
// a type safe interface to ToopsObjectList (see ToopsMessageList for example).
// The protected functions (their names all beginning with an underscore to
// indicate their internal status) are exported by the list classes with
// respect to type safety (only the needed functions are exported) and
// supplemented with class specific functions.
class ToopsObjectList : public ToopsObject
{
public:
    // get the number of elements contained in the list.
    instance_counter len() const { return elements;}

    // reset the iterator
    void reset(void) { current = 0; }

    virtual void write(int depth=0, int mode=0) const;
    
    DECLARE_CLASSINFO(ToopsObjectList); 

protected:
    ToopsObjectList() : ToopsObject()
        { elements = (instance_counter)0; current = 0 ;last = 0; }
    virtual ~ToopsObjectList() {}

    // Set current to a specified link.
    void _set(ToopsObjectLink &l) { _set(&l); }
    void _set(ToopsObjectLink *l);

    // Insert at the lists top (current remains unchanged).
    void _insert(ToopsObjectLink &ri) { _insert (&ri); }
    void _insert(ToopsObjectLink *pi);

    // Append at the lists end (current remains unchanged).
    void _append(ToopsObjectLink &ri) { _append (&ri); }
    void _append(ToopsObjectLink *pi);

    // Insert before the current entry. If the current entry is not set,
    // insert at the list's top (current remains unchanged).
    void _inserthere(ToopsObjectLink &ri) { _inserthere (&ri); }
    void _inserthere(ToopsObjectLink *pi);

    // Append after the current entry. If the current entry is not set,
    // append it at the list's end (current remains unchanged).
    void _appendhere(ToopsObjectLink &ri) { _appendhere (&ri); }
    void _appendhere(ToopsObjectLink *pi);

    // Read the current, first or last element (current remains unchanged).
    ENTRY _current(void) const
    {
        if (current)
            return current->who;
        else
            return 0;
    }
    ENTRY _last(void) const
    {
        if (last)
            return last->who;
        else
            return 0;
    }
    ENTRY _first(void) const
    {
        if (last && last->next)
            return last->next->who;
        else
            return 0;
    }

    // Read the current, first or last link (current remains unchanged).
    ToopsObjectLink* _currentLink(void)
    {
        if (current)
            return current;
        else
            return 0;
    }
    ToopsObjectLink* _lastLink(void)
    {
        if (last)
            return last;
        else
            return 0;
    }

    ToopsObjectLink *_firstLink(void)
    {
        if (last && last->next)
            return last->next;
        else
            return 0;
    }

    // Set the current pointer to the next or previous element.
    // These functions return 1 if successful, else 0. If unsuc-
    // cessful, the current element will be reset!
    int _setnext (void);
    int _setprev (void);

    // Get the next (previous) element and remove it from the list. The
    // actual remains the current element, if current is set. If current
    // is not set (initially or after a reset() e.g.), the first (last)
    // element will be passed to e. If the list is empty or there is
    // no element after (before) the current one, 0 will be returned
    // and the current element pointer will be reset. If a next (previous)
    // element exists, it will be returned.
    ENTRY _getnext();
    ENTRY _getprev();

    // The same like the functions above, but returning the link* instead
    // of the object*.
    ToopsObjectLink *_getnextlink();
    ToopsObjectLink *_getprevlink();

    // remove a given link, current will be reset. Be careful with this,
    // for the sake of performance no check is taken, whether the
    // link is in _this_ list, it will be removed anyway!!!!!
    // (debug version does an assert on this)
    void _remove(ToopsObjectLink*);

    // Get the <num>th element from the list. <num> starts with 0 to be
    // compatible with C++ arrays. Returns 0 if <num>th element not available.
    // 'Get' means here, that the element will be removed from the list.
    ToopsObject* _get(instance_counter num);

    // Get the <num>th element from the list. <num> starts with 0 to be
    // compatible with C++ arrays. Returns 0 if <num>th element not available.
    // 'touch' means here, that the element will remain in the list.
    ToopsObject* _touch(instance_counter num);

    // Delete the <num>th element in the list.
    void _del(instance_counter num);

    // Returns an array of pointers to ToopsObject, that holds all ToopsObjects
    // currently in the list. The length of the array is equal to the
    // list's len(). The array is allocated via new and _must_ be deleted
    // by the user after using it.
    ToopsObject** _touchAll(void);

    void writeList(int depth) const;

private:
    ToopsObjectLink *last;                        // points to the last element
    ToopsObjectLink *current;                     // the current element
    instance_counter elements;                // number of contained elements

    // prevent compiler from using copy constructor and assignment
    ToopsObjectList(const ToopsObjectList&) {}
    void operator=(const ToopsObjectList&) {}
};

#endif // #if !defined(TO_LIST_H)

