
//added 2 April 1995 by Manfred Kraess:
#ifndef __cplusplus
#error Must use C++ to compile TOOPS.
#endif

#ifndef TOOPS_H
#define TOOPS_H
#define NDEBUG

#include <string.h>

//-----------------------------------------------------------------------------
// some typedef's to ease the declarations
typedef unsigned int uint;
typedef unsigned long ulong;
typedef unsigned short ushort;

typedef void (*PVFPC)(const char * const);

// The following typedefs are introduced with respect to hold TOOPS
// as portable as possible. They are part of the programming interface.
// Change these types and recompile the whole system, if one of these
// types is too small or to big (performance !!) for the target system
// (processor / compiler / Your program).

// This type holds the simulation time. On most 16 and 32 bit compilers
// a unsigned long (usually 32 bit) should be sufficient even for a very
// high granularity of the simulation.
typedef ulong simtime;

// This type is a counter for the created instances of each type / all types.
// A maximum of 64k objects (on a PC) should be sufficient.
// Objects refers to the sum of a ToopsProcesses, ToopsProcessors, ToopsSockets,
// ToopsChannels and ToopsTimers. Simultaneous existing ToopsMessages are
// also up to 64k.
typedef ushort instance_counter;

// This type is used as a kind 'register' that indicates the amount
// of TOOPS' internal debug information to be displayed. Always use a
// unsigned type that is able to hold at least 16 bit.
typedef ushort dbgreg_t;

//-----------------------------------------------------------------------------
// some forward declarations for classes used internally by TOOPS
class ToopsProcessorList;   // declared in procunit.h
class ToopsChannelList;     // declared in channel.h
class ToopsProcessList;     // declared in tprocess.h
class ToopsTimerList;       // declared in timer.h
class ToopsMessageList;     // declared in message.h
class ToopsSocketList;      // declared in socket.h

class ToopsProcessorLink;   // declared in procunit.h
class ToopsChannelLink;     // declared in channel.h
class ToopsProcessLink;     // declared in tprocess.h
class ToopsTimerLink;       // declared in timer.h
class ToopsMessageLink;     // declared in message.h
class ToopsMessage;         // declared in message.h
class ToopsSocketLink;      // declared in socket.h

class ToopsContext;         // declared in context.h

class ToopsTimeStep;        // declared in timestep.h
class ToopsTimeStepList;    //                "

//-----------------------------------------------------------------------------
// ToopsError (source code -> error.cpp/.C)
// ====================================
// Documentation: see error.h (the error numbers for all TOOPS classes
//                are also defined there)
class ToopsError
{
public:
    enum severity { fatal, error, warning, ignore };
    typedef void (*ERRFUNC) (int, ToopsError::severity, const char*);
    ToopsError(int errno, severity s, const char *who, const char *desc);
    static void setErrorHandler(ERRFUNC);
    severity setSeverity (severity newSeverity);
    severity getSeverity(void) const { return sev; }
    void err(const char *xwho = 0, const char *xdesc1 = 0,
             const char *xdesc2 = 0, const char *xdesc3 = 0,
             const char *xdesc4 = 0, const char *xdesc5 = 0 );
private:
    int number;
    severity sev;
    const char *whocausedit;
    const char *description;
    static ERRFUNC userErrorHandler;
};

//-----------------------------------------------------------------------------
// Class ToopsClassInfo holds the available runtime information for all classes
// derived from class ToopsObject, which serves as baseclass for mostly ALL TOOPS
// classes belonging to the TOOPS user interface (some internal classes
// are not inherited from ToopsObject).

// ToopsClassInfo (source code -> tobject.cpp )
// ========================================
class ToopsClassInfo
{
friend class ToopsObject;
public:
    // construction
    ToopsClassInfo( const char *name, ToopsClassInfo* base);
    // get the class' name
    const char * name(void) const { return t_className;}
    // get the class info for the parent class, if there is one    
    
    //2nd const removed to avoid Microsoft C++ warning:  3 Apr 95 mjk
    //const ToopsClassInfo * const base(void) const
    const ToopsClassInfo * base(void) const
        { return t_baseClassInfo; }
    // get the unique class id
    uint nClassID(void) const { return t_nClassID; }
    
    ToopsClassInfo();  // used for ToopsObject (-> tobject.cpp) / pt 9.94

private:
    const char *t_className;                 // name
    ToopsClassInfo *t_baseClassInfo;             // parent
    ToopsClassInfo *t_nextClass;                 // next link in the list of
                                             // all registered classes
    uint t_nClassID;                         // unique id number
    static ToopsClassInfo *t_firstClass;         // 'hook' for the list of
                                             // all registered classes
    static uint         t_nClassIDctr;       // counter for registered
                                             // classes

    // Set up assignment operator, default and copy constructor as
    // private members to prevent the compiler and the user from using it.
    //    ToopsClassInfo();  // used for ToopsObject (-> tobject.cpp) / pt 9.94 jetzt public Z.103
    ToopsClassInfo( ToopsClassInfo&);
    void operator=(const ToopsClassInfo&);

    // add debugging aids, if required
#if !defined(NDEBUG)
private:
    dbgreg_t t_DebugControl;
public:
    dbgreg_t& GetDebugControl() {return t_DebugControl; }
    dbgreg_t DebugControl() { return t_DebugControl; }
#endif
};

//-----------------------------------------------------------------------------
// ToopsObject (source code -> tobject.cpp)
// ====================================
// ToopsObject is the abstract base class of mostly all TOOPS classes. It has
// no nonstatic data menbers, but it's methods make up the TOOPS debug
// system and provide runtime class information. Parts of the design
// were influenced by the class library MFC 1.0, delivered with Microsofts
// C/C++ 7.0.
// Possible extension to be located here: serialisation of objects (->MFC)
class ToopsObject
{
public:
    virtual ~ToopsObject() {}

    // Get the correct runtime class information.
    virtual ToopsClassInfo* GetClassInfo(void) const;

    // Get the runtime class info of this class
    static ToopsClassInfo* ThisClassInfo(void) { return &classToopsObject; }

    // Write the object's state, depth = 0 means for the class this
    // function is called for, 1 means that class and it's base,...
    // mode = 0 means full information, 1 reduced information.
    // Since ToopsObject has no nonstatic members, no output for ToopsObject is needed.
    virtual void write(int depth = 0, int mode = 0) const {};

	 // returns 1, if 2 ToopsObjects are of the same class, otherwise 0;
    int isSameClass(const ToopsObject* toCompare) const
        { return ( GetClassInfo()->nClassID()
                   == toCompare->GetClassInfo()->nClassID()); }

    // returns 1, if toCompare is a base class (direct or indirect)
    // of this, otherwise 0.
    int isInheritedFrom(const ToopsObject *toCompare) const;

    static ToopsClassInfo classToopsObject;

    // Sets an user defined function for all TOOPS internal output
    // except errors and returns the old function. If not set or set to
    // 0, TOOPS will direct all output via cout.
    static PVFPC setOutputFunction (PVFPC newOutPutFunction);

    // Unconditional output to the debug output function
    static void DBGout(const char* text);

    // Add the conditional debugging output function: text will be output
    // only, if (description & classDebugRegister) > 0 (see the macros below)
#if !defined(NDEBUG)
    static void DBGout(dbgreg_t description,
                dbgreg_t classDebugRegister,
                const char* const text);
#endif

protected:
    ToopsObject() {}
    static ToopsError fOutOfMem;

private:
    // prevent compiler from using copy constructor and assignment
    //ToopsObject(const ToopsObject&); //mjk 280895
    void operator=(const ToopsObject&);

    static PVFPC userOutputFunction;         // takes the strings returned
                                             // from the DBG... macros
                                             // and from the write()
                                             // functions.
};

//-----------------------------------------------------------------------------
// This macro has to be placed at the end of each class' declaration that
// is derived (directly or indirectly) from ToopsObject. If it is left out,
// no runtime information about this class will be available.
// Argument: name is the class' name _WITHOUT_ quotation marks.
#define DECLARE_CLASSINFO(name)                                           \
    static ToopsClassInfo class##name;                                        \
    virtual ToopsClassInfo* GetClassInfo(void) const;                         \
    static ToopsClassInfo* ThisClassInfo(void) { return &class##name; }

// This macro has to be placed in the file (or in only ONE of the files)
// that set up Your class' functions (a .cpp or .C file).
// Arguments: name is the class' name
//            basename is the parent class' name (both _WITHOUT_ quotation
//              marks)
#define IMPLEMENT_CLASSINFO(name, basename)                        \
    ToopsClassInfo name::class##name (#name,                       \
                                   &basename::class##basename);        \
    ToopsClassInfo* name::GetClassInfo(void) const                 \
    { return &name::class##name; }


// The following macros specify the TOOPS debug system:
// All output done via these functions is redirected to the user output
// function set in ToopsObject::setOutputFunction().
// This system works with all classes derived from ToopsObject (you can use
// the debug version of the library, toops_d.lib / toops_dlib.a, if
// you assume an error in the TOOPS library.).
// How to use: for example --> t_tobj.h, tobject.cpp/.C
// For final version: compile with NDEBUG defined and link with the nondebug
// library (toops.lib / toopslib.a) and no performance overhead will be
// caused by leaving all debug information in the program.
#if !defined(NDEBUG)
    #define DBG_INIT()        ostrstream _dbgstream;
    #define DBG(info)         _dbgstream << info;
    #define DBG_WRITE(descr)  _dbgstream << ends;                         \
                              char *_dbgstring = _dbgstream.str();        \
                              ToopsObject::DBGout(descr,                      \
                              ThisClassInfo()->DebugControl(),_dbgstring);\
										delete _dbgstring;
    #define DBG_OUT(text, descr) ToopsObject::DBGout(descr,                    \
                               ThisClassInfo()->DebugControl(),text);
    #define DBG_DISABLE(classname, what)                                  \
                      classname::ThisClassInfo()->GetDebugControl() &=(~what);
    #define DBG_ENABLE(classname, what)                                   \
                      classname::ThisClassInfo()->GetDebugControl() |=what;
    #define _DBG(action,descr)  if (ThisClassInfo()->GetDebugControl()    \
                                   & descr) action;
#else
    #define DBG_INIT()
    #define DBG(info)
    #define DBG_WRITE(descr)
    #define DBG_OUT(text, descr)
    #define DBG_DISABLE(classname, what)
    #define DBG_ENABLE(classname, what)
    #define _DBG(action,descr)
#endif
#define DBGends "-----\n"

//-----------------------------------------------------------------------------
// ToopsSimulation (source code --> tsim.cpp/.C)
// =========================================
//
class ToopsSimulation : public ToopsObject
{
public:
	 // simstate describes the circle of states in a simulation:
    // - clear:  no registered simulation objects exist, a new simulation
    //           can be set up.
    // - notStarted: at least one registered object exists and the
    //               simulation can be started.
    // - running: the simulation has been started.
    // - stopped: the sim has been stopped or run to it's end, no new
    //            sim can be started.
    // - clear: all registered objects have been destroyed, all internal
    //          statics reset, a new simulation can be set up.
    enum simstate { clear, notStarted, running, stopped };
    // Get the simulation's state.
    static simstate state(void) { return s_state; }

    // The property 'responsibility' of a TOOPS object determines the way,
    // TOOPS manages the destruction of this object:
    // - user: the user _must_ care for the object's destruction:
    //   + if it is an automatic object (placed on the stack), let it go
    //     out of scope, when the simulation has ended.
    //   + if it is in the free store, delete it.
    //   + if it static, it will not be destroyed before the program
    //     quits. NOTE: if only one STATIC simulation object exists, ONLY
    //     ONE SIMULATION can be started per program run. It is NOTrecommended
    //     at all to create static simulation objects.
    //   NOTE: all objects created not in the free store MUSThave the
    //         responsibility user (this will be detected for automatic objects,
    //         but there is no portable way to do so for static objects)
    // - TOOPSatEnd: TOOPS will destroy the object, before control returns
    //               from ToopsSimulation::start().
    // - TOOPSdirect: TOOPS will destroy the object as early as possible.
    //               (a ToopsProcess, when it is terminated; a ToopsTimer or ToopsSocket,
    //                when the owner terminates, ToopsProcessors, ToopsChannels and
    //                ToopsMessages at simulation end)
    // NOTE: For ToopsProcessor, ToopsChannel and ToopsProcess objects responsibility can
    //       be selected at creation time. ToopsTimers and ToopsSockets always have
    //       the same responsibility as their owner (ToopsProcess).
    // NOTE: NEVER set up automatic TOOPS objects, when the simulation has
    //       started. Reason: the stack is moved for starting coroutines,
    //       so adresses on the stack are not reliable. This will cause a
    //       FATAL ToopsError, and the simulation will be exited immediatly.
    // NOTE: be careful with pointers or references to objects with a TOOPS...
    //       responsibility, the objects might have been deleted.
    enum responsibility { user = 0, TOOPSatEnd = 1, TOOPSdirect = 2 };
    responsibility responsible(void) const { return s_responsibility; }

    // Why stopped the simulation ?
    // timeStop: timeOut was reached
    // regularStop: no events left, i.e all ToopsProcesses have terminated
    //              or are untimed waiting
    // userStop: the user program called stopSimulation()
    // timerOverFlowStop: (shouldn't really happen) TOOPS stopped the
    //                    simulation, because the time exceeded SIMTIME_MAX
    //                    (declared in portable.h)
    // errorStop: a ToopsError of severity ToopsError::error occured or the user
    //            program called stopWithError()
    // _next: internally used, is never returned by ToopsSimulation::start()
    enum stopReason { timeStop=1, regularStop, userStop,
                      timeOverFlowStop, errorStop, _next};

    // The actual simulation time (if state() is 'running'); always 0, if
    // 'clear' or 'notStarted', the stop time, if state() is 'stopped'
    static simtime time(void) { return s_time; }

    // The total number of instances of classes derived from ToopsSimulation
    static instance_counter objects(void)  { return s_objects;}
    // Each registered simulation object holds an unique number as ID.
    instance_counter order(void) const { return s_order;}

    // Start the simulation
    // stop at <end> or run until one of the other stop reasons appears
    static stopReason start(simtime end = 0);

    static void stopSimulation(void);  // stop immediatly, return userStop
    static void stopWithError(void); // stop immediatly, return errorStop

    // Get the time the simulation is about to end.
    static simtime getTimeOut(void) { return timeOut; }

    // How many instances do exist ?
    static instance_counter numProcessors(void);
    static instance_counter numChannels(void);
    static instance_counter numMessages(void);
    static instance_counter numProcesses(void);
    static instance_counter numTimers(void);
    static instance_counter numSockets(void);

    virtual ~ToopsSimulation();

    // For graphical representation (and debugging): set a function, that
    // is called each time the time advances. The function must take 2
    // parameters: <now> holds the actual, <next> the next point of time.
    // Defaults to 0. Reset to 0, if function not needed.
    typedef void (*PVF2S)(simtime now, simtime next);
    static PVF2S setTimeAdvanceCallBack(PVF2S);  // returns old value

    // to control debug output
    enum dbgInfo { cstr=0, onStart=1, onGetNextEvent=2,
                   dstr=4 };
    virtual void write(int depth = 0, int mode = 0) const;
    
    DECLARE_CLASSINFO(ToopsSimulation); 

protected:
    // The next three function are called for all ToopsProcessors, ToopsChannels,
    // ToopsTimers, ToopsSockets and ToopsProcesses at a defined point of time:

    // atStart() is called after ToopsSimulation::start(), but before any
    // any ToopsProcess starts(note: ToopsProcessor::atStart() is used internally,
    // if you want to add something, call ToopsProcessor::atStart() in your
    // class' atStart() function, if you don't, the simulation will quit).
    // atStart() will not be called for objects set up while the simulation
    // is running.
    virtual void atStart(void) {}

    // atEnd() is called after the simulation stopped. It will not be called
    // for objects deleted while the simulation is running.
    virtual void atEnd(void) {}

    // atStateChange will be called every time, an object's state is modified.
    // note: the C++ definition allows the call of virtual functions in
    // constructors, but when virtual functions are called in a constructor,
    // the function invoked is the function defined for the constructor's
    // own class (or inherited from it's bases), but NEVER the function
    // of a derived class. So: atStateChange() never sees the status create,
    // that is defined for several classes of TOOPS (e.g. ToopsTimer, ToopsProcess,..)
    virtual void atStateChange(void) {}

    ToopsSimulation (responsibility who);

    virtual void activate(void) {};  // don't ever use these functions
    inline void nextEvent(void);
    static void _stop(stopReason r); //(inline ?)

    // Get the lists
    static ToopsProcessorList * processors(void)   { return s_processors; }
    static ToopsChannelList   * channels(void)     { return s_channels; }
    static ToopsMessageList   * allMessages(void)  { return s_allMessages; }
    static ToopsProcessList   * allProcesses(void) { return s_allProcesses;}
    static ToopsTimerList     * allTimers(void)    { return s_allTimers; }
    static ToopsSocketList    * allSockets(void)   { return s_allSockets; }
    static ToopsTimeStepList  * timeSteps(void)    { return s_timeSteps; }
    static ToopsProcessorList * processorsToActivate(void)
           { return s_processorsToActivate; }

private:

	 void initSimulation(responsibility who);
    static void callAtStart(void);
    static void callAtEnd(void);
    static void cleanUp(void);

    static void activateProcessors(void);
    static ToopsSimulation* getNextEvent(void);

    static simtime timeOut;                  // time for simulation stop

    static ToopsProcessorList *s_processors;     // lists (->initSimulation())
    static ToopsProcessorList *s_processorsToActivate;
    static ToopsChannelList   *s_channels;
    static ToopsMessageList   *s_allMessages;
    static ToopsProcessList   *s_allProcesses;
    static ToopsTimerList     *s_allTimers;
    static ToopsSocketList    *s_allSockets;
    static ToopsTimeStepList  *s_timeSteps;

    static ToopsContext *mainContext;            // stores the jmp_buf for stop

    static simtime s_time;                   // the actual simulation time
    static simstate s_state;                 // the state of the simulation

    static PVF2S s_timeAdvanceCallBack;

    static instance_counter s_objects;       // total number of objects of
                                             // all derived classes
    instance_counter s_order;                // unique number for each instance
    responsibility s_responsibility;         // who must delete the object

    static ToopsError fCannotStart;
    static ToopsError fNoProcessor;
    static ToopsError fNoProcesses;

    // prevent compiler from using copy / default constructor and assignment
    //ToopsSimulation (ToopsSimulation& s); //mjk 280895
    ToopsSimulation();
    void operator=(const ToopsSimulation&);

    friend class ToopsMessage;
};

//-----------------------------------------------------------------------------
// ToopsNamedObject (source code --> tsim.cpp/.C)
// ==========================================
// Introduces the concept of named objects.
class ToopsNamedObject : public ToopsSimulation
{
public:
    const char* name(void) const { return szName; }
    virtual void write(int depth = 0, int mode = 0) const;

    // no specific debug information available

    // see error.h
    static ToopsError eName;
    
    DECLARE_CLASSINFO(ToopsNamedObject); 

protected:
    ToopsNamedObject(const char * szN, responsibility who);
    ~ToopsNamedObject() {}

private:
    char *szName;
    // prevent compiler from using copy / default constructor and assignment
    ToopsNamedObject();
    //ToopsNamedObject( ToopsNamedObject&);  //mjk 280895
    void operator=(const ToopsNamedObject&);
};
#endif // if !defined(TOOPS_H)
