#if !defined(TPROCESS_H)
#define TPROCESS_H

#include "toops.h"
#include "timer.h"
#include "socket.h"
#include "context.h"
#include "tno_list.h"

class ToopsProcessor;
class ToopsProcess;
//-----------------------------------------------------------------------------
// ToopsProcessLink (source code --> tprocess.cpp/.C)
// ==============================================
// Set up a link to a ToopsProcess. See also to_list.h.
class ToopsProcessLink : public ToopsNamedObjectLink
{
public:
    ToopsProcessLink (ToopsProcess *p) : 
         ToopsNamedObjectLink(( ToopsNamedObject*)p) {}
    ToopsProcess* object(void) { return (ToopsProcess*) _object();}
    inline void unLink(void); 
    virtual void write(int depth = 0, int mode = 0) const;
    
    DECLARE_CLASSINFO(ToopsProcessLink); // pt 9.94
    
private:
    // prevent compiler from using copy / default constructor and assignment
    ToopsProcessLink();
    //ToopsProcessLink(const ToopsProcessLink&); mjk 280895
    void operator=(const ToopsProcessLink&);
};

//-----------------------------------------------------------------------------
// ToopsProcess (source code --> tprocess.cpp/.C)
// ==========================================
//
//class ToopsProcess : private ToopsNamedObject / pt 9.94
class ToopsProcess : public ToopsNamedObject
{
friend class ToopsProcessor;  // calls only cancelRunning(), setRunning()
friend void ToopsTimer::stateChange(void);   // calls only remember(ToopsTimerLink*);
// pt 9.94 oben: 'void' nach 'friend' eingestetzt
friend void ToopsSocket::stateChange(void);  // calls only remember(ToopsSocketLink*);
// pt 9.94 oben: 'void' nach 'friend' eingestetzt
public:
    // ToopsProcess' basic states:
    // - ready: ToopsProcess wants to become running
    // - running: ToopsProcess is active (i.e. consuming time or the code (that
    //    doesn't consume any simulation time) in behavior is active).
    // - waiting: ToopsProcess is waiting for something.
    // - terminated: ToopsProcess terminated or was terminated by another ToopsProcess.
    // - create, destroy: used internally at construction and destruction
    //    ( and to indicate destruction to to atStateChange())
    enum processState { ready=0, running=1, waiting=2, terminated=3,
                        destroy=7 };
	 processState status() const
		  { return (processState) (p_state & destroy); }

    // The basic states are extended (and precised) by the sub states:
    // - behavior has not been called yet.
    int isNotStarted() const { return (p_state & notStarted); }
    // - the next three sub states describe the kind of ToopsProcess' waiting:
    //   + ToopsProcess suspended itself or was suspended by another ToopsProcess
    int isSuspended()  const { return (p_state & suspended); }
    //   + ToopsProcess called one of the synchronize() functions, or it was called
    //     for one of the ToopsProcess' ToopsSockets.
    int isInSync()  const { return (p_state & inSync); }
    //   + ToopsProcess called setWaiting()
    int isSetWaiting() const { return (p_state & setWait); }
    // - the next two sub states are reset every time a controlling function
    //   like setWaiting, consumeTime, ... is called.
    //   + A ToopsTimer expired since the last time the code was (re)activated.
    int isTimerExpired() const { return (p_state & timerExpired); }
    //   + A ToopsMessage arrived since the last time the code was (re)activated.
    int isNewMessage() const { return (p_state & newMessage); }

    // Get the ToopsProcess' priority
    uint priority(void) const { return p_priority; }

    // Get the ToopsProcess' owner (a ToopsProcessor)
	 ToopsProcessor *owner(void) const { return p_owner; }

/*    // re-export some information lost by the private inheritance:
    ToopsObject::isInheritedFrom;
    ToopsObject::DBGout;
    ToopsObject::isSameClass;
    ToopsNamedObject::name;
    ToopsSimulation::simstate;
    ToopsSimulation::state;
    ToopsSimulation::responsibility;
    ToopsSimulation::responsible;
    ToopsSimulation::stopReason;
    ToopsSimulation::time;
    ToopsSimulation::objects;
    ToopsSimulation::order;
    ToopsSimulation::stopSimulation;
    ToopsSimulation::stopWithError;
    ToopsSimulation::getTimeOut;
    ToopsSimulation::numProcessors;
    ToopsSimulation::numChannels;
    ToopsSimulation::numMessages;
    ToopsSimulation::numProcesses;
    ToopsSimulation::numTimers;
    ToopsSimulation::numSockets;*/	// pt 9.94

    virtual ~ToopsProcess();

    // Get a ToopsProcess* by it's name.
    inline static ToopsProcess* search(const char *n);

    // Get a list of all ToopsProcesses in the system. The list _must_ be deleted
    // by the user. Get the list's length via numProcesses();
    inline static ToopsProcess** systemProcesses(void);

    // Get the number of this ToopsProcess' ToopsTimers.
    instance_counter timers(void) const { return p_timers.len(); }
    // Get a list of all ToopsTimers on this ToopsProcess. The list _must_ be deleted
    // by the user. Length: timers().
    ToopsTimer** myTimers(void)
        { return p_timers.touchAll(); }

    // Get the number of this ToopsProcess' ToopsSockets.
    instance_counter sockets(void) const { return p_sockets.len(); }
    // Get a list of all ToopsSockets on this ToopsProcess. The list _must_ be deleted
    // by the user. Length: sockets().
	 ToopsSocket** mySockets(void)
        { return p_sockets.touchAll(); }

    // used internally
    simtime desiredTime(void) const { return p_desiredTime; }
    simtime restConsumeTime(void) const { return p_restConsumeTime; }
	 static ToopsProcess *thisProcess(void) { return p_thisProcess; }

    static ToopsError wNoTerminate;

    // debugging:
    virtual void write(int depth = 0, int mode = 0 ) const;
    enum DBGinfo { cstr=1, dstr=2, onConsumeTime=4,
                   onSetWaiting=8, onSuspend=16, onResume=32,
                   onTerminate=64, onSync=256, onSetRunning=512,
                   onCancelRunning=1024, onWaitUntilSync=2048,
                   onActivate=4096, onRemember=8192 };

    // If information on the succes or on the reasons of failing for the
    // next functions is needed, pass them a callError*.
    // callError describes the effect of a call to resume(), suspend()
    // and terminate():
    // callOK:  call was succesful
    // the other callErrors can appear in many combinations:
    // callTerminated: call ignored, because the function was called
    //                 for a terminated process
    // callHigher: call ignored, because the priority of the other process
    //             was equal or higher than the caller's priority
    // callToOtherProcessor: call ignored, because the other process is not
    //                       running on the same processor
    // callUseLess: call ignored, because the target process of the function
    //              is already in the state that would be caused by this
    //              function (e.g. resume() for a process, that is not
    //              suspended, ...)
    enum callError { callOK=0, callTerminated=1, callHigher=2,
                     callToOtherProcessor=4, callUseless=8};

    void suspend(callError* e=0); // suspend this or _another_ process on
                                  // the same processor
    void resume(callError* e=0);  // resume _another_ process on the same processor
    void terminate(callError* e=0); // terminate this or _another_ process on
                                  // the same processor

    DECLARE_CLASSINFO(ToopsProcess); // pt 9.94
        
protected:
    // create new processes
    ToopsProcess( ToopsProcessor *p, uint prio, const char *name,
             responsibility r=user);
    ToopsProcess( ToopsProcessor &p, uint prio, const char *name,
             responsibility r=user);

    // that's the user process
    virtual void behavior (void)=0;

    // specify the amount of time consumed by the actions of _YOUR_ tasks
public: void consumeTime(simtime s);  // Attention: This change in accessibility
//                                                  was made in Feb. 28th, 1994 by a
//						    user of TOOPS to call this function
//						    by a class which is not derived from
//						    ToopsProcess.
protected:  void setWaiting(simtime t);   // set this process waiting until time t
                                  // has passed. After this time the process
                                  // will be put in the list of runnable pro-
                                  // cesses and rescheduled ( at time()+t or
                                  // later, if there are ready processes
                                  // with higher priority)

    // The behaviour of the synchronize(...) function defines as follows:
    // - the function returns immediatly, if at least one of the specified
    //   ToopsSockets has a ToopsMessage in it's queue.
    // - otherwise the ToopsProcess will wait until at least one ToopsMessage arrives
    //   at one of the specified ToopsSockets
    //   or a timer has expired
    // synchronize on all ToopsSockets
    void synchronize();
    // synchronize on the specified ToopsSockets
    void synchronize(ToopsSocket *t1, ToopsSocket *t2=0, ToopsSocket *t3=0,
                 ToopsSocket *t4=0, ToopsSocket *t5=0, ToopsSocket *t6=0);
    // synchronize on the ToopsSockets specified in the array. The last element
    // of the array _must_ be 0.
    void synchronize(ToopsSocket** TSarray);

    // explained in toops.h
    virtual void atStart(void) {}
    virtual void atEnd(void) {}
    virtual void atStateChange(void) {}

    virtual void activate(void);   // don't use this at all

private:
    // used to represent the sub states
    enum stateFlags { suspended = 8,
                      inSync = 16,
                      setWait = 32,
                      notStarted = 128,
                      newMessage = 256,
                      timerExpired = 512,
                      i_state = suspended+inSync+setWait+notStarted+
                                newMessage+timerExpired };

    // Variables:

    // the process' state and related stuff
    uint p_priority;                  // priority: 0 is 'highest', 1 is lower..
    ToopsProcessor *p_owner;              // the processor 'running' the process
    stateFlags p_state;
    simtime p_desiredTime;            // the time the process wants to be
                                      // restarted. (absolute sim time!!)
    simtime p_restConsumeTime;        // the time to consume left (rel. time)
    simtime p_setToRunning;           // the last time the process was set
                                      // running (abs. time)
    ToopsContext context;                 // the context of this process

    ToopsTimerList p_timers;              // manage the process' timers
    ToopsSocketList p_sockets;            //   "     "     "     sockets

    ToopsProcessLink sysLink;             // link for the list of all processes
    ToopsProcessLink processorLink;       // link to owner's list

    static ToopsProcess *p_thisProcess;   // the actual active ToopsProcess (i.e. the
                                      // ToopsProcess running it's code)

    static ToopsError fCTNoAccess;
    static ToopsError fSetWaitingNoAccess;
    static ToopsError fDestroyNotTerminated;
    static ToopsError fPriorityToLow;
    static ToopsError fWrongOwner;
    static ToopsError fCannotCreateToopsProcess;
    static ToopsError fSyncNoAccess;
    // modify p_state
    void modifyStatus(processState s)
        { p_state = ( (stateFlags) ((p_state & i_state) | s)); }
    void setState(stateFlags s) { p_state = (stateFlags) (p_state | s); }
    void clrState(stateFlags s) { p_state = (stateFlags) (p_state &(stateFlags)(~s));} // pt 9.94
    //                                        ^ diesen Cast-Op. eingefuegt

    // a security check function (seems to be obsolete)
    int mustrecurse(void);

    int contextSave(void);  // save the ToopsProcess' context
    void restore(void);     // restore the ToopsProcess' context
    void checkCaller(callError&);

    void start(void);       // (re)activates the code in behavior()

    // the only member accessed by class ToopsTimer
    void remember(ToopsTimerLink* t);
    // the only member accessed by class ToopsSocket
    void remember(ToopsSocketLink* s);

    // the only 2 members accessed by ToopsProcessor
    void cancelRunning(void);
    void setRunning(simtime when);

    // called by the constructors, sets up the process
    void initProcess( ToopsProcessor *p, uint prio);

    void waitUntilSync(void);

};

//-----------------------------------------------------------------------------
// ToopsProcessList (source code --> tprocess.cpp/.C)
// ==============================================
// Class ToopsProcessList forms a doubly linked list with a built in iterator.
// ( --> to_list.h, tno_list.h ). See ToopsTimerList (timer.h) for explanations.
class ToopsProcessList : public ToopsNamedObjectList
{
public:
    ToopsProcessList() : ToopsNamedObjectList() {}
    ~ToopsProcessList() {}
    void insert(ToopsProcessLink &r) { insert(&r); }
    void insert(ToopsProcessLink *p) { _insert((ToopsObjectLink*)p); }
    void append(ToopsProcessLink &r) { append(&r); }
    void append(ToopsProcessLink *p) { _append((ToopsObjectLink*)p); }
    void inserthere(ToopsProcessLink &r) { inserthere(&r); }
    void inserthere(ToopsProcessLink *p) { _inserthere((ToopsObjectLink*)p); }
    void appendhere(ToopsProcessLink &r) { appendhere(&r); }
    void appendhere(ToopsProcessLink *p) { _appendhere((ToopsObjectLink*)p); }
    ToopsProcess *current(void) { return (ToopsProcess*) _current(); }
    ToopsProcess *first(void) { return (ToopsProcess*) _first(); }
    ToopsProcess *last(void) { return (ToopsProcess*) _last(); }
    ToopsProcess *currentLink(void) { return (ToopsProcess*) _currentLink(); }
    ToopsProcess *firstLink(void) { return (ToopsProcess*) _firstLink(); }
    ToopsProcess *lastLink(void) { return (ToopsProcess*) _lastLink(); }

    int next(ToopsProcess*& rp) 
    {
        if (_setnext())
            { rp = (ToopsProcess*) _current(); return 1; }
        else
            { rp = 0; return 0; }
    }
    int prev(ToopsProcess*& rp)
    {
        if (_setprev())
            { rp = (ToopsProcess*) _current(); return 1; }
        else
            { rp = 0; return 0; }
    }
    int getnext(ToopsProcess*& rp)
    {
        ToopsProcess * pptmp;
        if ((pptmp = (ToopsProcess*) _getnext()) != 0)
            { rp = pptmp; return 1; }
        else
            { rp = 0; return 0; }
    }
    int getprev(ToopsProcess*& rp)
    {
        ToopsProcess* pptmp;
        if ((pptmp = (ToopsProcess*) _getprev()) != 0)
            { rp = pptmp; return 1; }
        else
            { rp = 0; return 0; }
    }
    int next(ToopsProcessLink*& rpl)
    {
        if (_setnext())
            { rpl = (ToopsProcessLink*) _currentLink(); return 1; }
        else
            { rpl = 0; return 0; }
    }
    int prev(ToopsProcessLink*& rpl)
    {
        if (_setprev())
            { rpl = (ToopsProcessLink*) _currentLink(); return 1; }
        else
            { rpl = 0; return 0; }
    }
    int getnext(ToopsProcessLink*& rpl)
    {
        ToopsProcessLink * pptmp;
        if ((pptmp = (ToopsProcessLink*) _getnextlink()) != 0)
            { rpl = pptmp; return 1; }
        else
            { rpl = 0; return 0; }
    }
    int getprev(ToopsProcessLink*& rpl)
    {
        ToopsProcessLink* pptmp;
        if ((pptmp = (ToopsProcessLink*) _getprevlink()) != 0)
            { rpl = pptmp; return 1; }
        else
            { rpl = 0; return 0; }
    }
    
    void remove(ToopsProcessLink* t) { _remove((ToopsObjectLink*)t); }
    ToopsProcess* search(const char *name)
        { return (ToopsProcess*)_search(name); }
    ToopsProcess** touchAll(void)
        { return (ToopsProcess**) _touchAll(); }
    void write(int depth = 0, int mode = 0) const;
    
    DECLARE_CLASSINFO(ToopsProcessList); // pt 9.94
    
private:
    // prevent compiler from using copy constructor and assignment
    //ToopsProcessList(const ToopsProcessList&); //mjk 280895
    void operator=(const ToopsProcessList&);
};


inline ToopsProcess* ToopsProcess::search(const char *n)
    { return allProcesses()->search(n); }
inline ToopsProcess** ToopsProcess::systemProcesses(void)
    { return allProcesses()->touchAll(); }
inline void ToopsProcessLink::unLink(void)
    { ((ToopsProcessList*) linkedTo())->remove(this); }

#endif // if !defined(TPROCESS_H)

