#ifndef ServiceElements_H
#define ServiceElements_H
/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
file name       :   sdu.h
class name      :   SDU (base class), DataReq, DataInd
author          :   Andreas Rinkel

documentation   :   Service Primitives for Service Data Units. SDU is
					  the (abstract) base class, DataReq and DataInd are
					  the service primitives for sending and receiving 
					  data. 

version         :   1.0
history         :   25.04.96  Manfred Kraess
                    clean-up and modification
--------------------------------------------------------------------------*/

// C++ standards

// TOOPS library
#include <message.h>
#include <str.h>

// Project header


//------------------------------------------------------------------------
// Type definition for type of SDU
typedef enum ServiceElements { data_req = 1, data_ind = 2 };


//------------------------------------------------------------------------
// SDU is a base class for Service Data Units: the service primitive
// that is passed to / received from the upper layers (from the protocol
// entity's point of view) through the entity's Service Access Point SAP. 
class SDU: public ToopsMessage
{
public:
	// return the type of the SDU
	ServiceElements sdu() 
		{ return _type; }
           
protected:
	// type of the SDU 
	ServiceElements _type;
};

//------------------------------------------------------------------------
// DataReq (Data Request service primitive) is received from the upper
// layer (from the protocol entity's point of view) through the SAP .
class DataReq: public SDU
{
public:
	// Default constructor
    DataReq (): SDU()
    {  
        _type = data_req; 
        _text  = "";
    }

	// 2nd constructor
    DataReq ( String str, int time ): SDU()
    {
        _type      = data_req;   // enum value
        _text      = str;
        _timeStamp = time;
    }

	// duplicate function as required by TOOPS
    virtual ToopsMessage *duplicate()
    { 
        return new DataReq( _text, _timeStamp ); 
    }

    // get content of the message
    // Important: The return value of getText() is only a pointer to
    //            the internal String.              
    const char* getText() { return _text; }
    int         getTimeStamp() { return _timeStamp; }
    
private:
    // SDU's contents
    String _text;
    int    _timeStamp;
};



//------------------------------------------------------------------------
// DataInd (Data Indication service primitive) is passed to the upper
// layer through the SAP.
class DataInd: public SDU
{
public:
	// Constructor
    DataInd ( String st, int time ): SDU()
    {
        _text      = st;
        _timeStamp = time;
        _type      = data_req; 
    }

	// duplicate function as required by TOOPS
    virtual ToopsMessage* duplicate ()
        { return new DataInd( _text, _timeStamp); }

    // get content of the message
    // Important: The return value of getText() is only a pointer to
    //            the internal String.              
    const char* getText() { return _text; }
    int         getTimeStamp() { return _timeStamp; }

private:
    // SDU's contents
    String _text;
    int    _timeStamp;   
};


#endif     


