/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
filename:       simpprot.cpp

documentation:  This program simulates a simple protocol.

                * Two service users "source" and "sink" use the service
                provided by the protocol at the service access points 
                sap_A and sap_B.
                
                * Two entities entity_A  and entity_B provide this
                service (implementation of a simple protocol).
                
                * This example assumes that all entities work in parallel.
                Therefore, each entity (service users "source" and "sink",
                protocol entities "entity_A" and "entity_B") runs as a
                process (classes Source, Sink and Entity derived from 
                ToopsProcess) on a dedicated processor (class ToopsProcessor).
                
                * The communication between the protocol entities and the
                service users is assumed to be synchronous. Consequently,
                the service access points (SAPs) are modelled as objects of
                class ToopsChannel. The service users and the protocol
                entities are connected to the SAPs through synchronous sockets
                (class Synchron_Socket, derived from ToopsSocket).
                
                * "line" is an object of class ToopsChannel that connects the
                two protocol entities entity_A and entity_B (through objects
                of class ToopsSocket).
                
                See the header files of each class for further details.
                
                For a simulation using a little more sophisticated protocol,
                see the sample program "abprot" that simulates the 
                "Alternating Bit Protocol".

author:         Andreas Rinkel
version      :  1.0
history      :  23.04.96  Manfred Kraess
                clean-up and modification
--------------------------------------------------------------------------*/

// C++ standards
#include <iostream.h>

// TOOPS library
#include <toops.h>
#include <socket.h>
#include <channel.h>


// Project header
#include "source.h"
#include "sink.h"
#include "entity.h"

int main()
{ 
    simtime duration = 20;

    // create model processors 
    ToopsProcessor source_CPU("source_cpu");
    ToopsProcessor sink_CPU ("sink_cpu");
    ToopsProcessor entity_A_CPU("entity_A_cpu");
    ToopsProcessor entity_B_CPU("entity_B_cpu");

    // create channels
    ToopsChannel sap_A ("SAP_A");
    ToopsChannel sap_B ("SAP_B");
    ToopsChannel line  ("line");

    // create processes that model the protocol entities
    Source source (&source_CPU, &sap_A, "source");
    Sink   sink   (&sink_CPU,  &sap_B, "sink");
    Entity entity_A (&entity_A_CPU, &sap_A, &line, "entity_A");
    Entity entity_B (&entity_B_CPU, &sap_B, &line, "entity_B");

    // Definition of a variable for the return value of the simulation
    ToopsSimulation::stopReason reason;

    // Start of the simulation
    reason = ToopsSimulation::start(duration);

    // Output reporting the successful completion of the simulation.
    cout << "\n Simulation simpprot completed. Return value is: ";
    cout <<  (int)reason << "\n";
    
    // Wait for user input
    // (may cause a comiler warning: "variable is set but never used")
    cout << "\n -> Press <return> to continue: ";
    char in;
    in = cin.get ();              
        
    return reason;
}
