#include "tprocess.h"
#include "procunit.h"
#include "to_error.h"
#include "timestep.h"
#include "strstrea.h"
#include <string.h>
#include <assert.h>

#if defined(NDEBUG) && defined(TPROCESSOR_LISTS_DEBUG)
    #error procunit: List debugging not available with NDEBUG defined
#endif

//-----------------------------------------------------------------------------
// ToopsProcessorLink (declared in procunit.h)
// =======================================

IMPLEMENT_CLASSINFO( ToopsProcessorLink, ToopsNamedObjectLink);

void ToopsProcessorLink::write(int depth, int mode) const
{
    ToopsNamedObjectLink::write(depth,mode);
}

//-----------------------------------------------------------------------------
// ToopsProcessorList (declared in procunit.h)
// =======================================
IMPLEMENT_CLASSINFO( ToopsProcessorList, ToopsNamedObjectList);

void ToopsProcessorList::write(int depth, int mode) const
{
	 ToopsNamedObjectList::write(depth,mode);
}


//-----------------------------------------------------------------------------
// ToopsProcessor (declared in procunit.h)
// ===================================
IMPLEMENT_CLASSINFO( ToopsProcessor, ToopsNamedObject);

// static members:
ToopsError ToopsProcessor::wNoProcess(TE_TPROCESSOR_NOPROCESS,
			ToopsError::warning,
			ToopsProcessor::ThisClassInfo()->name(),
         "no ToopsProcess defined, status set to inactive.");
// pt 9.94 'ToopsProcessor::' in Zeile 44 vor 'ThisClassInfo() ergaenzt

ToopsError ToopsProcessor::fDestroy (TE_TPROCESSOR_DESTROY,
                             ToopsError::fatal,
                             ToopsProcessor::ThisClassInfo()->name(),
                             "ToopsProcesses left.");
// pt 9.94 'ToopsProcessor::' in Zeile 44 vor 'ThisClassInfo() ergaenzt


ToopsProcessor::ToopsProcessor(const char *n, simtime sw, responsibility r) :
    ToopsNamedObject(n, r), sysLink(this), toActivateLink(this)
{
    DBG_INIT();
    DBG(ThisClassInfo()->name() << "::" << ThisClassInfo()->name() <<
        "(" << n << ", " << sw << ")\n\n")
    DBG_WRITE(cstr);

    if (eName.getSeverity() != ToopsError::ignore )
    {
        DBG_OUT(" checking name (ToopsError, if empty or not unique).\n", cstr)
        if (!processors()->isNameValid(name()))
            eName.err(name());
    }


    p_timeToContextSwitch = sw;
    runningProcess = 0;
    nextActivatedProcessLink = 0;
    p_state = active;
    p_timeOfContextSwitch = (simtime)0;

    // insert the ToopsProcessor into the list of all ToopsProcessors
    processors()->append(&sysLink);
}


ToopsProcessor::~ToopsProcessor()
{
    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::~" << ThisClassInfo()->name() <<
        "()  " << "name: " << name() << ", removed from all lists\n\n")
    DBG_WRITE(dstr)

    if (processes() > 0 && ToopsSimulation::state() == ToopsSimulation::running)
        fDestroy.err(name());

    // Remove the ToopsProcessor from the system list.
    processors()->remove(&sysLink);

    // Remove the ToopsProcessor from the 'to activate' list, if necessary.
    if (toActivateLink.isLinked())
        processorsToActivate()->remove(&toActivateLink);
}


// This function must choose the first ToopsProcess to be started.
// Called by ToopsSimulation::callAtStart(), when the simulation starts.
void ToopsProcessor::atStart(void)
{

    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::atStart(), name: " << name()
        << endl)
    DBG_WRITE(onAtStart)

	 // All ToopsProcesses for this ToopsProcessor are inserted in readyProcesses
	 // when the simulation starts (done by ToopsProcessor::remember())
	 if (readyProcesses.len())
	 {
		  DBG_INIT()
		  DBG( " ToopsProcess " << readyProcesses.first()->name() <<
				 " ,priority " << readyProcesses.first()->priority()  <<
				 "; selected as first running ToopsProcess.\n" )
		  DBG_WRITE(onAtStart)

		  // Get the first ToopsProcess on the list
		  // (sorted -> TPRocessor::remember)
		  readyProcesses.reset();
		  readyProcesses.getnext(nextActivatedProcessLink);
		  runningProcess = nextActivatedProcessLink->object();

        // Insert this ToopsProcess' link in the event list.
		  DBG_OUT(" timeSteps()->sortIn(); ", onStart)
		  timeSteps()->sortIn(nextActivatedProcessLink);
        nextActivatedProcessLink = 0;

        // And tell ToopsSimulation::getNextEvent not to forget us
        DBG_OUT("activating ToopsProcessor\n", onStart)
        processorsToActivate()->append(&toActivateLink);
    }
    else
    {
        DBG_OUT( " no ToopsProcesses defined, status inactive\n",onAtStart)
        p_state = inactive;
        wNoProcess.err(name());
    }
}

void ToopsProcessor::remember( ToopsProcessLink* changed)
{
	 // Each state transition of an owned ToopsProcess is reported by a call to
	 // this function. The job of this function is to reorder the processor's
	 // lists to enable ToopsProcessor::activate() to decide about the active
	 // process and about the next process to 'activated'.
	 // Terminated ToopsProcesses with the responsibility TOOPSdirect will be
	 // deleted immediatly.

	 DBG_INIT()
	 DBG(endl << ThisClassInfo()->name() << "::remember() for: " <<
        name() << endl <<
        (status() == active ? " is active" :
          " set to active, appended to processorsToActivate()") << endl )
    assert(this != 0);
    assert(changed != 0);


    if (status() == inactive)
    {
        // Remember ToopsSimulation::getNextEvent() to call us after all events
        // of this point of time() have been processed, because then possibly
        // another ToopsProcess must become the active one.
        processorsToActivate()->append(&toActivateLink);
        p_state = active;
    }


    ToopsProcess* changedProcess = changed->object();
    uint prio = changedProcess->priority();
	 simtime dtime = changedProcess->desiredTime();  //!!
	 DBG( " ToopsProcessLink is" << (changed->isLinked() ? " " : " not ") <<
			"linked; " <<
			( nextActivatedProcessLink == changed ?
			 "nextActivatedProcessLink set to 0" : "")  <<
			 "\n changed ToopsProcess data:\n")
	 DBG_WRITE(onRemember)
	 _DBG(changedProcess->ToopsProcess::write(0,1), onRemember)

	 ToopsProcess* inList;

	 // First remove changed, if it is linked to one of the internal lists
	 if (changed->isLinked())
		  changed->unLink();
	 // make nextActivatedProcessLink invalid, if necessary
	 if (changed == nextActivatedProcessLink)
		  nextActivatedProcessLink = 0;

	 // Reorder the lists
	 DBG_OUT("ToopsProcessor::remember() is reordering lists\n", onRemember)
	 switch (changedProcess->status())
    {
        // status changed to ready, so sort changed into readyProcesses
		case ToopsProcess::ready:
            // readyProcesses is sorted by priority, if two or more processes
            // have an equal priority, the longer waiting process has to be
            // _before_ the newly inserted. (? round robin ?)
            readyProcesses.reset();
            while (readyProcesses.next(inList) &&
						 (inList->priority() <= prio) );

            if (inList)
                readyProcesses.inserthere(changed);
            else
                readyProcesses.append(changed);
            DBG_OUT(" ready ToopsProcess sorted into list readyProcesses\n",
                    onRemember)

            #if defined(TPROCESSOR_LISTS_DEBUG)
                _DBG(readyProcesses.write(10,0), onRemember)
            #endif
            break;

        // status changed to waiting
        case ToopsProcess::waiting:
            if (changedProcess->isSetWaiting())
            {
                // insert process into timedProcesses
                timedProcesses.reset();
                while (timedProcesses.next(inList))
                {
						  if (inList->desiredTime() >= dtime)
						  {
								if (inList->desiredTime() > dtime)
                            break;
                        else if (inList->priority() > prio)
									 break;
                    }
                }
					 if (inList)
                    timedProcesses.inserthere(changed);
                else
                    timedProcesses.append(changed);

                DBG_OUT(" timed waiting ToopsProcess sorted "
                        "into list timedProcesses\n", onRemember)
                #if defined(TPROCESSOR_LISTS_DEBUG)
                    _DBG(timedProcesses.write(10,0), onRemember)
                #endif

            }
            else
            {
                // insert it to unTimedProcesses
                unTimedProcesses.insert(changed);
                DBG_OUT(" untimed waiting ToopsProcess inserted "
                        "in list unTimedProcesses\n", onRemember)
                #if defined(TPROCESSOR_LISTS_DEBUG)
                    _DBG(unTimedProcesses.write(10,0), onRemember)
                #endif
            }
            break;

		  // status changed to terminated
        case ToopsProcess::terminated:
			if (((ToopsSimulation*)changedProcess)->responsible() !=
                  ToopsSimulation::TOOPSdirect)
            {
                unTimedProcesses.append(changed);
                DBG_OUT(" terminated ToopsProcess inserted "
                        "in list unTimedProcesses\n", onRemember)
                #if defined(TPROCESSOR_LISTS_DEBUG)
                    _DBG(unTimedProcesses.write(10,0), onRemember)
                #endif
            }
            else
            {
                DBG_OUT(" deleting terminated ToopsProcess, reason: "
                        "responsible == TOOPSdirect\n", onRemember)
                delete changedProcess;
            }
            break;

        case ToopsProcess::running:
            runningProcess = changedProcess;
            readyProcesses.insert(changed);

            DBG_OUT(" runningProcess set to changedProcess\n",onRemember)
				DBG_OUT(" running ToopsProcess inserted "
                    " as top of list readyProcesses\n", onRemember)
            #if defined(TPROCESSOR_LISTS_DEBUG)
                _DBG(readyProcesses.write(3,0), onRemember)
            #endif
            break;

		case ToopsProcess::destroy:

            if(changed->isLinked())
                changed->unLink();
            DBG_OUT(" destroyed ToopsProcess unLinked\n",onRemember)
            break;
    }

    // Care for runningProcess, if it terminated or is going to wait for
    // something, it can't be the running process.

    if ( runningProcess && 
         ((runningProcess->status() == ToopsProcess::waiting) ||
          (runningProcess->status() == ToopsProcess::terminated)) )
    {
        runningProcess = 0;
        DBG_OUT(" runningProcess->status != running/ready ==> set to 0\n",
                onRemember);
	 }
	 DBG_OUT("ToopsProcessor::remember() end\n", onRemember)
}

void ToopsProcessor::activate(void)
{
    // jobs:
    // - decide, which process is / becomes the running process
    //   (ToopsProcess* runningProcess, link remains in readyProcesses)
    //   and manage context change timing.
    // - decide, which process is the one with the next event
    //   (ToopsProcessLink* nextActiveProcessLink, this link is removed
    //   from the processor's lists and sorted into timeSteps() ).
    // note: this function can be called for several times at the same
    //   point of simulation time. Possible reasons:
    //   + consumeTime(0) or setWaiting(0) was called
    //   + activate() switched to another process and timeForContextSwitch
    //     is 0
    //   + another process sends a message (with transport time 0), that wakes
    //     up another process on this processor. (If that process has a higher
    //     priority than the actual running process, a context change is
    //     required.)
    // note: ToopsProcessor::remember() has been at least called once before
    //       ToopsProcessor::activate() is called (at the same sim time).


    // Determine the active process:
    // - note: a context change cannot be interrupted, so if a context change
    //   is taking place, no decision about the active process is needed.
    // - ToopsProcess' state transitions  more!!
    // - a context change is necessary, if:
    //   a) no runningProcess exists, but there is at least one ready process or
    //   b) a ready process has a higher priority than the runningProcess (if
    //      both have the same priority, the runningProcess remains running.)
    // check, whether a context change is going on:
    // - p_timeOfContextSwitch is always set to time(), when a context change
    //   has started (and if p_timeToContextSwitch is > 0).
    // - so if p_timeOfContextSwitch is > 0 and p_timeOfContextSwitch +
    //   p_timeToContextSwitch > time(), a contextSwitch is still going on.
    // - if another ToopsProcess with a higher priority is becoming ready at
    //   the same point of time the context change started, it can
    //   become the running ToopsProcess.
    DBG_INIT()
    DBG(endl << ThisClassInfo()->name() << "::activate() for: "
        << name() << endl
        << " runningProcess is "
        << (runningProcess ? runningProcess->name() : "0") << endl
        << " nextActivatedProcessLink belongs to "
        << (nextActivatedProcessLink ?
            nextActivatedProcessLink->object()->name() : "0") << endl
        << " context change: active at " << p_timeOfContextSwitch
        << ", time to change " << p_timeToContextSwitch << ", time() "
        << time() << "\n")

    // determine whether we may decide about the running ToopsProcess:
    int isInContextSwitch = (p_timeOfContextSwitch &&
        (p_timeOfContextSwitch + p_timeToContextSwitch  > time()) );
    int ContextSwitchCanBeInterrupted = (p_timeOfContextSwitch == time());

    DBG( " ToopsProcessor is " << (isInContextSwitch ? " " : "not " ) <<
         "in a context change" )
    DBG( ( (isInContextSwitch && ContextSwitchCanBeInterrupted) ?
          ", that can be interrupted.\n" : ".\n") )
    DBG_WRITE(onActivate)

    if ( (p_timeToContextSwitch == 0) || (!isInContextSwitch) ||
         (isInContextSwitch && ContextSwitchCanBeInterrupted) )
        // we're not in a context change or the context change started at
        // the same point of time.
    {
        DBG_OUT("determining the active ToopsProcess:\n",
                onActivate)

        if(runningProcess && readyProcesses.len() &&
           readyProcesses.first()->priority() < runningProcess->priority() )
        {
            DBG_OUT( " context change needed: a ready ToopsProcess has higher\n"
                     "priority, runningProcess->cancelRunning()\n", onActivate)

            // condition b) for a context change
            runningProcess->cancelRunning();
            runningProcess = readyProcesses.first();
            runningProcess->setRunning(time()+timeToContextSwitch());
            if (p_timeToContextSwitch)
                p_timeOfContextSwitch = time();
        }
        else if ((!runningProcess) && readyProcesses.len())
        {

            DBG_OUT( " context change needed:\n"
                     " no running ToopsProcess, but a ready ToopsProcess exists\n",
                     onActivate)

            // condition a) for a context change
            runningProcess = readyProcesses.first();
            runningProcess->setRunning(time()+timeToContextSwitch());
            if (p_timeToContextSwitch)
                p_timeOfContextSwitch = time();
        }
        else
        {
            ;
            DBG_OUT( " no context change took place\n",
                     onActivate)
        }

        DBG_INIT()
        DBG(" p_timeOfContextSwitch set to " << p_timeOfContextSwitch << endl)
        DBG_WRITE(onActivate)

    }
    else
    {
        // we are in a context change, that cannot be interrupted
        ; // do nothing, the brackets are for debugging
        DBG_OUT( "no decision about the running process needed\n", onActivate)
    }

    // Decide, which process is the next event (nextActivatedProcessLink).
    // Take the process with the smallest desiredTime() of these processes:
    // - runningProcess
    // - timedProcesses.first()
    // note: both might be 0

    assert(nextActivatedProcessLink == 0 /* */);

    if (!runningProcess && !timedProcesses.first())
    {
        nextActivatedProcessLink = 0;
    }
    else if ( runningProcess && (!timedProcesses.len() ||
              (timedProcesses.len() &&
               (runningProcess->desiredTime() <= timedProcesses.first()->desiredTime() ) ) ) )
    {
        readyProcesses.reset();
        readyProcesses.getnext(nextActivatedProcessLink);
    }
    else
    {
        timedProcesses.reset();
        timedProcesses.getnext(nextActivatedProcessLink);
    }

    // Make the selected ToopsProcess an event
    if(nextActivatedProcessLink)
        timeSteps()->sortIn(nextActivatedProcessLink);

    // Indicate, that our job is done so far
    p_state = inactive;

    _DBG(ToopsProcessor::write(3,0), onActivate)

}

// Return a List of all ToopsProcesses on this ToopsProcessor.
ToopsProcess** ToopsProcessor::myProcesses(void)
{
    ToopsProcess** all = new ToopsProcess*[processes()];
    if (all == 0)
        fOutOfMem.err("", "ToopsProcessor::myProcesses()");

    instance_counter i=0;
	ToopsProcess* p;  //"register" removed mjk 310895
    readyProcesses.reset();
    timedProcesses.reset();
    unTimedProcesses.reset();

    while (readyProcesses.next(p))
        all[i++] = p;
    while (timedProcesses.next(p))
        all[i++] = p;
    while (unTimedProcesses.next(p))
        all[i++] = p;
    return all;
}

void ToopsProcessor::write(int depth, int mode) const
{
    if (depth-1 >= 0)
        ToopsNamedObject::write(depth-1, mode);

    ostrstream o;

    if (depth==0 && !mode)
        o << DBGends << endl;

    o << ThisClassInfo()->name() << ":";

    o << "\n timeToContextSwitch(): " << timeToContextSwitch()
      << ", p_timeOfContextSwitch: " << p_timeOfContextSwitch
      << ", status(): " << (int)status() << endl; 
    // status casted to int to avoid a bug in Microsoft Visual C 7 Apr 95 mjk  

    o << " nextActivatedProcessLink: ";
    if (nextActivatedProcessLink)
    {
        o << nextActivatedProcessLink->object()->name()
          << " at " << nextActivatedProcessLink->object()->desiredTime();
    }
    else
        o << "-";

    o << "\n  runningProcess: ";
    if (runningProcess)
    {
        o << runningProcess->name();
    }
    else
        o << "-";

    o << "\n ready processes: " << readyProcesses.len() << ", timed Processes: "
      << timedProcesses.len() << ", untimedProcesses: "
      << unTimedProcesses.len() << endl << ends;

    char *out = o.str();
    ToopsObject::DBGout(out);
    delete out;

    if (!mode)
    {
        DBGout("\n lists:\n\n readyProcesses:\n");
        readyProcesses.write();
        DBGout("\n timedProcesses:\n");
        timedProcesses.write();
        DBGout("\n unTimedProcesses:\n");
        unTimedProcesses.write();
    }
}

