#include "to_list.h"
#include "strstrea.h"
#include <assert.h>

//-----------------------------------------------------------------------------
// ToopsObjectLink (declared in to_list.h)
// ===================================


IMPLEMENT_CLASSINFO( ToopsObjectLink, ToopsObject)

// Remove the link from the list and leave the list linked.
void ToopsObjectLink::unlink(void)
{
    assert(this !=0);
    
    if (prev)
        prev->next = next;
    if (next)
        next->prev = prev;

    next = prev = 0;
    list = 0;
}

// Insert a new link bfore this link.
void ToopsObjectLink::insert( ToopsObjectLink *newlink, 
    ToopsObjectList* insList)
{
    assert(this !=0);
    assert(newlink != 0);
    assert(newlink->list == 0);
    assert(insList != 0);

    newlink->list = insList;

    if (prev)
        prev->next = newlink;
    newlink->next = this;
    newlink->prev = prev;
    prev = newlink;
}

// Append a new link after this link.
void ToopsObjectLink::append(ToopsObjectLink *newlink, ToopsObjectList *appList)
{
    assert(this !=0);
    assert(newlink != 0);
    assert(newlink->list == 0);
    assert(appList != 0);

    newlink->list = appList;

    if (next)
        next->prev = newlink;
    newlink->next = next;
    newlink->prev = this;
    next = newlink;
}

void ToopsObjectLink::write(int, int mode) const
{
    // ignore depth, because ToopsObject has no nonstatic status information
    ostrstream o;
    if (mode)
        o << DBGends << endl;

    o << ThisClassInfo()->name() << ":  "
      << "next: " << next << ", prev: " << prev
      << ", list: " << list << ", object: " << who << endl << ends;
    char *c = o.str();
    DBGout(c);
    delete c;
}

//-----------------------------------------------------------------------------
// ToopsObjectList (declared in to_list.h)
// ===================================

IMPLEMENT_CLASSINFO( ToopsObjectList, ToopsObject)

// Insert a new link at the list's top.
void ToopsObjectList::_insert( ToopsObjectLink *newlink)
{
    assert(this !=0);
    assert(newlink != 0);
    assert(!(newlink->isLinked()));
    elements++;
    // are there any entries in the list ?
    if (last)
        // the list is a circle, so last->next is the first element
        // in the list and a call to last->next->insert() will insert the
        // newlink as first element in the list.
        last->next->insert(newlink, this);
    else
    {
        // no entries, create the first
        last = newlink;
        // and set up the circle
        last->next = last->prev = last;
        last->list = this;
    }
}

// Append a new link at the list's end.
void ToopsObjectList::_append( ToopsObjectLink *newlink)
{
    assert(this !=0);
    assert(newlink != 0);
    assert(!(newlink->isLinked()));
    // This place shows some of the elegance of Hansen's (->simlist.h)
    // concept of a doubly linked list. Designing it as a circulary
    // linked list with only a "last" pointer as "anchor" the action of
    // appending to the list is the same as to insert, only the last pointer
    // has to be corrected afterwards.
    _insert(newlink);   // increases the element counter
    last = last->next;

}

// Insert a new link before the current link.
void ToopsObjectList::_inserthere( ToopsObjectLink *newlink)
{
    assert(this !=0);
    assert(newlink != 0);
    assert(!(newlink->isLinked()));
    if (current)
    {
        elements++;
        current->insert(newlink, this);
    }
    else
        _insert(newlink);   // increases elements
}

// Append a new link after the current link.
void ToopsObjectList::_appendhere( ToopsObjectLink *newlink)
{
    assert(this !=0);
    assert(newlink != 0);
    assert(!(newlink->isLinked()));
    if (current)
    {
        elements++;
        current->append(newlink, this);
        if (current == last)
            last = newlink;
    }
    else
       _append(newlink);   // increases elements
}

// Set current to the next link.
int ToopsObjectList::_setnext(void)
{
    assert(this !=0);
    // Check if the list is empty
    if(!last)
        return 0;  // no way to a next element !!

    // Check, if current was set
    if(current)
        // if it points to the last element, clear current and note
        // it to the user
        if(current == last)
        {
            current = 0;
            return 0;
        }
        else
            current = current->next;
    else
        // current not set, so set it to the first element
        current = last->next;

    return 1;
}

// Set current to the previous link.
int ToopsObjectList::_setprev(void)
{
    // First check for an empty list
    if (!last)
        return 0;

    // Move current backward.
    if (current)
        // Test, if we reached the list's top
        if (current == last->next)
        {
            // we are, so indicate failure
            current = 0;
            return 0;
        }
        else
           current = current->prev;
    else
       current = last;

    return 1;
}

// Remove the next link from the list and return the object it points to.
ENTRY ToopsObjectList::_getnext()
{
    assert(this !=0);
    if (!last)
        return 0;

    ToopsObjectLink* i;
    if (current)
        if (current == last)
        {
            current = 0;
            return 0;
        }
        else
            i = current->next;
    else
        // get the first element of the list !!!
        i = last->next;

    // security checks:
    // 1. is this element the least element of the list ?
    // 2. is this element equal to last ?

    // 1.
    if (len() == 1)
        last = current = 0;
    // 2.
    else if (i == last)
        last = last->prev;

    i->unlink();
    elements--;
    return i->who;
}


// Remove the previous link from the list and return the object it points to.
ENTRY ToopsObjectList::_getprev()
{
    assert(this !=0);
    if (!last)
        return 0;

    ToopsObjectLink* i;;
    if (current)
        if (current == last->next)
        {
            current = 0;
            return 0;
        }
        else
            i = current->prev;
    else
        // get the _last_ element of the list
        i = last;

    // security checks:
    // 1. is this element the least one in the list ?
    // 2. is the element to be removed equal to last ?

    // 1.
    if (len() == 1 )
        last = current = 0;
    // 2.
    else if(i == last)
        last = last->prev;

    i->unlink();

    elements--;
    return i->who;
}


// Remove the next link from the list and return the link.
ToopsObjectLink * ToopsObjectList::_getnextlink()
{
    assert(this !=0);
    if (!last)
        return 0;

    ToopsObjectLink* i;
    if (current)
        if (current == last)
        {
            current = 0;
            return 0;
        }
        else
            i = current->next;
    else
        // get the first element of the list !!!
        i = last->next;

    // security checks:
    // 1. is this element the least element of the list ?
    // 2. is this element equal to last ?

    // 1.
    if (len() == 1)
        last = current = 0;
    // 2.
    else if (i == last)
        last = last->prev;

    i->unlink();
    elements--;
    return i;
}


// Remove the previous link from the list and return the link.
ToopsObjectLink * ToopsObjectList::_getprevlink()
{
    assert(this !=0);
    if (!last)
        return 0;

    ToopsObjectLink* i;;
    if (current)
        if (current == last->next)
        {
            current = 0;
            return 0;
        }
        else
            i = current->prev;
    else
        // get the _last_ element of the list
        i = last;

    // security checks:
    // 1. is this element the least one in the list ?
    // 2. is the element to be removed equal to last ?

    // 1.
    if (len() == 1 )
        last = current = 0;
    // 2.
    else if(i == last)
        last = last->prev;

    i->unlink();

    elements--;
    return i;
}


// Remove a given link from the list. CAUTION: in the non debug version
// it will be not checked whether the link is in this list.
void ToopsObjectList::_remove( ToopsObjectLink* toRemove)
{
    assert(toRemove != 0);
    assert(toRemove->linkedTo() == this);
    // security checks:
    // 1. is this element the least one in the list ?
    // 2. is the element to be removed equal to last ?
    if (len() == 1)
        last = 0;
    else if(toRemove == last)
        last = last->prev;

    toRemove->unlink();
    elements-- ;
    reset();
}

// Make l the current link.
void ToopsObjectList::_set( ToopsObjectLink *l)
{
    assert(this !=0);
    assert(l->linkedTo() == this);
    current = l;
}

// Get the <num>th element oo the list (num starts with 0).
ToopsObject* ToopsObjectList::_get(instance_counter num)
{
    assert(this != 0);

    if (len() == 0 || len() <= num )
        return 0;

    instance_counter pos;   //"register" removed mjk 310895

    reset();
    for(pos = 0; pos < num; pos++)
        _setnext();

    return _getnext();

}

// Touch the <num>th element oo the list (num starts with 0).
ToopsObject* ToopsObjectList::_touch(instance_counter num)
{
    assert(this != 0);

    if (len() == 0 || len() <= num )
        return 0;

    instance_counter pos; //"register" removed mjk 310895

    reset();
    for(pos = 0; pos <= num  ; pos++)
        _setnext();

    return _current();
}

// Delete the <num>th element of the list (num starts with 0).
void ToopsObjectList::_del(instance_counter num)
{
    assert(this != 0);

    if (len() == 0 || len() <= num )
        return;

    instance_counter pos;  //"register" removed mjk 310895

    reset();
    for(pos = 0; pos < num  ; pos++)
        _setnext();

    delete _getnext();
}

// Return an array holding all objects in the list.
ToopsObject** ToopsObjectList::_touchAll(void)
{
    // necessary, because this function could (and can) be called for
    // a list that doesn't exist (system... functions)
    if (this == 0)
        return 0;

    // fails in Borland C++, if more than 64k allocated
    ToopsObject** all = new ToopsObject*[len()];
    if (all == 0)
        return 0;

    reset();
    for(instance_counter i = 0; i < len(); i++)
    {
        _setnext();
        all[i] = _current();
    }
    return all;
}


// Calls for each contained element the virtual function write(depth,1),
// outputs only, if list is empty. Intended to be called by write(...).
// Works also with the inherited list classes.
void ToopsObjectList::writeList(int depth) const
{
    ToopsObjectLink *t = last;
    if (t)
    {
        instance_counter i;
        for(i = 0; i < elements; i++)
        {
            t = t->next;
            (t->who)->write(depth,1);
        }
    }
    else
        DBGout("  -> list is empty\n");
}

void ToopsObjectList::write(int depth, int mode) const
{
    if (mode)
    {
        ostrstream o;
        o << ThisClassInfo()->name() <<":  elements: " << len() << endl
          << ends;
        char* c = o.str();
        DBGout(c);
        delete c;
    }
    else
    {
        writeList(depth);
    }
}

