
#include "setjump.h"
#include <stdlib.h>
#include <string.h>
#include "strstrea.h"
#include <stddef.h>
#include <assert.h>
#include "toops.h"
#include "to_error.h"
#include "timer.h"
#include "procunit.h"
#include "socket.h"
#include "tprocess.h"
#include "event.h"
#include "portable.h"


//-----------------------------------------------------------------------------
// ToopsProcessLink (declared in tprocess.h)
// =====================================

IMPLEMENT_CLASSINFO( ToopsProcessLink, ToopsNamedObjectLink)
void ToopsProcessLink::write(int depth, int mode) const
{
    ToopsNamedObjectLink::write(depth, mode);
}

//-----------------------------------------------------------------------------
// ToopsProcessList (declared in tprocess.h)
// =====================================

IMPLEMENT_CLASSINFO( ToopsProcessList, ToopsNamedObjectLink)

void ToopsProcessList::write(int depth, int mode) const
{
    // mode > 0 -> reduced info
    ostrstream o;
	 o << "\n " << ThisClassInfo()->name() << " debug information:  "
      << len() << " elements\n" << ends;
    char *out = o.str();
    ToopsObject::DBGout(out);
    delete out;
    if (mode)
    {
        if (len())
        {
            DBGout(" first: ");
            _first()->write(depth,1);
            if (len() > 1)
            {
               DBGout(" last: ");
               _last()->write(depth,1);
            }
        }
    }
    else
    {
        DBGout(" list contents:\n");
        ToopsObjectList::write(depth,0);
        DBGout("ToopsProcessList end debug information\n");
    }
}

//-----------------------------------------------------------------------------
// ToopsProcess (declared in tprocess.h)
// =================================

IMPLEMENT_CLASSINFO( ToopsProcess, ToopsNamedObject)

// void sayState(ostrstream &o, const ToopsProcess * const p); / pt 9.94
static void sayState(ostrstream &o, const ToopsProcess * const p);

// initialize the static members:
ToopsProcess* ToopsProcess::p_thisProcess = 0;

ToopsError ToopsProcess::fCTNoAccess(TE_TPROCESS_CT_NOACCESS,
                             ToopsError::fatal,
                             ToopsProcess::ThisClassInfo()->name(),
                             "consumeTime() called from other ToopsProcess");
// pt 9.94 Z.77 'ToopsProcess::' ergaenzt vor 'ThisClassInfo'

ToopsError ToopsProcess::fSetWaitingNoAccess(TE_TPROCESS_SETWAITING_NOACCESS,
                                     ToopsError::fatal,
                                     ToopsProcess::ThisClassInfo()->name(),
                                     "setWaiting() called from"
                                     " another ToopsProcess");
// pt 9.94 Z.83 'ToopsProcess::' ergaenzt vor 'ThisClassInfo'

ToopsError ToopsProcess::wNoTerminate(TE_TPROCESS_NOTERMINATE,
                              ToopsError::warning,
                              ToopsProcess::ThisClassInfo()->name(),
                              "ended without terminate().");
// pt 9.94 Z.90 'ToopsProcess::' ergaenzt vor 'ThisClassInfo'

ToopsError ToopsProcess::fDestroyNotTerminated(TE_TPROCESS_DESTROY,
                                       ToopsError::fatal,
                                       ToopsProcess::ThisClassInfo()->name(),
                                       "not terminated and destructor called"
                                       " in simulation" );
// pt 9.94 Z.96 'ToopsProcess::' ergaenzt vor 'ThisClassInfo'


ToopsError ToopsProcess::fPriorityToLow(TE_TPROCESS_PRIORITY_TO_LOW,
                                ToopsError::fatal,
                                ToopsProcess::ThisClassInfo()->name(),
                                "cannot create a ToopsProcess with a higher"
                                " priority");
// pt 9.94 Z.104 'ToopsProcess::' ergaenzt vor 'ThisClassInfo'

ToopsError ToopsProcess::fWrongOwner(TE_TPROCESS_WRONG_OWNER,
                             ToopsError::fatal,
                             ToopsProcess::ThisClassInfo()->name(),
                             "cannot create a ToopsProcess on another"
                             " ToopsProcessor");
// pt 9.94 Z.111 'ToopsProcess::' ergaenzt vor 'ThisClassInfo'

ToopsError ToopsProcess::fSyncNoAccess(TE_TPROCESS_SYNC_NOACCESS,
                               ToopsError::fatal,
                               ToopsProcess::ThisClassInfo()->name(),
                               "synchronize() called from another ToopsProcess");
// pt 9.94 Z.118 'ToopsProcess::' ergaenzt vor 'ThisClassInfo'

ToopsError ToopsProcess::fCannotCreateToopsProcess(TE_TPROCESS_CANNOT_CREATE,
                                       ToopsError::fatal,
                                       ToopsProcess::ThisClassInfo()->name(),
                             "a ToopsProcess can only be created from"
                             " ToopsProcess::behavior()");
// pt 9.94 Z.124 'ToopsProcess::' ergaenzt vor 'ThisClassInfo'


// Set up a ToopsProcess
ToopsProcess::ToopsProcess( ToopsProcessor &p, uint prio, 
    const char *n, responsibility r) :
    ToopsNamedObject(n, r), sysLink(this), processorLink(this)
{
    initProcess(&p, prio);
}


ToopsProcess::ToopsProcess( ToopsProcessor *p, uint prio, 
    const char *n, responsibility r) :
    ToopsNamedObject(n, r), sysLink(this), processorLink(this)
{
    initProcess(p, prio);
}


// Does most of the constructor's real work.
void ToopsProcess::initProcess( ToopsProcessor *p, uint prio)
{
    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::"<< ThisClassInfo()->name() << "("
        << p->name() << ", " << prio << ", " << name() << ")\n")
    DBG_WRITE(cstr)

    if (ToopsSimulation::state() == ToopsSimulation::running) 
    {
        if (!thisProcess())
            fCannotCreateToopsProcess.err();
        else if (thisProcess()->owner() != p)
            fWrongOwner.err(thisProcess()->name());
        else if (thisProcess()->priority() >= prio)
            fPriorityToLow.err(thisProcess()->name());
    }

    if (eName.getSeverity() != ToopsError::ignore )
    {
        DBG_OUT(" checking name (ToopsError, if empty or not unique).\n", cstr)
        if (!allProcesses()->isNameValid(name()))
            eName.err(name());
    }

    p_priority = prio;
    p_owner = p;
    p_desiredTime = p_restConsumeTime = p_setToRunning = 0;

    p_state = (stateFlags) (ready | notStarted);

    _DBG(ToopsProcess::write(0,1),cstr)

    // 'register' our ToopsProcess in the list of all processes
    DBG_OUT(" inserting sysLink in allProcesses()\n", cstr)
    allProcesses()->insert(&sysLink);

    DBG_OUT(" inserting processorLink in owner's processList\n", cstr)
    owner()->remember(&processorLink);

    DBG_OUT(DBGends, cstr)
}


ToopsProcess::~ToopsProcess()
{
    DBG_INIT()
    DBG("ToopsProcess::~ToopsProcess " << name() << " on ToopsProcessor ")
    DBG(owner()->name() << endl)
    DBG_WRITE(dstr)

    p_state = (stateFlags) destroy;
    atStateChange();
    // fatal, if simstate == running AND process' status != terminated
    // more !! (test, if process is complete)
    if (ToopsSimulation::state() == running && status()!=terminated)
        fDestroyNotTerminated.err(name());

    allProcesses()->remove(&sysLink);

    // Ok, delete all ToopsTimers and ToopsSockets for this ToopsProcess, if
    // responsibilty == TOOPSdirect, because this ToopsProcess is
    // deleted immediatly after termination.
    if (responsible() == TOOPSdirect)
    {   
        /* this code replaced due to a Microsoft C error 3 Apr 95 mjk
        ToopsSimulation *t;
        p_timers.reset();
        while(p_timers.getnext((ToopsTimer*)t))   
            delete t;
        p_sockets.reset();
        while(p_sockets.getnext((ToopsSocket*)t))
            delete t;
        */
        ToopsTimer *t1;
        ToopsSocket *t2;
        p_timers.reset();
        while(p_timers.getnext(t1))   
				delete (ToopsSimulation *)t1;
        p_sockets.reset();
        while(p_sockets.getnext(t2))
            delete (ToopsSimulation *)t2;
        // end of modification 3 Apr 95 mjk    
    }

    owner()->remember(&processorLink);
    DBG_OUT(DBGends, dstr)
}


int ToopsProcess::contextSave()
{
    // Copy the stack out, but first make sure we're above StackBase
	 // Probably obsolete (not checked), but left here for security.
	 if (mustrecurse())
        return contextSave();

	 // get an address that is surely above the current process'
    // top of stack
	 char *px = ToopsStackDir::plusone();
    // evaluate the difference and allocate memory for the stack
	 context.StackSize = ToopsStackDir::diff(px, context.StackBase);
	 // delete the old saved stack
	 if (context.SaveStack) {
		  delete[] context.SaveStack;
	 }
	 context.SaveStack = new char[context.StackSize];
	 if (context.SaveStack == 0)
		  fOutOfMem.err("","ToopsProcess: internal");

	 // copy the stack to SaveStack
	 if (ToopsStackDir::grows_up())
		  memcpy(context.SaveStack, context.StackBase, context.StackSize);
	 else
		  memcpy(context.SaveStack, px, context.StackSize);

	// prepare for longjmp()
	int ret = setjmp(context.JB);

	 return ret;
}


// Restore a process to be the current process.
// It doesn't return but instead it comes back through contextSave().
// It works by memcpy()'ing the process' copy of the stack into the real
// stack.
// First adjust the stack to make certain that we are above what
// needs to be restored. This is done by recursively invoking restore().

// CAUTION: no member function !!
static void restorestack(char *lowerbound, char *from,
	 ptrdiff_t size, jmp_buf jb)
{
	memcpy(lowerbound, from, size);

	longjmp(jb, 1);
	// NOTREACHED !!
}


void ToopsProcess::restore()
{
	 char x;
	 char *px = &x;

	 // If necessary, bump up the stack size.
	 ptrdiff_t diff = ToopsStackDir::diff (px, context.StackBase);
	 if (diff < context.StackSize)
		  restore();

	 // Restore the stack
	 char *lowerbound = ToopsStackDir::grows_up()  ?
				context.StackBase :
				context.StackBase - context.StackSize;

	 restorestack(lowerbound, context.SaveStack,
            context.StackSize, context.JB);
	 // NOTREACHED !!
}


void ToopsProcess::consumeTime(simtime s)
{
	 // check if call is legal
	 if (thisProcess() != this)
		  fCTNoAccess.err(thisProcess()->name());

	 DBG_INIT()
	 DBG( endl << ThisClassInfo()->name() << "::consumeTime(" << s
			<< ") for: " << name() << ", at " << time() << endl)
	 clrState(newMessage);
	 clrState(timerExpired);

	 // state transition 32 (if s > 0), else no transition
	 p_restConsumeTime = s;
	 p_desiredTime = time() + s;

	 DBG(" state transition 32; restConsumeTime: " << p_restConsumeTime
		  << ", desiredTime: " << p_desiredTime << "\n owner()->remember()\n")
	 DBG_WRITE(onConsumeTime)

	 owner()->remember(&processorLink);

	 if (timeOverFlow(p_desiredTime))
		  ToopsSimulation::_stop(timeOverFlowStop);

	 // reset p_thisProcess
	 p_thisProcess = 0;

	 atStateChange();

	 // save the context and call the next process
	 if (!contextSave())
		  nextEvent();
}


void ToopsProcess::setWaiting(simtime s)
{
	 // check if call is legal
	 if (thisProcess() != this)
			fSetWaitingNoAccess.err(thisProcess()->name());

    DBG_INIT()
    DBG( endl << ThisClassInfo()->name() << "::setWaiting(" << s
         << ") for: " << name() << ", at " << time() << endl)

    p_desiredTime = time() + s;

    // state transition 37
    modifyStatus(waiting);
    setState(setWait);
    atStateChange();

    clrState(newMessage);
    clrState(timerExpired);

    DBG(" state transition 37; status: waiting setWaiting"
        ", desiredTime: " << p_desiredTime << "\n  owner()->remember()\n")
    DBG_WRITE(onSetWaiting)

    owner()->remember(&processorLink);

    if (timeOverFlow(p_desiredTime))
        ToopsSimulation::_stop(timeOverFlowStop);

    // reset p_thisProcess
    p_thisProcess = 0;

    if (!contextSave())
        nextEvent();
}


void ToopsProcess::checkCaller( ToopsProcess::callError& error)
{
    error = callOK;

    // check for possible errors
    if (status() == terminated)
        error = callTerminated;

    if (thisProcess())
    {
        if (thisProcess()->owner() != owner())
            error = (callError) (error | callToOtherProcessor); // pt 9.94 
    //             ^ dieser Cast-Op. wurde hinzugefuegt

        // if another process calls this function, it must have a higher priority,
        // i.e. the callers priority must be < this->priority()
        if ( (thisProcess() != this) && (thisProcess()->priority() >= priority()) )
            error = (callError) (error | callHigher); // pt 9.94
    //              ^ dieser Cast-Op. wurde hinzugefuegt
    
    }
    else
    {
        error = (callError) (error | callUseless); // pt 9.94
    //              ^ dieser Cast-Op. wurde hinzugefuegt    
    }
}


void ToopsProcess::suspend(callError* pError)
{
    callError error;
    checkCaller(error);
    if(isSuspended())
        error = (callError) (error | callUseless); // pt 9.94
    //              ^ dieser Cast-Op. wurde hinzugefuegt    

    if (pError)
        *pError = error;

    if (error)
    {
        DBG_INIT()
        DBG(endl << ThisClassInfo()->name() <<
            "::suspend() for: " << name() << " ignored because:\n  "
            << ((error && callTerminated) ?
               "  - ToopsProcess is terminated, call useless\n" : "")
            << ((error && callHigher) ? "  - caller's priority to low \n" : "")
            << ((error && callToOtherProcessor) ?
               "  - caller is running on another ToopsProcessor\n" : "")
            << ((error && callUseless) ?
               "  - ToopsProcess is suspended, call useless\n" : "") )

        DBG_WRITE(onSuspend)

        return;
    }

    if (thisProcess() != this)
    {
        // called for another but the active process, call is o.k.
        // state transitions 25, 45, 68, 79
        DBG_INIT()
        DBG(endl << ThisClassInfo()->name() <<"::suspend() for: " << name() <<
                   "\n state transitions 25, 45, 68, 79\n")
        DBG_WRITE(onSuspend)

        modifyStatus(waiting);
        setState(suspended);

        atStateChange();

        _DBG(ToopsProcess::write(2,1), onSuspend)
        DBG_OUT(" remembering owner()\n\n",onSuspend)

        owner()->remember(&processorLink);
        return;
    }

    DBG_INIT()
    DBG(endl << ThisClassInfo()->name() <<"::suspend() for: " << name() <<
        "\n state transition 35\n")
    DBG_WRITE(onSuspend)

    // state transition 35
    modifyStatus(waiting);
    setState(suspended);

    atStateChange();

    _DBG(ToopsProcess::write(2,1), onSuspend)
    DBG_OUT(" remembering owner()\n\n",onSuspend)

    // always clear these 2 flags, if the caller is the active process
    clrState(newMessage);
    clrState(timerExpired);

    owner()->remember(&processorLink);

    // reset p_thisProcess
    p_thisProcess = 0;

    if (!contextSave())
        nextEvent();
}


void ToopsProcess::resume(callError* pError)
{

    callError error;
    checkCaller(error);
    if (!isSuspended())
        error = (callError) (error | callUseless); // pt 9.94
    //              ^ dieser Cast-Op. wurde hinzugefuegt

    if (pError)
        *pError = error;

    if (error)
    {
        DBG_INIT()
        DBG(endl << ThisClassInfo()->name() <<
            "::resume() for: " << name() << " ignored because:\n  "
            << ((error && callTerminated) ?
               "  - ToopsProcess is terminated, call useless\n" : "")
            << ((error && callHigher) ? "  - caller's priority to low \n" : "")
            << ((error && callToOtherProcessor) ?
               "  - caller is running on another ToopsProcessor\n" : "")
            << ((error && callUseless) ?
               "  - ToopsProcess is not suspended, call useless\n" : "") )
        DBG_WRITE(onResume)
        return;
    }

    DBG_INIT()
    DBG(endl << ThisClassInfo()->name() <<"::resume() for: " << name() <<
                   "\n state transitions 54, 86, 97\n")
    DBG_WRITE(onResume)

    // state transitions 54, 86, 97
    clrState(suspended);
    if (! (isInSync() || isSetWaiting()) )
        modifyStatus(ready);

    atStateChange();

    _DBG(ToopsProcess::write(2,1), onResume)
    DBG_OUT(" remembering owner()\n\n",onResume)

    owner()->remember(&processorLink);
}



void ToopsProcess::terminate(callError* pError)
{
    callError error;
    checkCaller(error);
    if (status() == terminated)
        error = (callError)(error | callUseless); // pt 9.94
    //              ^ dieser Cast-Op. wurde hinzugefuegt    

    if (pError)
        *pError = error;

    if (error)
    {
        DBG_INIT()
        DBG(endl << ThisClassInfo()->name() <<
            "::terminate() for: " << name() << " ignored because:\n  "
            << ((error && callTerminated) ?
               "  - ToopsProcess is terminated, call useless\n" : "")
            << ((error && callHigher) ? "  - caller's priority to low \n" : "")
            << ((error && callToOtherProcessor) ?
               "  - caller is running on another ToopsProcessor\n" : "") )
        DBG_WRITE(onTerminate)

        return;
    }

    DBG_INIT()
    DBG(endl << ThisClassInfo()->name() <<"::terminate() for: " << name() <<
        "\n state transition x1\n status: terminated, remembering owner()\n")
    DBG_WRITE(onTerminate)

    // state transitions 21, 31, 41, 51, 61, 71, 81, 91
    modifyStatus(terminated);
    atStateChange();
    p_desiredTime = 0;

    if (context.SaveStack)
        delete context.SaveStack;
    context.SaveStack = 0;

    owner()->remember(&processorLink);


    if (thisProcess() == this)  // terminated itself
    {
        // always clear these 2 flags, if the caller is the active process
        clrState(newMessage);
        clrState(timerExpired);
        DBG_OUT(" ToopsProcesses terminated itself, calling nextEvent()\n",
                 onTerminate)
        // reset p_thisProcess
        p_thisProcess = 0;


        nextEvent();
    }
}


void ToopsProcess::cancelRunning(void)
{

    DBG_INIT()
    DBG(endl << ThisClassInfo()->name() <<"::cancelRunning() for: " << name() <<
        "\n state transition 24\n")
    DBG_WRITE(onCancelRunning)

    // there is still time to consume, but another process becomes running
    // state transition 24
    simtime consumedTime = time() - p_setToRunning;
    p_restConsumeTime -= consumedTime;
    assert(p_restConsumeTime != 0);
    p_setToRunning = p_desiredTime = 0;
    modifyStatus(ready);

    atStateChange();

    _DBG(ToopsProcess::write(2,1), onCancelRunning)
    DBG_OUT(" remembering owner()\n\n",onCancelRunning)


    owner()->remember(&processorLink);
}

void ToopsProcess::setRunning(simtime when)
{
    // the process shall become running again. So state transitions 42 and 43
    // have to be prepared for (re)activation
    p_desiredTime = when;

    DBG_INIT()
    DBG(endl << ThisClassInfo()->name() <<"::setRunning() for: " << name() <<
        "\n preparing state transitions 43, 42; desiredTime:" <<
        desiredTime() << endl << endl)
    DBG_WRITE(onSetRunning)

}


void ToopsProcess::activate(void)
{
    int noStartNeeded = 1;

    DBG_INIT()
    DBG("\nToopsProcess::activate() for: " << name() << endl)
    DBG_WRITE(onActivate)

    switch (status())
    {
        case running:
       {    // pt 9.94
            assert (p_restConsumeTime + p_setToRunning == time() );
            // consumeTime() ends
            // state transition 23
            p_restConsumeTime = (simtime)0;
            p_setToRunning = time();
            noStartNeeded = 0;
            DBG_INIT()
            DBG(" state transition 23 (end of consumeTime)\n")
            DBG_WRITE(onActivate)

            //owner()->remember(&processorLink);
            // call to owner()->remember() obsolete, because there is no way
            // to leave behavior() without a call to a function that
            // calls owner()->remember() !!
            break;
       }    // pt 9.94 '{}' eingefuegt

        case ready:

            if (restConsumeTime())
            {
                // the process has to become running again, but there
                // is time to consume left
                // state transition 42
                modifyStatus(running);
                p_desiredTime = time() + restConsumeTime();
                p_setToRunning = time();
                DBG_INIT()
                DBG(" state transition 42 (ready->running, restConsumeTime"
                    " > 0)\n restConsumeTime: "<< restConsumeTime() <<
                    ", desiredTime: " << desiredTime() <<
                    ", p_setToRunning: " << p_setToRunning <<
                    "\n owner()->remember()")
                DBG_WRITE(onActivate)

                owner()->remember(&processorLink);
            }
            else
            {
                // process becomes active
                // state transition 43
                p_setToRunning = time(); //?
                noStartNeeded = 0;
                modifyStatus(running);
                DBG_INIT()
                DBG(" state transition 43 (ready->running, restConsumeTime"
                    " == 0)\n restConsumeTime: "<< restConsumeTime() <<
                    ", p_setToRunning: " << p_setToRunning << endl)
                DBG_WRITE(onActivate)

                //owner()->remember(&processorLink);
                // no call to owner()->remember() needed. (see above)
            }
            break;

        case waiting:
            // a setWaiting ended
            {
                DBG_INIT()
                DBG(" setWaiting ended \n state transition " <<
                    (isSuspended() ?
                     "95 (waiting setWaiting suspended - > waiting suspended)" :
                     "74 (waiting setWaiting -> ready)" ) <<
                    "\n owner()->remember()")
                DBG_WRITE(onActivate)

                assert(desiredTime() == time());
                // state transitions 74, 95
                clrState(setWait);
                if (!isSuspended())
                    modifyStatus(ready);

                owner()->remember(&processorLink);
                break;
            }
		default:  ; //NOP   mjk 310895
    }

    atStateChange();

    if (noStartNeeded)
    {
        DBG_OUT("ToopsProcess::activate() calling nextEvent()\n-----\n",
                 onActivate )
        nextEvent();
    }
    else
    {
        DBG_OUT("ToopsProcess::activate() calling ToopsProcess::start()\n-----\n",
                 onActivate )
        // more?
        start();
    }
}


void ToopsProcess::start(void)
{

    DBG_INIT()
    DBG("\nToopsProcess::start() for: " << name() << endl)
    DBG_WRITE(onActivate)

    if (isNotStarted())
    {
        char x;
        char *px = &x;

        if (ToopsStackDir::less(px, context.StackBase))
            start();
        else
        {
            DBG_OUT(" process not started, calling behavior()\n",onStart)
            p_thisProcess = this;
            clrState(notStarted);
            behavior();
            DBG_OUT(" process did not terminate, calling terminate()\n",
                    onActivate)
            if (status() != terminated)
            {
                //  process ended without terminate -> emit warning
                wNoTerminate.err("",name());
                //  and do it yourself
                terminate();
            }
        }
    }
    else
    {
        p_thisProcess = this;
        DBG_OUT(" process is yet started, calling restore()\n",onActivate)
        restore();
    }

    DBG_OUT("\n\n\nToopsProcess::start() internal error. exit()\n\n",onActivate)
    _DBG(exit(1), onActivate);
}


int ToopsProcess::mustrecurse()
{
	 return  ToopsStackDir::less(ToopsStackDir::plusone(), context.StackBase);
}

void ToopsProcess::remember( ToopsTimerLink *changed)
{
    ToopsTimer* cT= changed->object();

    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::remember(ToopsTimerLink " << cT->name()
        << ") at " << time() << "\n");
    DBG_WRITE(onRemember)

    switch (cT->status())
    {
        case ToopsTimer::expired:
            DBG_OUT(" ToopsTimer expired\n ",onRemember)
            setState(timerExpired);

            if ( isInSync() && (status() != terminated) )
            {
                DBG_OUT("synchronize() ends, clearing sync ToopsSockets\n\n",
                         onRemember)
                clrState(inSync);
                // clear all synchronizing ToopsSockets
                p_sockets.reset();
                ToopsSocket* s; //"register" removed mjk 310895
                while(p_sockets.next(s))
                    if(s->status() == ToopsSocket::inSync)
                        s->cancelSync();

                DBG_OUT("\n ToopsProcess::remember() \n",
                             onRemember)
                if (!isSuspended())
                {
                    DBG_OUT(" -> state transition 64\n",onRemember)
                    // state transition 64
                    modifyStatus(ready);
                }
                else
                {
                    ;
                    DBG_OUT(" -> state transition 85\n",onRemember)
                }

                atStateChange();

                _DBG(ToopsProcess::write(2,1), onRemember)

                DBG_OUT(" remembering owner()\n\n", onRemember)
                owner()->remember(&processorLink);
            }
            break;
        case ToopsTimer::destroy:
            DBG_OUT(" destroy: ToopsTimer removed from list\n\n", onRemember)
            p_timers.remove(changed);
            break;
        case ToopsTimer::create:
            DBG_OUT(" create: ToopsTimer inserted in list\n\n", onRemember)
            p_timers.insert(changed);
            break;
	   default:  ; //NOP   mjk 310895
    } // switch(cT->status())
}

void ToopsProcess::remember(ToopsSocketLink *changed)
{
    ToopsSocket* cS = changed->object();

    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::remember(ToopsSocketLink " << cS->name()
        << ") at " << time() << "\n");
    DBG_WRITE(onRemember)

    switch (cS->status())
    {
        case ToopsSocket::destroy:
            DBG_OUT(" destroy: ToopsSocket removed from list\n\n", onRemember)
            if (changed->isLinked())
                p_sockets.remove(changed);
            break;

        case ToopsSocket::create:
            DBG_OUT(" create: ToopsSocket inserted in list\n\n", onRemember)
            p_sockets.insert(changed);
            break;

        case ToopsSocket::inSync:
            DBG_OUT(" inSync: calling waitUntilSync\n\n", onRemember)
            waitUntilSync();
            break;

        case ToopsSocket::gotMessage:
            DBG_OUT(" gotMessage:\n", onRemember)
            setState(newMessage);
            if(isInSync() && status() != terminated )
            {
                DBG_OUT(" synchronize() ended, clearing all sync ToopsSockets",
                        onRemember)
                clrState(inSync);
                // clear all synchronizing ToopsSockets
                p_sockets.reset();
                ToopsSocket* s;  //"register" removed mjk 310895
                while(p_sockets.next(s))
                    if(s->status() == ToopsSocket::inSync)
                        s->cancelSync();

                DBG_OUT("\n ToopsProcess::remember() \n", onRemember)
                if (!isSuspended())
                {
                    DBG_OUT(" -> state transition 64\n",onRemember)
                    // state transition 64
                    modifyStatus(ready);
                }
                else
                {
                    ;
                    DBG_OUT(" -> state transition 85\n",onRemember)
                }

                atStateChange();

                _DBG(ToopsProcess::write(2,1), onRemember)

                DBG_OUT(" remembering owner()\n\n", onRemember)

                owner()->remember(&processorLink);
            }
            break;
		default:  ; //NOP   mjk 310895
    } // switch(cT->status())
}



void ToopsProcess::synchronize(void)
{
    // check for caller
    if (thisProcess() != this)
        fSyncNoAccess.err(thisProcess()->name());

    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::synchronize() for " << name() << " at " <<
        time() << "\n");

    // check the ToopsSockets for having a message in their queues
    int noMessages = 1;
    ToopsSocket* s;  //"register" removed mjk 310895
    p_sockets.reset();
    while(p_sockets.next(s))
    {
        if (s->hasMessage())
        {
            noMessages = 0;
            break;
        }
    }

    DBG( (noMessages ? " no " : " ") << "messages in the ToopsProcess' ToopsSockets\n" )
    DBG_WRITE(onSync)

    if (noMessages)
    {
        // well, make them synchronize
        DBG_OUT(" calling startSync() for all ToopsSockets\n", onSync)
        p_sockets.reset();
        while(p_sockets.next(s))
            s->startSync();

        DBG_OUT(" ToopsProcess::synchronize() calling waitUntilSync()\n\n", onSync)
        waitUntilSync();
        DBG_INIT()
        DBG("\n ToopsProcess::synchronize() for " << name() <<
            "\n returns from waitUntilSync() at " <<
            time() << endl << endl)
        DBG_WRITE(onSync);
    }

}

void ToopsProcess::synchronize( ToopsSocket *t1, ToopsSocket *t2, 
                           ToopsSocket *t3,
                           ToopsSocket *t4, ToopsSocket *t5, ToopsSocket *t6)
{
    // check for caller
    if (thisProcess() != this)
        fSyncNoAccess.err(thisProcess()->name());

    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::synchronize(" <<
        t1->name() << ", " <<
        (t2 ? t2->name(): "-") << ", " <<
        (t3 ? t3->name(): "-") << ", " <<
        (t4 ? t4->name(): "-") << ",\n                " <<
        (t5 ? t5->name(): "-") << ", " <<
        (t6 ? t6->name(): "-") << ") for " << name() <<
        " at " << time() << "\n")
    DBG(" forming an array and passing it to synchronize(ToopsSocket**)\n")
    DBG_WRITE(onSync)

    ToopsSocket *ToopsSarray[7];
    ToopsSarray[0] = t1;
    ToopsSarray[1] = t2;
    ToopsSarray[2] = t3;
    ToopsSarray[3] = t4;
    ToopsSarray[4] = t5;
    ToopsSarray[5] = t6;
    ToopsSarray[6] = 0;

    synchronize(ToopsSarray);

}

void ToopsProcess::synchronize( ToopsSocket** ToopsSarray)
{
    // check for caller
    if (thisProcess() != this)
        fSyncNoAccess.err(thisProcess()->name());

    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::synchronize(ToopsSocket**) for " << name()
        << "at " << time() << "\n")


    // check, if at least one of the given ToopsSockets holds a ToopsMessage
    int noMessages = 1;
    for (register uint i = 0; ToopsSarray[i] != 0; i++)
    {
        if(ToopsSarray[i]->hasMessage())
        {
            noMessages = 0;
            break;
        }
    }

    DBG( (noMessages ? " no " : " ") <<
         "messages in the specified ToopsSockets\n" )
    DBG_WRITE(onSync)

    if (noMessages)
    {
        // well, make them sync
        DBG_OUT(" calling startSync() for all specified ToopsSockets\n", onSync)
        for (register uint i = 0; ToopsSarray[i] != 0; i++)
            ToopsSarray[i]->startSync();
        DBG_OUT(" ToopsProcess::synchronize() calling waitUntilSync()\n\n", onSync)
        waitUntilSync();
        DBG_INIT()
        DBG("\n ToopsProcess::synchronize() for " << name() <<
            "\n returns from waitUntilReceived() at " <<
            time() << endl << endl)
        DBG_WRITE(onSync);
    }

}

void ToopsProcess::waitUntilSync()
{

    DBG_INIT()
    DBG(ThisClassInfo()->name() << "::waitUntilSync for " << name()
        << "at " << time() << "\n")
    DBG(" state transition 36, remembering owner()\n\n")
    DBG_WRITE(onWaitUntilSync)

    // state transition 36
    setState(inSync);
    modifyStatus(waiting);

    atStateChange();

    owner()->remember(&processorLink);

    if (!contextSave())
		  nextEvent();

}

void ToopsProcess::write(int depth, int mode) const
{
    assert(this != 0);

    if (depth-1 >= 0)
        ToopsNamedObject::write(depth-1, mode);

    ostrstream o;

    if (depth==0 && !mode)
        o << DBGends << endl;

    o << ThisClassInfo()->name() << ":"
      << " name " << name() << " on ToopsProcessor " << owner()->name()
      << " priority: " << priority() << endl;
    o << " time info:  desired: " << desiredTime()
      << ", to consume: " << restConsumeTime()
      << ", set running at: " << p_setToRunning << endl ;

    sayState(o,this);

    if (!mode)
    {
        o << " thisProcess is " << (p_thisProcess ? p_thisProcess->name() :"0")
          << " on ToopsProcessor: " <<
          (p_thisProcess ? p_thisProcess->owner()->name() : " -") << endl;
        // lists ? !!more!!
    }
    o << ends;
    char *out = o.str();
    ToopsObject::DBGout(out);
    delete out;
}

// no member function!!
static void sayState(ostrstream &o, const ToopsProcess * const p)
{
    o << " ToopsProcess stateFlags: "
      << (p->isNotStarted() ? "not " : "    ") << "started, ";
    switch(p->status())
    {
        case ToopsProcess::ready:      o << "ready     "; break;
        case ToopsProcess::running:    o << "running   "; break;
        case ToopsProcess::terminated: o << "terminated"; break;
        case ToopsProcess::waiting:    o << "waiting   "; break;
		default:  ; //NOP   mjk 310895
    }

    o << "\n  flags: "
      << (p->isSuspended()    ? "suspended,     " : "")
      << (p->isInSync()       ? "inSync,        " : "")
      << (p->isSetWaiting()   ? "set to wait,   " : "")
      << (p->isTimerExpired() ? "timer expired, " : "")
      << (p->isNewMessage()   ? "new message.   " : "");
    o << endl;
}







//--------------------------------------------------------------------------
// ToopsProcess state transition table
// ===============================
//
// possible (and in their behaviour distinguishible) states of ToopsProcess:
// (state flags: sus(pended), (in)syn(c), sw (setWaiting),
//               ~ don't care, + set, - clear , 0 always clear)
//
//                 | state flags  |
// No.| status() = |sus |syn | sw | comment
// ---+------------+----+----+----+---------------------------------
//  1.| terminated | ~  | ~  | ~  | (*)
//  2.| running    | 0  | 0  | 0  | restConsumeTime() >  0   (**)
//  3.| running    | 0  | 0  | 0  | restConsumeTime() == 0   (***)
//  4.| ready      | 0  | 0  | 0  |
//  5.| waiting    | +  | -  | -  | suspend() called for this process
//  6.| waiting    | -  | +  | -  | synchronize(..), but no message in ToopsSocket
//  7.| waiting    | -  | -  | +  | process called setWaiting()
//  8.| waiting    | +  | +  | -  | -->6., but suspended by another process
//  9.| waiting    | +  | -  | +  | -->7., but suspended by another processs
//
//  (*)   the state flags won't be cleared (user information)
//  (**)  process is consuming time
//  (***) process has consumed time and it's code is started / restarted
//        (this state is sometimes also called 'active')
//
//
//
// state transition table:
//  \to |
//   \  |
// from\| 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9 |
// -----+---+---+---+---+---+---+---+---+---+
//   1  | a | x | x | x | x | x | x | x | x |
//  ----+---+---+---+---+---+---+---+---+---+
//   2  | b | x | f | i | m | x | x | x | x |
//  ----+---+---+---+---+---+---+---+---+---+
//   3  | c | d | x | x | n | o | p | x | x |
//  ----+---+---+---+---+---+---+---+---+---+
//   4  | b | e | h | x | m | x | x | x | x |
//  ----+---+---+---+---+---+---+---+---+---+
//   5  | b | x | x | j | x | x | x | x | x |
//  ----+---+---+---+---+---+---+---+---+---+
//   6  | b | x | x | k | x | x | x | m | x |
//  ----+---+---+---+---+---+---+---+---+---+
//   7  | b | x | x | l | x | x | x | x | m |
//  ----+---+---+---+---+---+---+---+---+---+
//   8  | b | x | x | x | k | j | x | x | x |
//  ----+---+---+---+---+---+---+---+---+---+
//   9  | b | x | x | x | l | x | j | x | x |
//  ----------------------------------------
//
// triggers for state transitions:
// x  transition is impossible
// a  any action on a terminated process will be ignored
// b  another process calls terminate() for this processor, if the process
//    was waiting, the state flags are not cleared
// c  the process calls terminate() for itself.
// d  consumeTime() called
// e  process became the process with the highest priority on it's processor
//    (decision prepared by ToopsProcessor::remember(), decided by ToopsProcessor::
//     activate, state transition performed by ToopsProcessor::activate() )
//    and continues to consume time
// f  restConsumeTime() == 0, ToopsProcess::activate() will be called by
//    ToopsSimulation::start(), activate() will start / restart
//    ToopsProcess::behavior()
// g  --
// h  process became the process with the highest priority on it's processor
//    (decision prepared by ToopsProcessor::remember(), decided by ToopsProcessor::
//     activate, state transition performed by ToopsProcessor::activate() )
//     and is started / restarted
// i  a process with a higher priority becomes ready, so the process stops
//    consuming time.
// j  resume() called by another for this process
// k  message arrival or timer alert, indicated to the process by a call to
//    ToopsProcess::remember(ToopsTimer* | ToopsSocket*), noted to the processor
//    via ToopsProcessor::remember()
// l  the time to wait specified via setWaiting() ended, this is managed by
//    ToopsProcess::activate()
// m  another process calls suspend() for this process
// n  suspend() called by this process.
// o  synchronize() called and none of the specified ToopsSockets contains a
//    ToopsMessage
// p  setWaiting(t) called
//
// note: in the source code comments to procunit.cpp/.C, tprocess.cpp/.C
//       is sometimes taken a reference to the table above, the transitions
//       are named xy, where x,y are the status numbers introduced above
//       and xy means the transition from status x to status y,
//       e.g. 31 means that a process' state changes from running(active) to
//       terminated. 
// note: one more state is defined for ToopsProcess: destroy, but
//       this is used only to signal to ToopsProcessor::remember() that
//       processorLink has to be removed from the processor's
//       scheduling lists (also used to indicate destruction to
//       atStateChange())
