#include "toops.h"
#include "tno_list.h"
#include "tprocess.h"
#include "procunit.h"
#include "timer.h"
#include "socket.h"
#include "channel.h"
#include "message.h"
#include "timestep.h"
#include "to_error.h"
#include "portable.h"
#include "strstrea.h"
#include "setjump.h"
#include "event.h"
#include <assert.h>
#include <string.h>

//-----------------------------------------------------------------------------
// ToopsSimulation (declared in toops.h)
// =================================


IMPLEMENT_CLASSINFO( ToopsSimulation, ToopsObject);

// Set up class ToopsSimulation's static data members:
// The system wide lists (holding all objects of the corresponding class)
ToopsProcessorList *ToopsSimulation::s_processors = 0;
ToopsChannelList   *ToopsSimulation::s_channels     = 0;
ToopsMessageList   *ToopsSimulation::s_allMessages  = 0;
ToopsProcessList   *ToopsSimulation::s_allProcesses = 0;
ToopsTimerList     *ToopsSimulation::s_allTimers = 0;
ToopsSocketList    *ToopsSimulation::s_allSockets = 0;

// Holds all ToopsProcessors that must be activated at a time.
ToopsProcessorList *ToopsSimulation::s_processorsToActivate = 0;

// holds all ToopsTimeSteps:
ToopsTimeStepList  *ToopsSimulation::s_timeSteps = 0;

simtime ToopsSimulation::s_time = 0;      // simulation time
simtime ToopsSimulation::timeOut = 0;     // end of simulation
// the state of the simulation (initially clear, i.e. no sim objects exist)
ToopsSimulation::simstate ToopsSimulation::s_state = ToopsSimulation::clear;
// the sum of all ToopsProcessors, ToopsTimers, ToopsProcesses, ToopsSockets and ToopsChannels
instance_counter ToopsSimulation::s_objects = 0;
// holds the stack address at start, needed for _stop(...)
ToopsContext *ToopsSimulation::mainContext = 0;
// this function is called, when the time advances
ToopsSimulation::PVF2S ToopsSimulation::s_timeAdvanceCallBack = 0;

// the errors (->to_error.h)
ToopsError ToopsSimulation::fCannotStart(TE_TSIMULATION_CANNOTSTART,
                                 ToopsError::fatal,
                                 ToopsSimulation::ThisClassInfo()->name(),
											"cannot start simulation");
// pt 9.94 'ToopsSimulation::' in Z.53 vor 'ThisClassInfo' ergaenzt

ToopsError ToopsSimulation::fNoProcessor(TE_TSIMULATION_NOPROCESSOR,
                                 ToopsError::fatal,
                                 ToopsSimulation::ThisClassInfo()->name(),
                                 "cannot start simulation w/o a ToopsProcessor");
// pt 9.94 'ToopsSimulation::' in Z.59 vor 'ThisClassInfo' ergaenzt

ToopsError ToopsSimulation::fNoProcesses(TE_TSIMULATION_NOPROCESSES,
                                 ToopsError::fatal,
                                 ToopsSimulation::ThisClassInfo()->name(),
                                 "cannot start simulation w/o a ToopsProcess");
// pt 9.94 'ToopsSimulation::' in Z.65 vor 'ThisClassInfo' ergaenzt

ToopsSimulation::ToopsSimulation (responsibility who)
{
    DBG_OUT("ToopsSimulation::ToopsSimulation()\n",cstr)

    if (s_objects++ == 0)
	 {
         DBG_OUT(" first object: setting up the internal lists,"
                 " state = notStarted\n", cstr)
         s_processors   = new ToopsProcessorList;
         s_processorsToActivate = new ToopsProcessorList;
         s_channels     = new ToopsChannelList;
			s_allMessages  = new ToopsMessageList;
         s_allProcesses = new ToopsProcessList;
         s_allTimers    = new ToopsTimerList;
         s_allSockets   = new ToopsSocketList;

         s_timeSteps    = new ToopsTimeStepList;

         s_state = notStarted;
    }
    s_order = s_objects;
    s_responsibility = who;
    _DBG(write(0,1),cstr)
}

ToopsSimulation::~ToopsSimulation()
{
   DBG_OUT("ToopsSimulation::~ToopsSimulation()\n",dstr)
   if (--s_objects == 0)
   {
			DBG_OUT(" last object: deleting internal lists, resetting "
                 "static members\n state = clear", dstr)
         delete s_processors;
         delete s_processorsToActivate;
         delete s_channels;
         delete s_allMessages;
         delete s_allProcesses;
         delete s_allTimers;
			delete s_allSockets;

         delete s_timeSteps;

         s_processors   = 0;
         s_processorsToActivate = 0;
         s_channels     = 0;
         s_allMessages  = 0;
         s_allProcesses = 0;
         s_allTimers    = 0;
         s_allSockets   = 0;

         s_timeSteps = 0;

         s_time = timeOut = 0;
         s_state = clear;
   }

}

// The num... functions must take care of the possible non existance of
// lists, when state == clear.
instance_counter ToopsSimulation::numProcessors(void)
{
    if (state() == clear)
		  return (instance_counter)0;
    else
		  return processors()->len();
}

instance_counter ToopsSimulation::numChannels(void)
{
    if (state() == clear)
		  return (instance_counter)0;
    else
        return channels()->len();
}

instance_counter ToopsSimulation::numMessages(void)
{
    if (state() == clear)
        return (instance_counter)0;
    else
        return allMessages()->len();
}


instance_counter ToopsSimulation::numProcesses(void)
{
    if (state() == clear)
        return (instance_counter)0;
    else
        return allProcesses()->len();
}

instance_counter ToopsSimulation::numTimers(void)
{
    if (state() == clear)
		  return (instance_counter)0;
	 else
		  return allTimers()->len();
}

instance_counter ToopsSimulation::numSockets(void)
{
	 if (state() == clear)
		  return (instance_counter)0;
	 else
		  return allSockets()->len();
}


ToopsSimulation::stopReason ToopsSimulation::start(simtime end)
{

	 DBG_OUT("ToopsSimulation::start()\n", onStart)

	 // check, if call is legal and the sim can be started
	 if (state() != notStarted)
		  fCannotStart.err();
	 if (numProcessors() == 0)
		  fNoProcessor.err();
	 if (numProcesses() == 0)
		  fNoProcesses.err();

	 // Make sure there is at least one ToopsTimeStep in timeSteps().
	 // This is the special ToopsTimeStep, indicating when to stop the simulation.
	 timeOut = end ? end : SIMTIME_MAX;
	 {
		  DBG_INIT()
		  DBG(" time for end: " << end
				<< ", ToopsTimeLink(" << timeOut << ",timeOutIndicator) appended\n")
		  DBG_WRITE(onStart)
	 }
	 timeSteps()->append(new ToopsTimeStep(timeOut,
														ToopsTimeStep::timeOutIndicator));

	 // call atStart for all ToopsProcessors, ToopsChannels, ToopsProcesses,
	 // ToopsTimers and ToopsSockets
	 DBG_OUT(" ToopsSimulation::start() calling atStart() for all objects\n",
				onStart)
	 callAtStart();
	 DBG_OUT("\n", onStart)

	 //timeSteps()->write(0,1);

	 mainContext = new ToopsContext;
	 stopReason ret;

	 ToopsSimulation* theNextEvent;

	 // Get the actual stack adress and save it.
	 char x;
	 char *px = &x;
	 ToopsContext::StackBase = px;

	 // mainContext is the context for the main procedure of the simulation.
	 // The stack address in the simulation will always be 'above' StackBase,
	 // so there is no need for saving a stack for the main procedure. Even
	 // a longjmp from anywhere in the simulation to mainContext->JB (set up
	 // only here) will not cause any problems. sic!
	 ret = (ToopsSimulation::stopReason) setjmp(mainContext->JB);

	 if ( ret > 0 )
	 {
		  if (ret == _next)
		  {
				DBG_OUT("\n! ToopsSimulation::start() getting the next event\n",
							onStart)
				theNextEvent = getNextEvent();
				DBG_OUT("\n! ToopsSimulation::start() activating the next event\n",
							onStart)
				theNextEvent->activate();
		  }
	 }
	 else
	 {
		  s_state = running;
		  DBG_OUT("! ToopsSimulation::start(), state: running, getting first event\n",
					 onStart)
		  theNextEvent = getNextEvent();
		  DBG_OUT("! ToopsSimulation::start(), activating first event\n",
					 onStart)
		  theNextEvent->activate();
	 }

	 DBG_OUT("\n! ToopsSimulation::start()  simulation ended, state = stopped\n",
				 onStart)

	 s_state = stopped;

	 DBG_OUT("! calling atEnd() for all objects \n",
				 onStart)
	 // call atEnd for all ToopsProcessors, ToopsChannels, ToopsProcesses, ToopsTimers, ToopsSockets
	 callAtEnd();

	 DBG_OUT("! deleting all remaining objects with responsibility != user\n",
				 onStart)
	 // delete all ToopsSockets, ToopsTimers, ToopsProcesses, ToopsChannels, ToopsProcessors,
	 // ToopsMessages with responsibility != user
	 cleanUp();

	 return ret;
}

ToopsSimulation* ToopsSimulation::getNextEvent(void)
{
    DBG_OUT("\n ToopsSimulation::getNextEvent()\n  timeSteps()->first() is ",
              onGetNextEvent)

    //cout << "!!!\n";
    //timeSteps()->write(0,0);

	 ToopsSimulation* event = 0;
	 ToopsTimeStep* t = timeSteps()->first();

	 if ( (event = t->nextEvent()) != 0 )
	 {
		  DBG_INIT()
		  DBG("at " << t->time() << ", " << event->GetClassInfo()->name()
				<< " " << ((ToopsNamedObject*)(event))->name() << " returned" << endl
				<< "  " << DBGends  << endl )
		  DBG_WRITE(onGetNextEvent)
		  return event;
	 }

	 DBG_OUT(" empty, activating ToopsProcessors\n",onGetNextEvent)
	 // let the processors schedule that must do it
	 activateProcessors();

	 //cout << "+++\n";
	 //timeSteps()->write(0,0);


	 _DBG(timeSteps()->write(0,0),onGetNextEvent)

	 // check, if they produced a new event for this time()
	 assert(timeSteps()->first() == t);
	 if ( (event = t->nextEvent()) != 0 )
	 {
		  DBG_INIT()
		  DBG("  New events produced, first is " << event->GetClassInfo()->name() <<
				 ((ToopsNamedObject*)(event))->name() << ", returned" << endl <<
				 "  " << DBGends << endl )
		  DBG_WRITE(onGetNextEvent)
		  return event;
	 }
	 // ok, we've got definitively no more event for this time()
	 assert (t->isEmpty());
	 DBG_OUT("\n  deleting the actual (empty) ToopsTimeStep\n",onGetNextEvent)
	 timeSteps()->remove(t);
	 delete t;

	 DBG_OUT("  searching for next not empty ToopsTimeStep:\n",onGetNextEvent)
	 // remove timeSteps()->first() from the list and delete it, do also
	 // so with all following empty timeSteps, but check for the special
	 // one (with type() == timeOutIndicator)

	 for (;;)
	 {
		  t = timeSteps()->first();

		  DBG_INIT()
		  DBG("  - timeSteps()->first() at " << t->time() << " is " <<
				(t->isEmpty() ? "" : "not " ) << "empty, ")
        DBG_WRITE(onGetNextEvent)

        if (t->type())
		  {
			// we have reached the end of the simulation, determine
               // the reason
			DBG_INIT()
            DBG("end indicator reached\n\n   !! END OF SIMULATION !!\n")
            stopReason why = timeStop; // timeOut reached
            if (t->isEmpty() && timeSteps()->len() == 1)
                why = regularStop; // no more events
            DBG( "   reason: " << (why == timeStop ? "time out" :
                 "no events left") << ", calling stop()\n" << DBGends << endl)
				DBG_WRITE(onGetNextEvent)

				_stop(why);
        }

        if (t->isEmpty())
        {
		DBG_OUT("removed\n",onGetNextEvent)
		timeSteps()->remove(t);
          delete t;
        }
        else
        {
            break;
        }
    } // for (;;)

    // the simulation time advances only here!!
    assert ( t->time() > time() );
	 {
        DBG_INIT()
        DBG("  time() advances from " << time() << " to "<< t->time() << endl)
		  DBG("  calling ... \n")
        DBG_WRITE(onGetNextEvent)
    }

	simtime old = s_time;
	s_time = t->time();

    // call user defined function !!
    if (s_timeAdvanceCallBack)
       (*s_timeAdvanceCallBack) (old, s_time);

    // now we've got a valid timeStep as first element in timeSteps
    event = t->nextEvent();

    {
        DBG_INIT()
        DBG("  Next event is " << event->GetClassInfo()->name() << " " <<
             ((ToopsNamedObject*)(event))->name() << ", returned" << endl <<
             "  " << DBGends << endl )
        DBG_WRITE(onGetNextEvent)
    }
    return event;
}

void ToopsSimulation::activateProcessors(void)
{
    ToopsProcessor* p;
    processorsToActivate()->reset();
    while(processorsToActivate()->len())
    {
        processorsToActivate()->getnext(p);
        ((ToopsSimulation*)p)->activate();
    }
}


void ToopsSimulation::callAtStart(void)
{
    // call atStart for all ToopsProcessors, ToopsChannels, ToopsProcesses, ToopsTimers
    // and ToopsSockets

/*    ToopsSimulation* toCall; / pt 9.94

    processors()->reset();
    while ( processors()->next((ToopsProcessor*)toCall) )
        toCall->atStart();

    channels()->reset();
    while ( channels()->next((ToopsChannel*)toCall) )
        toCall->atStart();

	 allProcesses()->reset();
    while ( allProcesses()->next((ToopsProcess*)toCall) )
        toCall->atStart();

    allTimers()->reset();
    while ( allToopsimers()->next(( ToopsTimer*)toCall) )
        toCall->atStart();

    allSockets()->reset();
	 while ( allSockets()->next((ToopsSocket*)toCall) )
        toCall->atStart();
*/ // pt 9.94
      
    ToopsSimulation* toCall;

	 processors()->reset();
    ToopsProcessor* tProc;
    while ( processors()->next(tProc) )
    {  
       toCall = (ToopsSimulation*)tProc; 
       toCall->atStart();
    }

    channels()->reset();
    ToopsChannel* tChan;
    while ( channels()->next(tChan) )
    {
		  toCall = (ToopsSimulation*)tChan;
        toCall->atStart();
    }

    allProcesses()->reset();
    ToopsProcess* tProcess;
    while ( allProcesses()->next(tProcess) )
    {
        toCall = (ToopsSimulation*)tProcess;
		  toCall->atStart();
    }
    
    allTimers()->reset();
    ToopsTimer* tTimer;
    while ( allTimers()->next(tTimer) )
    {
        toCall = (ToopsSimulation*)tTimer;
        toCall->atStart();
    }

    allSockets()->reset();
	 ToopsSocket* tSocket;
    while ( allSockets()->next(tSocket) )
    {
        toCall = (ToopsSimulation*)tSocket;
        toCall->atStart();
    }

// Ende der Ergaenzungen / pt 9.94    

}

void ToopsSimulation::callAtEnd(void)
{
    // call atEnd for all ToopsProcessors, ToopsChannels, ToopsProcesses, ToopsTimers, ToopsSockets

/* / pt 9.94
   
    ToopsSimulation* toCall;

    processors()->reset();
    while ( processors()->next((ToopsProcessor*)toCall) ) 
        toCall->atEnd();

    channels()->reset();
    while ( channels()->next((ToopsChannel*)toCall) )
        toCall->atEnd();

    allProcesses()->reset();
    while ( allProcesses()->next((ToopsProcess*)toCall) )
        toCall->atEnd();

    allTimers()->reset();
    while ( allToopsimers()->next(( ToopsTimer*)toCall) )
		  toCall->atEnd();

    allSockets()->reset();
    while ( allSockets()->next((ToopsSocket*)toCall) )
        toCall->atEnd();

*/  // pt 9.94    
    
    ToopsSimulation* toCall;

    processors()->reset();
    ToopsProcessor* tProc;
    while ( processors()->next(tProc) )
    {
        toCall = (ToopsSimulation*)tProc;
        toCall->atEnd();
    }

    channels()->reset();
    ToopsChannel* tChan;
    while ( channels()->next(tChan) )
    {
        toCall = (ToopsSimulation*)tChan;
        toCall->atEnd();
    }

    allProcesses()->reset();
	 ToopsProcess* tProcess;
    while ( allProcesses()->next(tProcess) )
    {
        toCall = (ToopsSimulation*)tProcess;
        toCall->atEnd();
    }

    allTimers()->reset();
    ToopsTimer* tTimer;
	 while ( allTimers()->next(tTimer) )
    {
        toCall = (ToopsSimulation*)tTimer;
        toCall->atEnd();
    }

    allSockets()->reset();
    ToopsSocket* tSocket;
    while ( allSockets()->next(tSocket) )
    {
        toCall = (ToopsSimulation*)tSocket;
        toCall->atEnd();
    }

// Ende der Ergaenzungen / pt 9.94  

}

void ToopsSimulation::cleanUp(void)
{
    // delete all ToopsMessages, ToopsSockets, ToopsTimers, ToopsProcesses, ToopsChannels,
    // ToopsProcessors, with responsibility != user

/* / pt 9.94   
   
    ToopsSimulation* toKill;

	 allMessages()->reset();
    while ( allMessages()->next((ToopsMessage*)toKill) )
    {
		  delete toKill;
    }

    allSockets()->reset();
    while ( allSockets()->next((ToopsSocket*)toKill) )
    {
        if (toKill->responsible() != user)
            delete toKill;
    }

    allTimers()->reset();
    while ( allToopsimers()->next(( ToopsTimer*)toKill) )
    {
        if (toKill->responsible() != user)
            delete toKill;
	 }

    allProcesses()->reset();
    while ( allProcesses()->next((ToopsProcess*)toKill) )
    {
        if (toKill->responsible() != user)
            delete toKill;
    }

	 channels()->reset();
    while ( channels()->next((ToopsChannel*)toKill) )
    {
        if (toKill->responsible() != user)
            delete toKill;
    }

    processors()->reset();
    while ( processors()->next((ToopsProcessor*)toKill) ) 
	 {
        if (toKill->responsible() != user)
            delete toKill;
    }

*/ // pt 9.94
        
    
    ToopsSimulation* toKill;

    allMessages()->reset();
    ToopsMessage* tMessage;
    while ( allMessages()->next(tMessage) )
    {
        toKill = (ToopsSimulation*) tMessage;
        delete toKill;
    }

	 allSockets()->reset();
    ToopsSocket* tSocket;
    while ( allSockets()->next(tSocket) )
    {
        toKill = (ToopsSimulation*)tSocket;
        if (toKill->responsible() != user)
            delete toKill;
    }

    allTimers()->reset();
    ToopsTimer* tTimer;
    while ( allTimers()->next(tTimer) )
    {
        toKill = (ToopsSimulation*)tTimer;
        if (toKill->responsible() != user)
				delete toKill;
    }

	 allProcesses()->reset();
    ToopsProcess* tProcess;
    while ( allProcesses()->next(tProcess) )
    {
        toKill = (ToopsSimulation*)tProcess;
        if (toKill->responsible() != user)
            delete toKill;
    }

	 channels()->reset();
    ToopsChannel* tChan;
    while ( channels()->next(tChan) )
    {
        toKill = (ToopsSimulation*)tChan;
        if (toKill->responsible() != user)
            delete toKill;
    }

    processors()->reset();
    ToopsProcessor* tProc;
    while ( processors()->next(tProc) ) 
    {
        toKill = (ToopsSimulation*)tProc;
        if (toKill->responsible() != user)
            delete toKill;
    }
    
// pt 9.94 Ende der Ergaenzungen
  
    
	 // cleanup
    delete mainContext;
    mainContext = 0;

    // delete remaining ToopsTimeSteps
    timeSteps()->reset();
	 ToopsTimeStep* toDelete;
    while (timeSteps()->len())
    {
        timeSteps()->getnext(toDelete);
        delete toDelete;
    }

}

void ToopsSimulation::stopSimulation(void)
{
    if (state() == running)
        _stop(userStop);
}

void ToopsSimulation::stopWithError(void)
{
    if (state() == running)
		  _stop(errorStop);
}

void ToopsSimulation::_stop(ToopsSimulation::stopReason why)
{
    // no check needed for state(), because this function is protected
    assert(state()==running);

    longjmp(mainContext->JB, why);
}


ToopsSimulation::PVF2S ToopsSimulation::setTimeAdvanceCallBack(PVF2S n)
{
    PVF2S tmp = s_timeAdvanceCallBack;
    s_timeAdvanceCallBack = n;
    return tmp;
}


void ToopsSimulation::write(int, int mode) const
{
    // ignore depth, because ToopsObject has NO debug info
    // mode = 0 outputs here also some info about the static members
    const char * const n = ThisClassInfo()->name();
    ostrstream o;

    if (!mode)
        o << DBGends << endl;

	 o << n << ":  order: " << order() << ", responsibility: ";

//  Microsoft C does not accept the following. 3 Apr 95 mjk 
//  It looks like responsible() and responsibility() got mixed up!
//    switch ( responsibility() )
    switch ( responsible() )
    {
        case user :
            o << "user"; break;
        case TOOPSatEnd :
            o << "TOOPSatEnd"; break;
        case TOOPSdirect :
            o << "TOOPSdirect"; break;
    }

    o << endl;

    if (!mode)
    {
        o << " state(): ";
        switch (state())
		  {
            case clear:      o << "clear     "; break;
            case notStarted: o << "notStarted"; break;
            case running:    o << "running   "; break;
            case stopped:    o << "stopped   "; break;
        }
		  o << " at time() " << time() << "\n ";
        o << objects() + numMessages() << " currently existing objects:\n";
        o << " " << numProcessors() << " ToopsProcessors, " << numChannels()
			 << " ToopsChannels, " << numTimers() << " ToopsTimers, \n ";
        o << numProcesses() << " ToopsProcesses, " << numSockets()
          << " ToopsSockets, " << numMessages() << " ToopsMessages\n";

        if (processorsToActivate())
        {
            o << " " << processorsToActivate()->len() << " ToopsProcessors to "
              << "activate:\n ";
            processorsToActivate()->reset();
            int j = 4;
            ToopsProcessor* p;
            while(processorsToActivate()->next(p))
            {
                o << p->name() << ", ";
                if(--j == 0)
                {
                    o << "\n ";
                    j = 4;
					 }
            }
            if (j != 4)
                o << "\n ";
        }
    }

    o << ends;

	 char *out = o.str();
    ToopsObject::DBGout(out);
    delete out;

    if (!mode && timeSteps())
        timeSteps()->write(0);

}

//-----------------------------------------------------------------------------
// ToopsNamedObject (declared in toops.h)
// ==================================


IMPLEMENT_CLASSINFO( ToopsNamedObject, ToopsSimulation);

// static member:
ToopsError ToopsNamedObject::eName( TE_TNO_NAME, ToopsError::error,
									 ToopsNamedObject::ThisClassInfo()->name(),
                            "\n  name is empty or not unique");
// pt 9.94 'ToopsNamedObject::' in Z.632 vor 'ThisClassInfo' ergaenzt

ToopsNamedObject::ToopsNamedObject(const char * szN, responsibility who) :
        ToopsSimulation(who) 
{
   szName = new char[strlen(szN)+1];
   strcpy(szName,szN);
}

void ToopsNamedObject::write(int depth, int mode) const
{
    ostrstream o;

    if (depth-1 >= 0)
        ToopsSimulation::write(depth-1, mode);

    if (!mode && depth == 0)
        o << DBGends << endl;

    o << ThisClassInfo()->name() << ":  name(): " << name() << endl << ends;

    char *out = o.str();
    ToopsObject::DBGout(out);
    delete out;
}

